#ifndef RFTABLEMODEL_H
#define RFTABLEMODEL_H

#include <QObject>
#include <QRect>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QAbstractTableModel>
#include <QReadWriteLock>
#include <QTimer>

#include "table_geometry.h"
#include "../database.h"

class rflookupconfig {
public:
  enum _lookup_type {
    LOOKUP_NEIGHBOUR = 0x00,
    LOOKUP_LINEAR = 0x01
  };
  _lookup_type lookup_type = LOOKUP_LINEAR;
  enum _tail_type {
    TAIL_LINEAR = 0x00,
    TAIL_FLAT = 0x01,
    TAIL_NULL = 0x02
  };
  _tail_type tail_type = TAIL_FLAT;
  dbdata encode() const {
    dbdata out;
    out.name = "LOOKUP_CONFIG";
    out.set("TYPE",lookup_type);
    out.set("TAIL",tail_type);
    return out;
  };
  bool decode(const dbdata &in) {
    lookup_type = (_lookup_type)in.value("TYPE",0).toInt();
    tail_type = (_tail_type)in.value("TAIL",0).toInt();
    return true;
  }
};

class rftablemodel : public QObject {
  Q_OBJECT
public:
  rftablemodel();
  virtual ~rftablemodel() {};

  enum _table_type {
    DATALOG, // a 3d table with arbitrary number of rows, text column names.
    TABLE2D, // a numerical 2d table of doubles
    TABLE3D, // a numerical 3d table of doubles
    NOTABLE // an invalid table
  };

  virtual _table_type table_type() const { return NOTABLE; };

  bool is_2d() const {return table_type() == TABLE2D;};
  bool is_3d() const {return table_type() == TABLE3D || table_type() == DATALOG;};

  virtual int n_columns() const = 0;
  virtual int n_rows() const = 0;
  virtual bool auto_col_size() const { return true; };

  virtual bool readonly() const;
  virtual bool valid() const;

  virtual QString col_name(const int &col) const;
  virtual QString row_name(const int &row) const;
  virtual int col_custom_size(const int &col) const;

  virtual table_axis x_axis() const;
  virtual table_axis y_axis() const;
  virtual double x_axis_value(const int &col) const;
  virtual double y_axis_value(const int &row) const;
  virtual QString x_axis_name() const;
  virtual QString y_axis_name() const;

  int col_index(const QString &s) const;
  int col_index(const double &d) const;
  int row_index(const double &d) const;

  QVector <double>col_vector(const int &col) const;
  QVector <double>row_vector(const int &row) const;
  virtual QVector <double>value_list() const;

  virtual QString string_at(const int &row, const int &col) const = 0;
  virtual double double_at(const int &row, const int &col) const;
  virtual bool is_null(const int &row, const int &col) const;

  QStringList column_string_list() const;
  QStringList row_string_list() const;

  virtual bool set(const QString &s, const int &row, const int &col);
  virtual bool set(const double &d, const int &row, const int &col);
  virtual bool set_null(const int &row, const int &col);

  virtual int append_row(); // add empty row

  int max_value_length(const int &col) const;
  int max_row_name_length() const;
  int max_col_name_length() const;

  virtual double min_value() const;
  virtual double max_value() const;

  virtual QStringList get_views() const;
  virtual void set_view(const QString &view);
  virtual QString current_view() const;

  void clear_selection();
  bool is_selected(const int &row, const int &col) const;
  bool all_selected() const;
  bool has_selection() const;
  int selected_cell_count() const;
  void select_row(const int &row, bool clear);
  void select_col(const int &col, bool clear);
  void select_single(const int &row, const int &col, bool clear);
  void select_span(const table_region &r, bool clear);
  void select_all();

  QString selection_to_clipboard_text() const;
  QString table_to_romraider() const;

  table_region last_selection() const;
  QVector<table_cell> selected_cells() const;
  QVector<table_region> selected_regions() const;
  QPoint selection_origin() const;

  QString to_ascii() const;

  double lookup(const rflookupconfig &config, double x, double y) const;
  double lookup_3d_linear(const double &x, const double &y, bool edge_filter = true) const;
  static double interp_linear(double x0, double x1, double y0, double y1, double x);
  double lookup_3d_neighbour(const double &x, const double &y) const;

  virtual double lowest_value() const;
  virtual double highest_value() const;

signals:
  // child models never emit these themselves.
  void selection_changed();
  void data_changed();
  void geometry_changed(); // all renderers need to assume data changed too.

protected:
  void change_data() { __data_changed = true; update_timer.start();};
  void change_geometry()  { __geometry_changed = true; update_timer.start();};
  void change_selection()  { __selection_changed = true; update_timer.start();};

private slots:
  void do_update();

private:
  QVector <table_region>selection;
  bool __selection_changed = false;
  bool __data_changed = false;
  bool __geometry_changed = false;
  QTimer update_timer;
};

class rftablemodelqtproxy : public QAbstractTableModel {
  Q_OBJECT
  // this is a scabby proxy for functions that need an abstract table model.
public:
  rftablemodelqtproxy(rftablemodel *table) {
    this->table = table;
    connect(table,&rftablemodel::data_changed,this,&rftablemodelqtproxy::_emit_change_sig);
    connect(table,&rftablemodel::geometry_changed,this,&rftablemodelqtproxy::_emit_change_sig_2);
  };
  rftablemodel *table = nullptr;
  int rowCount(const QModelIndex &parent = QModelIndex()) const {
    return table->n_rows();
  };
  int columnCount(const QModelIndex &parent = QModelIndex()) const {
    return table->n_columns();
  };
  QVariant data(const QModelIndex &index, int role = Qt::DisplayRole) const {
    if(role != Qt::DisplayRole) return QVariant();
    return table->double_at(index.row(),index.column());
  };
  QVariant headerData(int section, Qt::Orientation orientation, int role = Qt::DisplayRole) const {
    if(role != Qt::DisplayRole) return QVariant();
    switch(orientation) {
    case Qt::Horizontal:
      return table->col_name(section).toDouble();
      break;
    case Qt::Vertical:
      return table->row_name(section).toDouble();
      break;
    }
    return QVariant();
  };
public slots:
  void _emit_change_sig() {
    emit dataChanged(index(0,0),index(rowCount() - 1,columnCount() - 1));
  };
  void _emit_change_sig_2() {
    emit layoutChanged();
    emit dataChanged(index(0,0),index(rowCount() - 1,columnCount() - 1));
  };
};

#endif // RFTABLEMODEL_H
