#include "regression.h"

regression::regression() {
  clear();
}

int regression::calculate() {
  if(size() == 0) return 0;
  calc_coefficient();
  calc_constant_term();
  return size();
}

bool regression::is_valid() const {
  if(size() == 0) return false;
  return true;
}

bool regression::data_reliable(double max_err_sq) const {
  if(is_valid() == false) return false;
  if(size() < 5) return false; // .. 5 points seems a reasonable minimum anywhere
  double err = error_square();
  qDebug() << size() << err << max_err_sq;
  if(err != err) return false; // filter NaN
  if(err > max_err_sq) return false;
  return true;
}

void regression::clear() {
  x_values.clear();
  y_values.clear();
  coeff = 0;
  constTerm = 0;
  sum_y = 0;
  sum_y_square = 0;
  sum_x_square = 0;
  sum_x = 0;
  sum_xy = 0;
  min = __DBL_MAX__;
  max = __DBL_MIN__;
}

void regression::calc_coefficient() {
  double N = size();
  double numerator = (N * sum_xy - sum_x * sum_y);
  double denominator = (N * sum_x_square - sum_x * sum_x);
  coeff = numerator / denominator;
}

void regression::calc_constant_term() {
  double N = size();
  double numerator = (sum_y * sum_x_square - sum_x * sum_xy);
  double denominator = (N * sum_x_square - sum_x * sum_x);
  constTerm = numerator / denominator;
}

int regression::size() const {
  return x_values.size();
}

void regression::add_point(const double &x, const double &y) {
  if(y < min) min = y;
  if(y > max) max = y;
  sum_xy += x * y;
  sum_x += x;
  sum_y += y;
  sum_x_square += x * x;
  sum_y_square += y * y;
  x_values.append(x);
  y_values.append(y);
}

double regression::at(double x) const {
  return coeff * x + constTerm;
}

double regression::error_square() const {
  double ans = 0;
  for (int i = 0; i < x_values.size(); i++) {
    ans += ((at(x_values[i]) - y_values[i])
            * (at(x_values[i]) - y_values[i]));
  }
  return ans;
}

double regression::error_at(double num) const {
  for (int i = 0;
       i < x_values.size(); i++) {
    if (num == x_values[i]) {
      return (y_values[i] - at(x_values[i]));
    }
  }
  return 0;
}

regression_set::regression_set(QVector<double> points) {
  configure(points);
}

regression_set::regression_set() {

}

void regression_set::configure(QVector<double> points) {
  this->points = points;
  point_medians.resize(points.size() - 1);
  for(int x=0;x<point_medians.size();x++) {
    point_medians[x] = (points.at(x) + points.at(x+1)) / 2;
  }
  values.resize(point_medians.size());
}

void regression_set::calculate() {
  avg_err = 0;
  int n = 0;
  for(int x=0;x<values.size();x++) {
    values[x].calculate();
    double e = values[x].error_square();
    if(e == e) {
      avg_err += e;
      n++;
    }
  }
  if(n > 0) {
    avg_err = avg_err / (double)n;
  } else {
    avg_err = __DBL_MAX__;
  }
  qDebug() << "CALC AVG ERR" << values.size() << avg_err;
}

int regression_set::count(const double &x) const {
  int i = locate_index(x);
  return values[i].size();
}

double regression_set::min(const double &x) const {
  int i = locate_index(x);
  return values[i].min;
}

double regression_set::max(const double &x) const {
  int i = locate_index(x);
  return values[i].max;
}

void regression_set::print() {

}

double regression_set::err_square(const double &x) const {
  int i = locate_index(x);
  return values[i].error_square();
}

void regression_set::add_data(const double &x, const double &y) {
  int i = locate_index(x);
  values[i].add_point(x,y);
}

double regression_set::at(const double &x) const {
  int i = locate_index(x);
  int count = 0;
  double out = 0.00;
  double max_err = this->max_err();
  if(values.at(i).data_reliable(max_err) == true) {
    out = values.at(i).at(x);
    count++;
  }
  // pull the next cell into average as well if that data is reliable
  if(x < point_medians[i]) {
    i--;
    if(i >= 0 && values.at(i).data_reliable(max_err) == true) {
      out += values.at(i).at(x);
      count++;
    }
  } else {  // >=
    i++;
    if(i <= point_medians.size() - 1 && values.at(i).data_reliable(max_err) == true) {
      out += values.at(i).at(x);
      count++;
    }
  }
  if(count == 0 || out != out) return 0.00;
  if(count == 1) return out;
  return out / double(count);
}

bool regression_set::data_reliable(const double &x) const {
  int i = locate_index(x);
  if(values.at(i).data_reliable(max_err()) == true) return true;
  // pull the next cell into average as well if that data is reliable
  if(x < point_medians[i]) {
    i--;
    if(i >= 0 && values.at(i).data_reliable(max_err()) == true) return true;
  } else {  // >=
    i++;
    if(i <= point_medians.size() - 1 && values.at(i).data_reliable(max_err()) == true) return true;
  }
  return false;
}

int regression_set::locate_index(const double &value) const {
  //if(point_medians.size() <= 1) return 0; // array size of at least 2.
  int l = point_medians.size()  - 1;
  while(l > 0 && point_medians.at(l) >= value) l--;
  if(l == point_medians.size() - 1) return point_medians.size() - 1;  // if left value is at end, neighbour is last element for sure.
  double median = (point_medians.at(l) + point_medians.at(l+1)) / 2;
  if(value < median) return l;
  return l + 1;
}

double regression_set::max_err() const {
  return avg_err * 2; // tune me
}

regression_table::regression_table(table_axis x_axis, table_axis y_axis, _analysis_mode mode) {
  this->mode = mode;
  this->x_axis = x_axis;
  this->y_axis = y_axis;

  if(mode != YLINEAR) {
    x_regressions.resize(y_axis.count());
    for(int x=0;x<x_regressions.size();x++) {
      x_regressions[x].configure(x_axis.values);
    }
  }

  if(mode != XLINEAR) {
    y_regressions.resize(x_axis.count());
    for(int y=0;y<y_regressions.size();y++) {
      y_regressions[y].configure(y_axis.values);
    }
  }
}

regression_table::regression_table(table_axis y_axis) {
  mode = YLINEAR;
  this->y_axis = y_axis;
  y_regressions.resize(1);
  y_regressions[0].configure(y_axis.values);
}

void regression_table::add_data(const double &x, const double &y, const double &z) {
  if(mode != YLINEAR) {
    int yi = y_axis.locate_index(y);
    x_regressions[yi].add_data(x,z);
  }

  if(mode != XLINEAR) {
    int xi = x_axis.locate_index(x);
    y_regressions[xi].add_data(y,z);
  }
}

void regression_table::add_data(const double &y, const double &z) {
  y_regressions[0].add_data(y,z);
}

void regression_table::calculate() {
  if(mode != YLINEAR) {
    for(int x=0;x<x_regressions.size();x++) {
      x_regressions[x].calculate();
    }
  }
  if(mode != XLINEAR) {
    for(int y=0;y<y_regressions.size();y++) {
      y_regressions[y].calculate();
    }
  }
  calculated = true;
}

double regression_table::at(const int &xi, const int &yi) const {
  double avg = 0;
  int count = 0;
  if(mode != YLINEAR && x_regressions.at(yi).data_reliable(x_axis.at(xi)) == true) {
    avg += x_regressions.at(yi).at(x_axis.at(xi));
    count++;
  }
  if(mode != XLINEAR && y_regressions.at(xi).data_reliable(y_axis.at(yi)) == true) {
    avg += y_regressions.at(xi).at(y_axis.at(yi));
    count++;
  }
  if(count == 0 || avg != avg) return 0.00;
  return avg / (double)count;
}

double regression_table::at(const int &yi) const {
  if(y_regressions.at(0).data_reliable(y_axis.at(yi)) == true) {
    return y_regressions.at(0).at(y_axis.at(yi));
  }
  return 0.00;
}

int regression_table::count_at(const int &xi, const int &yi) const {
  double count = 0;
  if(mode != YLINEAR) {
    count += x_regressions.at(yi).err_square(x_axis.at(xi));
  }
  if(mode != XLINEAR) {
    count += y_regressions.at(xi).err_square(y_axis.at(yi));
  }
  /*
  if(mode != YLINEAR && x_regressions.at(yi).data_reliable(x_axis.at(xi)) == true) {
    //count += x_regressions.at(yi).err_square(x_axis.at(xi));
    count += x_regressions.at(yi).count(x_axis.at(xi));
  }
  if(mode != XLINEAR && y_regressions.at(xi).data_reliable(y_axis.at(yi)) == true) {
    //count += y_regressions.at(xi).err_square(y_axis.at(yi));
    count += y_regressions.at(xi).count(y_axis.at(yi));
  }*/
  return count;
}

int regression_table::count_at(const int &yi) const {
  if(y_regressions.at(0).data_reliable(y_axis.at(yi)) == true) {
    return y_regressions.at(0).count(y_axis.at(yi));
  }
  return 0.00;
}

double regression_table::min_at(const int &xi, const int &yi) const {
  double min = __DBL_MAX__;
  if(mode != YLINEAR && x_regressions.at(yi).data_reliable(x_axis.at(xi)) == true) {
    double m = x_regressions.at(yi).min(x_axis.at(xi));
    if(m  < min) min = m;
  }
  if(mode != XLINEAR && y_regressions.at(xi).data_reliable(y_axis.at(yi)) == true) {
    double m = y_regressions.at(xi).min(y_axis.at(yi));
    if(m  < min) min = m;
  }
  return min;
}

double regression_table::max_at(const int &xi, const int &yi) const {
  double max = __DBL_MIN__;
  if(mode != YLINEAR && x_regressions.at(yi).data_reliable(x_axis.at(xi)) == true) {
    double m = x_regressions.at(yi).min(x_axis.at(xi));
    if(m  > max) max = m;
  }
  if(mode != XLINEAR && y_regressions.at(xi).data_reliable(y_axis.at(yi)) == true) {
    double m = y_regressions.at(xi).min(y_axis.at(yi));
    if(m  > max) max = m;
  }
  return max;
}

double regression_table::min_at(const int &yi) const {
  double min = __DBL_MAX__;
  if(y_regressions.at(yi).data_reliable(y_axis.at(yi)) == true) {
    double m = y_regressions.at(yi).min(y_axis.at(yi));
    if(m  < min) min = m;
  }
  return min;
}

double regression_table::max_at(const int &yi) const {
  double max = __DBL_MIN__;
  if(y_regressions.at(yi).data_reliable(y_axis.at(yi)) == true) {
    double m = y_regressions.at(yi).min(y_axis.at(yi));
    if(m  > max) max = m;
  }
  return max;
}

bool regression_table::is_null(const int &xi, const int &yi) {
  if(calculated == false) return true;
  if(mode == XLINEAR && x_regressions.at(yi).data_reliable(x_axis.at(xi)) == false) return true;
  if(mode == YLINEAR && y_regressions.at(xi).data_reliable(y_axis.at(yi)) == false) return true;
  if(mode == BILINEAR && x_regressions.at(yi).data_reliable(x_axis.at(xi)) == false &&
     y_regressions.at(xi).data_reliable(y_axis.at(yi)) == false) return true;
  return false;
}

bool regression_table::is_null(const int &yi) {
  if(calculated == false) return true;
  if(y_regressions.at(0).data_reliable(y_axis.at(yi)) == false) return true;
  return false;
}

bool regression_table::dbg() {
  return true;
}
