#include "rfdatalog.h"
#include <QElapsedTimer>
#include <QDebug>
#include <QProgressDialog>

rfdatalog::rfdatalog() {
  static int id_counter = 0;
  _id = id_counter;
  id_counter++;
}

rfdatalog::rfdatalog(const rfdatalog &other) : rfdatalog() {
  configure_log(other.seperator,other.header_line,other.data_line,other.allow_shitty_csv,other.parse_boolean_text,other.max_lines,other.time_col_name);
  data_double = other.data_double;
  data_str = other.data_str;
  col_size_cache = other.col_size_cache;
  extra_col_data = other.extra_col_data;
  extra_col_names = other.extra_col_names;
  extra_col_size_cache = other.extra_col_size_cache;
  column_map = other.column_map;
  columns = other.columns;
  _n_cols = other._n_cols;
  file = nullptr;
}

rfdatalog::~rfdatalog() {
  if(file != nullptr) delete file;
}

int rfdatalog::id() const {
  return _id;
}

void rfdatalog::configure_log(char seperator, int header_line, int data_line,
                              bool allow_shitty_csv, bool parse_boolean_text,
                              int max_lines, const QString &time_col) {
  this->header_line = header_line;
  this->data_line = data_line;
  this->allow_shitty_csv = allow_shitty_csv;
  this->seperator = seperator;
  if(max_lines != -1) {
    this->max_lines = max_lines;
  } else {
    this->max_lines = __INT_MAX__;
  }
  this->parse_boolean_text = parse_boolean_text;
  this->time_col_name = time_col;
}

QStringList rfdatalog::common_columns(QVector<rfdatalog *> logs) {
  // build list of matching columns.
  QStringList all_cols;
  for(int x=0;x<logs.size();x++) {
    QStringList l = logs.at(x)->column_string_list();
    while(l.isEmpty() == false) {
      QString s = l.takeFirst();
      if(all_cols.contains(s) == false) all_cols.append(s);
    }
  }
  QStringList ok_cols = all_cols;
  while(all_cols.isEmpty() == false) {
    QString s = all_cols.takeFirst();
    for(int x=0;x<logs.size();x++) {
      if(logs.at(x)->index_of(s) == -1) {
        ok_cols.removeAll(s);
      }
    }
  }
  return ok_cols;
}

int rfdatalog::seek(const int &current_row, const double &time) const {
  if(time == 0.00) return current_row;
  if(time_col == -1) {
    int r = current_row + time;
    if(r < 0 || r > n_rows() - 1) return -1;
    return r;
  } else {
    double current_time = double_at(current_row,time_col);
    if(time > 0) {
      for (int x=current_row;x<n_rows();x++) {
        if(double_at(x,time_col) >= current_time + time) return x;
      }
    } else {
      for (int x=current_row;x>=0;x--) {
        if(double_at(x,time_col) <= current_time - time) return x;
      }
    }
  }
  return -1; // fell off the edge
}

bool rfdatalog::is_valid() const {
  if(_n_cols == -1) return false;
  return true;
}

bool rfdatalog::load_file(const QString &filename) {
  parse_errors.clear();
  columns.clear();
  column_map.clear();
  col_size_cache.clear();

  if(file != nullptr) delete file;
  file = new QFile();

  file->setFileName(filename);

  if(file->open(QFile::ReadOnly) == false) {
    parse_errors.append(file->errorString());
    return false;
  }

  _n_cols = -1;

  data_str.clear();
  data_double.clear();
  data_str.reserve(file->size() / 4);
  data_double.reserve(file->size() / 4);

  QDataStream d(file);

  QElapsedTimer t2;
  t2.start();
  if(parse_csv_new(&d,!allow_shitty_csv,seperator,max_lines,&parse_errors,file->size()) == false) {
    parse_errors.append("General parser error");
    _n_cols = -1;
    data_str.clear();
    data_double.clear();
    return false;
  }
  //qDebug() << "CSV BENCHMARK" << t2.elapsed() << "WITH FAST CONVERTER" << fast_converter;

  if(n_columns() == -1) {
    parse_errors.append("Parser found no columns.");
    return false;
  }

  if(n_rows() <= 0) {
    parse_errors.append("Parser found no data.");
    return false;
  }

  // get max value sizes
  col_size_cache.resize(n_columns());
  for(int col=0;col<n_columns();col++) {
    int cs = 0;
    for(int row=0;row<n_rows();row++) {
      int s = data_str.at(data_index(row,col)).size();
      if(s > cs) cs = s;
    }
    col_size_cache[col] = cs;
  }

  // get header
  for(int x=0;x<n_columns();x++) {
    QString s = header_at(x);
    if(col_size_cache[x] < s.size()) col_size_cache[x] = s.size(); // allow col header size to increase col size
    columns.append(s);
    column_map[s] = x;
  }

  // set time col (or -1 which is cool too)
  time_col = index_of(time_col_name);

  change_geometry();
  change_data();

  return true;
}

inline int rfdatalog::data_index(const int &row, const int &col) const {
  return ( ( row + 1 ) * _n_cols ) + col;
}

inline int rfdatalog::header_index(const int &col) const {
  return col;
}

QVector<double> rfdatalog::col_vector(const int &col) const {
  // for row ordered data we cache the column version of it for next time.
  if(col < _n_cols) {
    if(col_cache.contains(col)) return col_cache.value(col);
    QVector <double>out(n_rows());
    for(int row=0;row<n_rows();row++) {
      int i = ( ( row + data_line ) * _n_cols ) + col;
      out[row] = data_double.at(i);
    }
    col_cache[col] = out;
    return col_cache.value(col);
  } else {
    return extra_col_data[col - _n_cols];
  }
}

int rfdatalog::n_rows() const {
  return (data_double.size() / _n_cols) - 1;
}

QString rfdatalog::col_name(const int &col) const {
  if(col < _n_cols) {
    return columns.at(col);
  } else {
    return extra_col_names.at(col - _n_cols);
  }
}

QString rfdatalog::row_name(const int &row) const {
  return QString::number(row);
}

int rfdatalog::col_custom_size(const int &col) const {
  if(col < _n_cols) {
    return col_size_cache.at(col);
  } else {
    return extra_col_size_cache.at(col - _n_cols);
  }
}

QString rfdatalog::string_at(const int &row, const int &col) const {
  if(col < _n_cols) {
    return data_str.at(data_index(row,col));
  } else {
    return QString::number(double_at(row,col));
  }
}

QString rfdatalog::header_at(const int &col) const {
  return data_str.at(header_index(col));
}

double rfdatalog::double_at(const int &row, const int &col) const {
  if(col < _n_cols) {
    return data_double.at(data_index(row,col));
  } else {
    return extra_col_data[col - _n_cols][row];
  }
}

QStringList rfdatalog::column_string_list(bool exclude_extra_columns) const {
  QStringList out = columns;
  if(!exclude_extra_columns) out.append(extra_col_names);
  return out;
}

bool rfdatalog::matches(const QStringList &columns) const {
  if(columns == this->columns) return true;
  return false;
}

int rfdatalog::index_of(const QString &col_name) const {
  int result = column_map.value(col_name,-1);
  if(result == -1) {
    int i = extra_col_names.indexOf(col_name);
    if(i != -1) return _n_cols + i;
    return -1;
  } else {
    return result;
  }
}

int rfdatalog::n_columns() const {
  return _n_cols + extra_col_names.size();
}

bool rfdatalog::parse_csv_new(QDataStream *stream, bool errors_fatal, char seperator,
                              int max_lines, QStringList *parse_errors, int file_size) {

  if(header_line >= data_line) {
    parse_errors->append("Header must come before data.");
    return false;
  }

  const unsigned int chunk_size = 1024 * 4;
  QByteArray buf; // working buffer.
  buf.resize(chunk_size);

  const unsigned int max_field_size = 1024;
  QByteArray f; // current working field buffer
  f.reserve(max_field_size); // per field

  int cols = 0; // cols on current line
  int line_number = 0;

  enum _state {
    NORMAL,
    BACKSLASHED,
    QUOTED
  };
  _state state = NORMAL;
  bool just_quoted = false;
  int line_cursor = -1;
  _n_cols = -1;
  bool overrun = false;
  char last_char = 0x00;
  char c = 0x00;

  QProgressDialog d("Parsing CSV...","Abort",0,file_size);
  d.setWindowModality(Qt::WindowModal);
  d.setMinimumDuration(650);
  d.setRange(0,file_size);

  unsigned int read_bytes = 0;

  while(stream->atEnd() == false) {
    if(d.wasCanceled()) {
      parse_errors->append("Operation was cancelled.");
      return false;
    }

    // read chunk
    unsigned int length = stream->readRawData(buf.data(),chunk_size);

    read_bytes += length;
    d.setValue(read_bytes);

    // always line terminate the file.
    if(length < chunk_size) {
      buf[length] = '\n';
      length++;
    }

    for(unsigned int x=0;x<length;x++) {

      if(line_number >= data_line && f.size() > max_field_size) {
        parse_errors->append("Max field size exceeded, probably wrong seperator or misplaced quotation mark " + QString::number(line_number));
        return false;
      }

      line_cursor++;
      last_char = c;
      c = buf[x];

      if(overrun && c != '\r' && c != '\n') {
        continue; // continue until linefeed on column overrun
      }

      if(last_char == '\r' && c == '\n') continue; // shortcut dos CRLF

      switch(state) {
      case NORMAL:
      {
        switch(c) {
        case '"':
          if(handle_quotes == false) {
            f.append('"');
          } else {
            if(handle_literal_quotes == true) just_quoted = true;
            state = QUOTED;
          }
          break;

        case '\\':
          if(handle_backslash) {
            state = BACKSLASHED;
          } else {
            f.append('\\');
          }
          break;
          // seperator handling
        case '\t':
        case ',':
          if(c == seperator) {
            if(line_number == header_line || line_number >= data_line) append_raw_data(&f);  // CHANGE TO CORRECT OUTPUT FUNCTION
            f.clear();
            cols++;

            // check for column overrun
            if(line_number >= data_line && _n_cols > -1 && cols >= _n_cols) {
              if(errors_fatal) {
                parse_errors->append("Column count high on line " + QString::number(line_number));
                return false;
              }
              overrun = true;
            }
          } else {
            f.append(c);
          }
          break;

          // END OF LINE
        case '\r':
        case '\n':          
          if(overrun == false && line_cursor != 0) { // this effectively skips blank lines and multiple cr/lf entries in a row.
            if(line_number == header_line || line_number >= data_line) append_raw_data(&f);  // CHANGE TO CORRECT OUTPUT FUNCTION
            f.clear();
            cols++;

            if(line_number >= data_line && _n_cols > -1 && cols != 1 && cols != _n_cols) {
              if(errors_fatal) {
                parse_errors->append("Column count low on line " + QString::number(line_number));
                return false;
              } else {
                while(cols < _n_cols) {
                  append_raw_data(&f);
                  cols++;
                }
              }
            }

            // INSERT ANY "NEXT LINE" OUTPUT FUNCTIONS HERE
            if(line_number > max_lines) return true;
          }

          // if we are in the header pull column count.
          if(line_number == header_line) {
            if(cols == 0) {
              parse_errors->append("Header is empty");
              return false;
            }
            _n_cols = cols;
          }

          line_number++;
          line_cursor = -1;
          cols = 0;
          overrun = false;

          break;

        default:
          f.append(c);
        }
      }
        break;
      case BACKSLASHED:
        f.append(c);
        state = NORMAL;
        break;
      case QUOTED:
      {
        switch(c) {
        case '"':
          state = NORMAL;
          if(just_quoted == true) {
            f.append('"'); // double quote is literal quote
            just_quoted = false;
          }
          break;
        default:
          just_quoted = false;
          f.append(c);
        }
      }
        break;
      }

    }
  }

  return true;
}

inline void rfdatalog::append_raw_data(QByteArray *f) {
  data_str.append(*f); // always append the string

  // ADD STRINGS THAT MEAN 1 HERE
  static QSet <QByteArray>positive_values = {"ON","FALSE","OPEN","ACTIVE","FULL","GOOD","OK","OKAY","RUNNING","GO","YES","IN"};

  bool ok;
  double x = f->toDouble(&ok); // attempt double conversion,

  if(ok == false) { // if failed...
    if(parse_boolean_text == true && positive_values.contains(f->toUpper())) {
      data_double.append(1.00); // append 1 if we can decode a boolean value
    } else {
      data_double.append(0.00); // otherwise zero
    }
  } else {
    data_double.append(x); // if success append value
  }
}

void rfdatalog::reload_dynamic_columns(dynamic_column_list *list) {
  extra_col_data.clear();
  extra_col_names.clear();
  extra_col_size_cache.clear();
  for(int x=0;x<list->size();x++) {
    if(list->at(x).is_compatible(this)) {
      extra_col_names.append(list->at(x).name);
      extra_col_data.append(QVector<double>(n_rows()));
      extra_col_size_cache.append(list->at(x).name.size());
      int col = extra_col_data.size() - 1;
      for(int row=0;row<n_rows();row++) {
        extra_col_data[col][row] = list->at(x).value(this,row);
      }
    }
  }
  qDebug() << "CHANGE_DATA REACHED IN RELOAD_DYN";
  change_geometry(); // implies change in data
}

