#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "widgets/rftablewidget.h"
#include "modules/module_analyzer.h"
#include "modules/module_datalog.h"
#include "widgets/module_container_widget.h"
#include "modules/module_table.h"
#include "modules/module_compare.h"
#include "modules/module_grapher2d.h"
#include "dialogs/datalog_load_dialog.h"
#include "dialogs/import_xdf_dialog.h"
#include "widgets/filter_list_widget.h"
#include "datastructures/table_geometry.h"
#include "dialogs/dynamic_col_dialog.h"

#include <QClipboard>

#include <QListWidget>
#include <QFileDialog>
#include <QDebug>
#include <QProgressDialog>
#include <QSettings>
#include <QMessageBox>

MainWindow::MainWindow(QWidget *parent)
  : QMainWindow(parent)
  , ui(new Ui::MainWindow) {

  qApp->setHighDpiScaleFactorRoundingPolicy(Qt::HighDpiScaleFactorRoundingPolicy::Round);

  ui->setupUi(this);

  QCoreApplication::setOrganizationName("ecmhack");
  QCoreApplication::setOrganizationDomain("ecmhack.com");
  QCoreApplication::setApplicationName("tablehack");
  QCoreApplication::setApplicationVersion("v1.8BETA");

  database::init();

  update_selection_context();
  connect(ui->list,&QListWidget::itemSelectionChanged,this,&MainWindow::update_selection_context);

  // restore splitter
  {
    QList <int>sizes;
    sizes.append(database::get_config("SPLIT_A",271).toInt());
    sizes.append(database::get_config("SPLIT_B",893).toInt());
    ui->splitter->setSizes(sizes);
  }
}

MainWindow::~MainWindow() {
  QVector<module *> mods = modules();
  while(mods.isEmpty() == false) delete mods.takeLast();
  closing = true;
  delete ui;
}

QVector<module *> MainWindow::selected_modules() {
  QVector <module*>out;
  if(closing) return out;
  QList <QListWidgetItem*>selection = ui->list->selectedItems();
  while(selection.isEmpty() == false) out.append(module_for_list_item(selection.takeFirst()));
  return out;
}

QVector<module *> MainWindow::modules() const {
  QVector <module*>out;
  for(int x=0;x<ui->list->count();x++) {
    out.append(module_for_list_item(ui->list->item(x)));
  }
  return out;
}

module* MainWindow::module_for_list_item(QListWidgetItem *i) const {
  return module_list.value(i->data(Qt::UserRole).toUInt());
}

void MainWindow::clean_dependancies() {
  QVector <module*>list = modules();
  for(int x=0;x<list.size();x++) {
    QVector <int>deps = list.at(x)->children();
    for(int y=0;y<deps.size();y++) {
      if(module_list.contains(deps.at(y)) == false) {
        list.at(x)->remove_child(deps.at(y));
      }
    }
  }
}

void MainWindow::add_module(module *m) {
  ui->list->addItem(m->listwidget);
  ui->display->add_module(m);
  module_list[m->id()] = m;

  // select new item by default.
  ui->list->clearSelection();
  m->listwidget->setSelected(true);
}

void MainWindow::errormsg(const QString &s) {
  QMessageBox m(QMessageBox::Critical,"Error",s,QMessageBox::Ok);
  m.exec();
}

void MainWindow::on_delete_btn_clicked() {
  while(selected_modules().isEmpty() == false) {
    module *m = selected_modules().at(0);
    if(m->children().isEmpty() == false) {
      QStringList module_names;
      QVector <module*>tree = child_tree(m);
      while(tree.isEmpty() == false) module_names.append("*" + tree.takeFirst()->name());
      QMessageBox msgBox;
      msgBox.setText("Other modules may depend on what you are trying to delete.");
      msgBox.setInformativeText("If you delete it, they will also be removed:\n" + module_names.join('\n') +
                                "\nAre you sure?");
      msgBox.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
      msgBox.setDefaultButton(QMessageBox::Ok);
      int ret = msgBox.exec();
      if(ret == QMessageBox::Cancel) return;
    }
    remove_module(m);
  }
  //clean_dependancies();
}

QVector<module*> MainWindow::module_parents(module *m) {
  QVector <module*>modules = this->modules();
  QVector <module*>out;
  for(int x=0;x<modules.size();x++) {
    if(modules.at(x)->children().contains(m->id())) out.append(modules.at(x));
  }
  return out;
}

QVector<module*> MainWindow::child_tree(module *m) {
  QVector <module*>out;
  if(m == nullptr) return out;
  out.append(m); // add ourselves first
  for(int x=0;x<m->children().size();x++) {
    module *branch = module_list.value(m->children().at(x));
    QVector <module*>in = child_tree(branch); // recursively decend tree here
    while(in.isEmpty() == false) { // slow tree prune
      module *c = in.takeFirst();
      if(out.contains(c) == false) out.append(c); // do not add duplicate entries
      // .. this should also break tight circular dependancies, wide dependancies are not expected.
    }
  }
  return out;
}

void MainWindow::remove_module(module *m) {
  if(m == nullptr) return;
  QVector <module*>tree = child_tree(m); // flat child tree.
  while(tree.isEmpty() == false) {
    module *m = tree.takeFirst();
    int id = m->id();
    module_list.remove(id);
    delete m;
  }
}


void MainWindow::update_selection_context() {
  if(closing) return;
  QVector <module*>modules = selected_modules();

  ui->delete_btn->setEnabled(modules.size() > 0); // delete if any selection

  // enable analyzer only if datalogs (any amount of them) are selected.
  bool only_datalogs_selected = false;
  if(modules.size() > 0) {
    only_datalogs_selected = true;
    for(int x=0;x<modules.size();x++) {
      if(modules.at(0)->module_type() != module::MODULE_DATALOG) {
        only_datalogs_selected = false;
        break;
      }
    }
  }
  ui->analysis_btn->setEnabled(only_datalogs_selected);

  bool only_tables_selected = false;
  if(modules.size() > 0) {
    only_tables_selected = true;
    for(int x=0;x<modules.size();x++) {
      if(modules.at(0)->module_type() == module::MODULE_DATALOG) {
        only_tables_selected = false;
        break;
      }
    }
  }

  ui->graph_btn->setEnabled(modules.size() == 1 && modules.at(0)->module_type() == module::MODULE_TABLE);
  ui->graph_2d_btn->setEnabled(modules.size() == 1 && modules.at(0)->module_type() == module::MODULE_DATALOG);

  ui->table_btn->setEnabled(true);

  ui->compare_btn->setEnabled(only_tables_selected);

}

void MainWindow::on_splitter_splitterMoved(int pos, int index) {
  database::set_config("SPLIT_A",ui->splitter->sizes().at(0));
  database::set_config("SPLIT_B",ui->splitter->sizes().at(1));
}

void MainWindow::on_advanced_btn_clicked() {
  if(adv == nullptr) adv = new advanced();
  adv->show();
}

void MainWindow::on_datalog_btn_clicked() {
  QString last_dir = database::get_config("LAST_LOG_DIR").toString();
  QStringList files = QFileDialog::getOpenFileNames(nullptr,"Select log file",last_dir,
                                                    QString(),nullptr,QFileDialog::DontUseNativeDialog);

  if(files.isEmpty()) return;

  // configure based on the first file.
  QString first_file = files.at(0);

  // remember this directory.
  QFileInfo i(first_file);
  database::set_config("LAST_LOG_DIR",i.dir().path());

  // configuration dialog
  datalog_load_dialog d(first_file);
  if(d.exec() == datalog_load_dialog::Rejected) return;

  QApplication::setOverrideCursor(Qt::WaitCursor);

  QStringList errors;

  QProgressDialog prog("Loading logs...","Stop",0,files.size());
  prog.setMinimumDuration(250);
  prog.show();

  for(int x=0;x<files.size();x++) {
    prog.setValue(x);

    QString filename = files.at(x);
    QFileInfo i(filename);
    QString basename = i.baseName();

    prog.setLabelText("Loading " + basename);
    if(prog.wasCanceled()) break;

    rfdatalog *log = new rfdatalog();
    log->name = basename;
    d.config_datalog(log,false);

    QApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
    bool status = log->load_file(filename);

    if(status == false) {
      errors.append(filename + ":\n" + errors.join('\n'));
      delete log;
    } else {
      dynamic_column_list dyn = dynamic_columns();
      log->reload_dynamic_columns(&dyn);
      module_datalog *w = new module_datalog(log,log->name);
      add_module(w);
    }
  }

  if(errors.isEmpty() == false) errormsg("Some logs were not loaded.  Errors below:\n" + errors.join('\n'));

  QApplication::restoreOverrideCursor();
}

dynamic_column_list MainWindow::dynamic_columns() const {
  dbdata db;
  db.from_string(database::get_config("DYNCOL").toString(),"DYNCOL");
  dynamic_column_list dyn;
  dyn.import_data(db);
  return dyn;
}

void MainWindow::on_analysis_btn_clicked() {
  QVector <module*>selection = selected_modules();
  if(selection.size() == 0) return;
  QVector <rfdatalog*>logs;
  for(int x=0;x<selection.size();x++) {
    if(selection.at(x)->module_type() != module::MODULE_DATALOG) continue;
    module_datalog *d = (module_datalog*)selection.at(x);
    logs.append(d->get_datalog());
  }
  module *w = new module_analyzer(logs);
  for(int x=0;x<selection.size();x++) {
    if(selection.at(x)->module_type() != module::MODULE_DATALOG) continue;
    selection.at(x)->add_child(w->id());
  }
  add_module(w);
}

void MainWindow::on_table_btn_clicked() {
  if(QGuiApplication::keyboardModifiers().testFlag(Qt::ControlModifier)) {
    QVector <module*>selection = selected_modules();
    if(selection.size() == 1 && selection.at(0)->module_type() != module::MODULE_DATALOG) {
      module *m = new module_table(selection.at(0)->get_data());
      m->set_name(selection.at(0)->name() + " Copy");
      add_module(m);
      return;
    }
  }
  module *m = new module_table();
  add_module(m);
}

void MainWindow::on_graph_btn_clicked() {
  QVector <module*>selection = selected_modules();
  if(selection.size() != 1) return;
  module *p = selection.at(0); // parent module
  if(p->module_type() != module::MODULE_TABLE)  return;
  if(p->get_data() == nullptr) return;

  module_grapher *g = new module_grapher(p);
  p->add_child(g);
  add_module(g);
}

void MainWindow::on_list_itemSelectionChanged() {
  QVector <module*>selected = selected_modules();
  ui->display->set_active_modules(selected);
}

void MainWindow::on_compare_btn_clicked() {
  QVector <module*>selection = selected_modules();
  if(selection.isEmpty()) return;
  module_compare *c = new module_compare(selection);
  for(int x=0;x<selection.size();x++) selection.at(x)->add_child(c);
  add_module(c);
}

void MainWindow::on_graph_2d_btn_clicked() {
  QVector <module*>selection = selected_modules();
  if(selection.size() != 1) return;
  module *p = selection.at(0); // parent module
  if(p->module_type() != module::MODULE_DATALOG) return;
  if(p->get_data() == nullptr) return;
  module_grapher2d *g = new module_grapher2d(p);
  p->add_child(g);
  add_module(g);
}

void MainWindow::on_about_btn_clicked() {
    errormsg("Error: There is no real about dialog yet.\n"
           "This program was written by Steve Haslin of ecmhack.com.\n"
           "His email address is resfilter@resfilter.net.\n"
           "Please see ecmhack.com for more details.");
}

void MainWindow::on_clone_btn_clicked() {
  QVector <module*>selection = selected_modules();
  for(int x=0;x<selection.size();x++) {
    module *m = selection.at(x)->clone();
    if(m == nullptr) continue; // skip un-clonable modules
    add_module(m);
  }
}

QVector <module*> MainWindow::modules_of_type(module::_module_type mod_type) const {
  QVector <module*>m = modules();
  QVector <module*>out;
  for(int x=0;x<m.size();x++){
    if(m.at(x)->module_type() == mod_type) out.append(m.at(x));
  }
  return out;
}

QVector <rfdatalog*> MainWindow::all_datalogs() const {
  QVector <module*>m = modules_of_type(module::MODULE_DATALOG);
  QVector <rfdatalog*>logs;
  for(int x=0;x<m.size();x++) logs.append((rfdatalog*)m.at(x)->get_data());
  return logs;
}

void MainWindow::on_dynamic_col_btn_clicked() {
  // get list of all logs
  QVector <rfdatalog*>logs = all_datalogs();
  if(logs.isEmpty()) {
    errormsg("This feature requires logs to be loaded to acquire column names.");
    return;
  }

  dynamic_column_list dyn = dynamic_columns();

  dynamic_col_dialog d(logs,&dyn);

  if(d.exec() == dynamic_col_dialog::Accepted) {
    QProgressDialog p;
    p.setMaximum(logs.size());
    p.show();
    for(int x=0;x<logs.size();x++) {
      p.setValue(x);
      logs[x]->reload_dynamic_columns(&dyn);
    }
    // save to database
    database::set_config("DYNCOL",dyn.export_data().to_string("DYNCOL"));
  }
}
