#include "module_analyzer.h"
#include "ui_module_analyzer.h"
#include "../dialogs/table_config_dialog.h"
#include "../widgets/filter_list_widget.h"
#include "../database.h"
#include "../dialogs/database_dialog.h"
#include "../dialogs/color_scale_dialog.h"
#include <QInputDialog>

#include <QDebug>

void module_analyzer::set_view(const QString &view) {
  anl->set_view(view);
}

void module_analyzer::get_view() {
  QString s = anl->current_view();
  if(s == "AVG") {
    ui->avg_btn->setChecked(true);
  } else if(s == "MIN") {
    ui->min_btn->setChecked(true);
  } else if(s == "MAX") {
    ui->max_btn->setChecked(true);
  } else if(s == "COUNT") {
    ui->count_btn->setChecked(true);
  }
}

rftablemodel *module_analyzer::get_data() {
  return (rftablemodel*)anl;
}

module_analyzer::module_analyzer(QVector<rfdatalog *> logs) :
  module(nullptr),
  ui(new Ui::module_analyzer) {
  ui->setupUi(this);

  set_icon("share_icon&16.png");

  this->logs = logs;

  ui->data_selector->set_columns(rfdatalog::common_columns(logs));
  anl = new rfanalysis();

  ui->calc_err->setHidden(true);

  ui->mode_box->setCurrentIndex(anl->method);

  ui->cut_btn->setChecked(true);

  connect(ui->data_selector,&column_button::changed,this,&module_analyzer::new_data_selected);

  ui->renderer->set_data((rftablemodel*)anl);

  for(int x=0;x<logs.size();x++) {
    connect(logs.at(x),&rftablemodel::data_changed,this,&module_analyzer::source_log_data_changed);
    connect(logs.at(x),&rftablemodel::geometry_changed,this,&module_analyzer::source_log_data_changed);
  }
}

void module_analyzer::source_log_data_changed() {
  analyze();
}

void module_analyzer::new_data_selected() {
  if(ui->data_selector->is_valid() == false) {
    anl->set_data_col(QString());
  } else {
    anl->set_data_col(ui->data_selector->col_name);
    ui->renderer->set_precision(determine_precision(anl->data_col()));
  }
  refresh_name();
  analyze();
}

int module_analyzer::determine_precision(const QString &col_name) {
  int out = 0;
  for(int x=0;x<logs.size();x++) {
    int precision = determine_precision(col_name,logs.at(x));
    if(precision > out) out = precision;
  }
  return out;
}

int module_analyzer::determine_precision(const QString &col_name, rfdatalog *log) {
  double max = __DBL_MIN__;

  int idx = log->col_index(col_name);
  if(idx == -1) return 0;

  // determine max number size (+/-) up to record limit.
  int limit = 5000;
  for(int x=0;x<log->n_rows() && x < limit;x++) {
    double d = log->double_at(x,idx);
    if(d < 0) d = 0 - d;
    if(d > max) max = d;
  }

  if(max < 1) return 3;
  if(max < 10) return 2;
  if(max < 100) return 1;

  return 0;
}

module_analyzer::~module_analyzer() {
  delete anl;
  delete ui;
}

void module_analyzer::analyze() {
  if(is_valid() == false)  return;
  QApplication::setOverrideCursor(Qt::WaitCursor);
  QApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
  anl->analyze(logs);
  QApplication::restoreOverrideCursor();
  if(anl->errors.isEmpty() == false) {
    error("Analysis errors:\n" + anl->errors.join("\n"));
  }
}

void module_analyzer::configure_layout() {
  table_config_dialog d(logs);
  d.configure(anl->x_axis(),anl->y_axis());
  if(d.exec() == table_config_dialog::Accepted) {
    anl->set_layout(d.get_xaxis(),d.get_yaxis());
    refresh_name();
    analyze();
  }
}

void module_analyzer::reset_ui_context() {

}

dbdata module_analyzer::export_data() const {
  dbdata out;
  if(is_valid() == false) return out;
  out.table_name = "ANALYSIS";
  out.set_subdata("ANL",anl->export_data());
  out.name = this->name_override;
  out.set_subdata("RENDER",ui->renderer->export_data());
  return out;
}

bool module_analyzer::import_data(const dbdata &data) {
  if(anl->import_data(data.subdata("ANL")) == false) return false;

  bool refactor_success = true;
  for(int x=0;x<logs.size();x++) {
    if(anl->refactor(logs.at(x)) == false) {

      refactor_success = false;
      // FIXME.  WE CAN EITHER REDIRECT THE FAILED FIELDS OR DELETE THE FAILED LOGS.
    }
  }
  if(refactor_success == false) {
    error("Refactoring your selection failed due to missing columns.\n"
          "This will be a feature improvement later on.  Cancelling.");
    return false;
  }

  QSignalBlocker s1(ui->lagfilter_box);
  QSignalBlocker s2(ui->mincount_box);
  QSignalBlocker s3(ui->data_selector);
  QSignalBlocker s4(ui->mode_box);

  switch(anl->display) {
  case rfanalysis::DISPLAY_MIN:
    ui->min_btn->setChecked(true);
    break;
  case rfanalysis::DISPLAY_MAX:
    ui->max_btn->setChecked(true);
    break;
  case rfanalysis::DISPLAY_AVG:
    ui->avg_btn->setChecked(true);
    break;
  case rfanalysis::DISPLAY_COUNT:
    ui->count_btn->setChecked(true);
    break;
  case rfanalysis::DISPLAY_CALC:
    ui->calc_btn->setChecked(true);
    break;
  }

  ui->lagfilter_box->setValue(anl->lagfilter);
  ui->mincount_box->setValue(anl->mincount);
  ui->data_selector->set_column(anl->data_col());

  ui->renderer->import_data(data.subdata("RENDER"));

  ui->mode_box->setCurrentIndex(anl->method);
  ui->calc_expression->setText(anl->get_equation());

  set_name(data.name);
  name_override = data.name;
  update_ui();
  return true;
}

void module_analyzer::on_config_btn_clicked() {
  configure_layout();
}

void module_analyzer::refresh_name() {
  if(name_override.isEmpty() == false) {
    set_name(name_override);
    return;
  }

  if(anl->y_axis_name().isEmpty()) {
    set_name(QString());
    return;
  }

  QString out;
  if(anl->data_col().isEmpty()) {
    out.append("?\n");
  } else {
    out.append(anl->data_col() + "\n");
  }
  if(anl->table_type() == rftablemodel::TABLE3D) out.append(" VS " + anl->x_axis_name() + "\n");
  out.append(" VS " + anl->y_axis_name());
  set_name(out);
}

bool module_analyzer::is_valid() const {
  if(anl != nullptr && anl->is_valid() == true) return true;
  return false;
}

void module_analyzer::on_avg_btn_toggled(bool checked) {
  if(checked) set_view("AVG");
}

void module_analyzer::on_max_btn_toggled(bool checked) {
  if(checked) set_view("MAX");
}

void module_analyzer::on_min_btn_toggled(bool checked) {
  if(checked) set_view("MIN");
}

void module_analyzer::on_calc_btn_toggled(bool checked) {
  if(checked) set_view("CALC");
}


void module_analyzer::on_count_btn_toggled(bool checked) {
  if(checked) {
    _precision_memory = ui->renderer->get_precision();
    ui->renderer->set_precision(0);
    set_view("COUNT");
  } else {
    if(_precision_memory != -1) ui->renderer->set_precision(_precision_memory);
  }
}

void module_analyzer::on_filter_btn_clicked() {
  filter_list_dialog d(rfdatalog::common_columns(logs),anl->get_filters());
  if(d.exec() == filter_list_dialog::Accepted) {
    anl->set_filters(d.result);
    update_ui();
    analyze();
  }
}

void module_analyzer::update_ui() {
  ui->filter_btn->setText("Filters (" + QString::number(anl->n_filters()) + ")");
}

void module_analyzer::on_mincount_box_valueChanged(int arg1) {
  anl->mincount = arg1;
  analyze();
}

void module_analyzer::on_lagfilter_box_valueChanged(double arg1) {
  anl->lagfilter = arg1;
  analyze();
}

void module_analyzer::on_database_btn_clicked() {
  dbdata db_out = export_data();
  database_dialog dialog("ANALYSIS",db_out);
  if(dialog.exec() == database_dialog::Accepted) {
    dbdata db = dialog.selected_item;
    if(import_data(db) == false) {
      error("Could not decode analysis.");
      return;
    }
    analyze();
  }
}

void module_analyzer::on_cut_btn_toggled(bool checked) {
  anl->filter_edges = checked;
  analyze();
}

void module_analyzer::on_mode_box_currentIndexChanged(int index) {
  anl->method = (rfanalysis::_analysis_method)index;
  analyze();
}

void module_analyzer::on_calc_expression_editingFinished() {
  if(anl->set_equation(ui->calc_expression->text()) == false) {
    ui->calc_err->setHidden(false);
    ui->calc_err->setText(anl->errors.join('\n'));
    ui->calc_btn->setEnabled(false);
    ui->avg_btn->setChecked(true);
  } else {
    ui->calc_err->setHidden(true);
    ui->calc_btn->setEnabled(true);
    ui->calc_btn->setChecked(true);
  }
}

module *module_analyzer::clone() const {
  module_analyzer *out = new module_analyzer(logs);
  out->set_name(name() + " Copy");
  out->copy_children((module*)this);
  out->import_data(export_data());
  out->analyze();
  return out;
}
