#include "axis_configurator_widget.h"
#include "ui_axis_configurator_widget.h"
#include "database.h"
#include <QDebug>
#include <cmath>
#include <QRegularExpression>
#include <QInputDialog>

axis_configurator_widget::axis_configurator_widget(QWidget *parent) :
  QWidget(parent),
  ui(new Ui::axis_configurator_widget) {
  ui->setupUi(this);
  ui->layout_arbitrary_edit->setPlaceholderText("Enter a sequential numerical value list.");
  refresh_ui();
}

void axis_configurator_widget::load_axis(const QStringList &values) {
  ui->layout_arbitrary_edit->setPlainText(values.join('\n'));
  refresh_ui();
  emit changed();
}

void axis_configurator_widget::load_axis(const table_axis &axis) {
  QString out;
  for(int x=0;x<axis.count();x++) {
    out.append(QString::number(axis.at(x)));
    if(x < axis.count() - 1) out.append('\n');
  }
  ui->layout_arbitrary_edit->setPlainText(out);
  ui->precision_box->setValue(axis.precision());
  emit changed();
}

void axis_configurator_widget::autofill(double min, double max, int count) {
  ui->layout_arbitrary_edit->setPlainText(estimate_values(min,max,count).join('\n'));
}

QStringList axis_configurator_widget::estimate_values(double min, double max, int count) {
  QStringList out;
  static QVector<double>valid_intervals = {0.01, 0.05, 0.10, 0.25, 0.5, 1, 1.5, 5, 10, 15, 20, 25, 50, 100, 150, 200, 250, 500, 1000, 10000, 100000};
  double interval = max - min;
  for(int x=0;x<valid_intervals.size();x++) {
    if(interval / count <= valid_intervals.at(x)) {
      // find start
      double z = valid_intervals.at(x);
      double start = z;
      while(start < min) start += z;
      while(start > min) start -= z;
      QString out;
      while(start <= max) {
        out.append(QString::number(start));
        start += z;
      }
    }
    return out;
  }
  return QStringList();
}

axis_configurator_widget::~axis_configurator_widget() {
  delete ui;
}

int axis_configurator_widget::calc_precision(const QStringList &l) {
  int p = 0;
  for(int x=0;x<l.size();x++) {
    int i = l.at(x).indexOf('.');
    if(i != -1) {
      int ii = l.at(x).length() - i - 1;
      if(ii > p) p = ii;
    }
  }
  return p;
}

QStringList axis_configurator_widget::parse_input() const {
  static QRegularExpression r("\\s+");
  QString in = ui->layout_arbitrary_edit->toPlainText();
  // cheap workaround to accept commas and stuff (just turn them to whitespace)
  in.replace(',',' ');
  in.replace('|',' ');
  //in.replace()
  return in.split(r,Qt::SkipEmptyParts);
}

QVector <double>axis_configurator_widget::get_values(const QStringList &l) const {
  QVector <double>out;
  bool ok;
  for(int x=0;x<l.size();x++) {
    double d = l.at(x).toDouble(&ok);
    if(ok == false) return QVector <double>();
    out.append(d);
  }
  return out;
}

table_axis axis_configurator_widget::generate_axis() const {
  QStringList l = parse_input();
  return table_axis(get_values(l),QString(),ui->precision_box->value());
}

bool axis_configurator_widget::is_valid() const {
  return generate_axis().is_valid();
}

void axis_configurator_widget::refresh_ui() {
  ui->warning_logo->setHidden(true);
  QStringList l = parse_input();
  if(l.isEmpty()) {
    ui->warning_label->setText("Input empty or invalid.");
    return;
  }

  QVector<double>values = get_values(l);
  if(values.isEmpty()) {
    ui->warning_label->setText("Non-numerical input.");
    return;
  }

  table_axis t(values);
  if(t.is_sequential() == false) {
    ui->warning_label->setText("Input is not sequential.");
    return;
  }

  ui->warning_label->setText("OK");
  ui->warning_logo->setHidden(false);
}

void axis_configurator_widget::on_layout_arbitrary_edit_textChanged() {
  refresh_ui();
  ui->precision_box->setValue(calc_precision(parse_input()));
  emit changed();
}

void axis_configurator_widget::on_precision_box_valueChanged(int arg1) {
  emit changed();
}
