#ifndef TABLE_RENDERER_H
#define TABLE_RENDERER_H

#include <QWidget>
#include "../datastructures/rftablemodel.h"
#include <QScrollBar>
#include <QTimer>
#include <QContextMenuEvent>

class rftablewidget_controls;
class dbdata;

class color_scale {
public:
  struct point {
    QColor c;
    double p;
  };
  QColor value(const double &in) const;
  void clear();
  void add_point(double point, QColor color);
  void add_point(const point &p);
  void set_points(QVector <point>points);
  void set_fallback_color(QColor &c) { fallback_color = c; };
  QVector <point>get_points() const;
  static QColor blend(const QColor &scale_color1, const QColor &scale_color2, double r);
  dbdata export_data() const;
  bool import_data(const dbdata &data);
  void estimate(double min_value, double max_value);
private:
  QVector <point>points;
  QColor fallback_color;
};

class rftablewidget : public QWidget {
  Q_OBJECT

public:
  explicit rftablewidget(QWidget *parent = nullptr);
  ~rftablewidget();

  void set_data(rftablemodel *data);
  void change_font(QFont f);
  color_scale get_color_scale() const {return scale;};
  int get_precision() const;

  void wheelEvent(QWheelEvent *event) override;
  void mousePressEvent(QMouseEvent *event) override;
  void mouseDoubleClickEvent(QMouseEvent *event) override;
  void keyPressEvent(QKeyEvent *event) override;
  void contextMenuEvent(QContextMenuEvent *event) override;

  bool import_data(const dbdata &data);
  dbdata export_data() const;

public slots:
  void vscrollbar_changed(int value);
  void hscrollbar_changed(int value);
  void set_font_size(int size);
  void set_precision(int precision);
  int get_font_size() const;
  void select_all();
  void select_none();
  void set_color_scale(bool enable, color_scale scale);
  void selection_to_clipboard(bool cut = false);
  bool paste_from_clipboard();
  void copy_to_clipboard();
  void copy_table();
  void cut_to_clipboard();
  void popup_color_scale_dialog();

  void set_precision0();
  void set_precision1();
  void set_precision2();
  void set_precision3();
  void set_precision4();
  void set_precision5();
  void set_font_sm();
  void set_font_reg();
  void set_font_lg();

  void data_changed();
  void geometry_changed();
  void selection_changed();
  void delayed_redraw();
  void _repaint();

signals:
  void cell_doubleclicked(int row, int col);

private:
  QRect row_label_col_rect;
  QVector <QRect>col_size_map;
  QRect table_area;
  QRect left_area;
  QRect top_area;
  QRect right_area;
  QRect bottom_area;
  QScrollBar hscrollbar;
  QScrollBar vscrollbar;

  void update_widget_geometry();
  int display_row = 0;
  int col_offset = 0;
  int cell_height = 0;

  int margin = 1;
  int space_between_cells = 1;
  void paintEvent(QPaintEvent *event) override;
  void resizeEvent(QResizeEvent *event) override;
  int stretch_width = 0;

  void update_table_geometry();

  bool data_numerical = false;
  bool header_numerical = false;
  int precision = 2;

  QColor color_background;
  QColor color_text;
  QColor color_cell_dark;
  QColor color_cell_light;
  QColor color_cell_selected;
  QColor color_cell_hilight;

  color_scale scale;

  QTimer redraw_delay;

  int row_at_point(const QPoint &p) const;
  int col_at_point(const QPoint &p) const;
  int col_header_at_point(const QPoint &p) const;
  int row_header_at_point(const QPoint &p) const;

  bool data_valid() const;
  QString get_data(const int &row, const int &col) const;

  int col_size_cache = -1;
  int calc_equal_col_size();

  void draw_row(QPainter *p, const int &row);
  void draw_h_header(QPainter *p);

  QRect cell_rect(const int &col, const int &row) const;
  QRect col_rect(const int &col) const;
  QRect row_rect(const int &row) const;
  QRect row_label_rect(const int &row) const;
  QRect btn_rect; // generate
  QImage btn_image;

  rftablemodel *data = nullptr;

  void draw_background(QPainter *p, const QRect &r);
  void draw_background_dark(QPainter *p, const QRect &r);

  void draw_scaled_text(QPainter *p, const QRect &r, const QString &text);
  int n_display_rows() const;
  void draw_rotated_text(QPainter *p, const QString &s, const QRect &r);
  void draw_cell_data(QPainter *p, QRect r, const int &row, const int &col);
  void draw_cell_static(QPainter *p, QRect r, const QString &static_data);
  void draw_cell_static(QPainter *p, const int &col, const int &row, const QString &s, const QColor &bgcolor);
  void draw_cell_static(QPainter *p, QRect r, const QString &s, const QColor &bgcolor);

  void configure_actions();
  QAction *action_copy;
  QAction *action_cut;
  QAction *action_paste;
  QAction *action_color;
  QAction *action_select_all;
  QAction *action_select_none;
  QAction *action_font_sm;
  QAction *action_font_reg;
  QAction *action_font_lg;
  QAction *action_copy_table;
  QVector <QAction*>precision_actions;
  QMenu *create_menu();
};

#endif // TABLE_RENDERER_H
