#include "rftablemodel.h"
#include "table_geometry.h"
#include <QDebug>

rftablemodel::rftablemodel() {
  update_timer.setSingleShot(true);
  update_timer.setInterval(0);
  connect(&update_timer,&QTimer::timeout,this,&rftablemodel::do_update,Qt::QueuedConnection);
}

void rftablemodel::do_update() {
  if(__selection_changed) {
    emit selection_changed();
    __selection_changed = false;
  }

  // geometry changes imply data changes.
  if(__geometry_changed) {
    emit geometry_changed();
    __geometry_changed = false;
    __data_changed = false;
  }

  if(__data_changed) {
    emit data_changed();
    __data_changed = false;
  }
}

bool rftablemodel::readonly() const {
  return true;
}

bool rftablemodel::valid() const {
  if(table_type() == NOTABLE) return false;
  if(n_columns() > 0 && n_rows() > 0) return true;
  return false;
}

QStringList rftablemodel::column_string_list() const {
  QStringList out;
  for(int x=0;x<n_columns();x++) {
    out.append(col_name(x));
  }
  return out;
}

QStringList rftablemodel::row_string_list() const {
  QStringList out;
  for(int x=0;x<n_rows();x++) {
    out.append(row_name(x));
  }
  return out;
}

bool rftablemodel::set(const QString &s, const int &row, const int &col) {
  return false;
}

bool rftablemodel::set(const double &d, const int &row, const int &col) {
  return false;
}

bool rftablemodel::set_null(const int &row, const int &col) {
  return false;
}

int rftablemodel::append_row() {
  return -1;
}

QString rftablemodel::col_name(const int &col) const {
  return QString::number(col);
}

QString rftablemodel::row_name(const int &row) const {
  return QString::number(row);
}

int rftablemodel::col_custom_size(const int &col) const {
  return max_col_name_length(); // default to all equal columns.
}

table_axis rftablemodel::x_axis() const {
  // generate an axis
  table_axis out;
  for(int x=0;x<n_columns();x++) out.values.append(col_name(x).toDouble());
  return out;
}

table_axis rftablemodel::y_axis() const {
  // generate an axis
  table_axis out;
  for(int x=0;x<n_rows();x++) out.values.append(row_name(x).toDouble());
  return out;
}

double rftablemodel::x_axis_value(const int &col) const {
  return x_axis().at(col);
}

double rftablemodel::y_axis_value(const int &row) const {
  return y_axis().at(row);
}

QString rftablemodel::x_axis_name() const {
  return x_axis().name;
}

QString rftablemodel::y_axis_name() const {
  return y_axis().name;
}

int rftablemodel::col_index(const QString &s) const {
  return column_string_list().indexOf(s);
}

int rftablemodel::col_index(const double &d) const {
  // find the first data point that is higher than d.
  for(int x=0;x<n_columns();x++) {
    if(x_axis_value(x) >= d) return x;
  }
  return n_columns() - 1;
}

int rftablemodel::row_index(const double &d) const {
  for(int x=0;x<n_rows();x++) {
    if(y_axis_value(x) >= d) return x;
  }
  return n_rows() - 1;
}

QVector<double> rftablemodel::col_vector(const int &col) const {
  QVector <double>out(n_rows());
  for(int row=0;row<n_rows();row++) {
    out[row] = double_at(row,col);
  }
  return out;
}

QVector<double> rftablemodel::row_vector(const int &row) const {
  QVector <double>out(n_columns());
  for(int col=0;col<n_columns();col++) {
    out[col] = double_at(row,col);
  }
  return out;
}

QVector<double> rftablemodel::value_list() const {
  QVector<double>out;
  for(int col=0;col<n_columns();col++) {
    for(int row=0;row<n_rows();row++) {
      if(is_null(row,col) == false) out.append(double_at(row,col));
    }
  }
  return out;
}

double rftablemodel::double_at(const int &row, const int &col) const {
  return string_at(row,col).toDouble();
}

bool rftablemodel::is_null(const int &row, const int &col) const {
  Q_UNUSED(row);
  Q_UNUSED(col);
  return false;
}

int rftablemodel::max_value_length(const int &col) const {
  int max = 0;
  for(int x=0;x<n_rows();x++) {
    int s = string_at(x,col).size();
    if(s > max) max = s;
  }
  return max;
}

int rftablemodel::max_row_name_length() const {
  int max = 0;
  for(int x=0;x<n_rows();x++) {
    int s = row_name(x).size();
    if(s > max) max = s;
  }
  return max;
}

int rftablemodel::max_col_name_length() const {
  int max = 0;
  for(int x=0;x<n_columns();x++) {
    int s = col_name(x).size();
    if(s > max) max = s;
  }
  return max;
}

double rftablemodel::min_value() const {
  return 0;
}

double rftablemodel::max_value() const {
  return 0;
}

QStringList rftablemodel::get_views() const {
  return QStringList();
}

void rftablemodel::set_view(const QString &view) {

}

QString rftablemodel::current_view() const {
  return QString();
}

void rftablemodel::clear_selection() {
  selection.clear();
  selection.resize(0);
  change_selection();
}

bool rftablemodel::is_selected(const int &row, const int &col) const {
  QPoint p(col,row);
  for(int x=0;x<selection.size();x++) {
    if(selection.at(x).contains(row,col)) return true;
  }
  return false;
}

bool rftablemodel::has_selection() const {
  return ! selection.isEmpty();
}

table_region rftablemodel::last_selection() const {
  if(has_selection() == false) return table_region();
  return selection.last();
}

QVector<table_cell> rftablemodel::selected_cells() const {
  QVector<table_cell>out;
  for(int x=0;x<selection.size();x++) {
    table_region rect = selection.at(x);
    QVector<table_cell>cells = rect.cells();
    for(int x=0;x<cells.size();x++) {
      if(out.contains(cells.at(x)) == false) out.append(cells.at(x));
    }
  }
  return out;
}

QVector<table_region> rftablemodel::selected_regions() const {
  return selection;
}

bool rftablemodel::all_selected() const {
  for(int row=0;row<n_rows();row++) {
    for(int col=0;col<n_columns();col++) {
      if(is_selected(row,col) == false) return false;
    }
  }
  return true;
}

QPoint rftablemodel::selection_origin() const {
  if(selection.isEmpty()) return QPoint(0,0);
  QPoint p(n_columns(),n_rows());
  for(int x=0;x<selection.size();x++) {
    int col = selection.at(x).col;
    int row = selection.at(x).row;
    if(col < p.x() || row < p.y()) {
      p.setX(selection.at(x).col);
      p.setY(selection.at(x).row);
    }
  }
  return p;
}

QString rftablemodel::to_ascii() const {
  QString out;

  QVector <int>col_print_width; // store print width of columns
  for(int x=0;x<n_columns();x++) col_print_width.append(0); // init table at zero
  int col_print_spacing = 1; // default column spacing ...

  bool print_header = true;

  // need to determine max column sizes dynamically.... might be a bit slow
  for(int row=0;row<n_rows();row++) {
    for(int col=0;col<n_columns();col++) {
      int csz = string_at(row,col).size();
      if(col_print_width.at(col) < csz) col_print_width[col] = csz;
    }
  }

  // if printing header need to check that too ...
  if(print_header == true) {
    for(int x=0;x<n_columns();x++) {
      if(col_print_width.at(x) < col_name(x).size()) col_print_width[x] = col_name(x).size();
    }
  }

  // .. print the header
  if(print_header == true) {
    for(int x=0;x<n_columns();x++) {
      out.append(col_name(x));
      if(x < n_columns() - 1) {
        int cs = col_name(x).size();
        for(int z=cs;z<col_print_width.at(x) + col_print_spacing;z++) out.append(' ');
      } else {
        out.append('\n');
      }
    }
  }

  // print dashed lines under table header
  if(print_header == true) {
    for(int x=0;x<n_columns();x++) {
      for(int z=0;z<col_print_width.at(x);z++) out.append('-'); // print dashes
      if(x < n_columns() - 1) {
        for(int z=0;z<col_print_spacing;z++) out.append(' '); // pad if not at end
      } else {
        out.append('\n'); // linefeed at end
      }
    }
  }

  // .. print the table itself
  for(int row=0;row<n_rows();row++) {
    for(int col=0;col<n_columns();col++) {
      out.append(string_at(row,col));
      if(col < n_columns() - 1) {
        // not sure if this is faster than leftJustified of qstring or not .
        int cs = string_at(row,col).size(); // current field size
        for(int z=cs;z<col_print_width.at(col) + col_print_spacing;z++) out.append(' '); // pad to end
      } else {
        out.append('\n');
      }
    }
  }
  return out;
}

int rftablemodel::selected_cell_count() const {
  int sum = 0;
  for(int x=0;x<selection.size();x++) {
    sum += selection.at(x).count();
  }
  return sum;
}

void rftablemodel::select_row(const int &row, bool clear) {
  if(clear) selection.clear();
  table_region r(row,0,1,n_columns());
  selection.append(r);
  change_selection();
}

void rftablemodel::select_col(const int &col, bool clear) {
  if(clear) selection.clear();
  table_region r(0,col,n_rows(),1);
  selection.append(r);
  change_selection();
}

void rftablemodel::select_single(const int &row, const int &col, bool clear) {
  if(clear) selection.clear();
  table_region r(row,col);
  selection.append(r);
  change_selection();
}

void rftablemodel::select_span(const table_region &r, bool clear) {
  if(clear) selection.clear();
  selection.append(r);
  change_selection();
}

void rftablemodel::select_all() {
  select_span(table_region(0,0,n_rows(),n_columns()),true);
  change_selection();
}

QString rftablemodel::selection_to_clipboard_text() const {
  QStringList out;
  QVector <table_region>selection = selected_regions();
  if(selection.isEmpty()) { // cp entire table if no selection.
    selection.append(table_region(0,0,n_rows(),n_columns()));
  }
  while(selection.isEmpty() == false) {
    table_region r = selection.takeFirst();
    for(int row=0;row<r.height;row++) {
      QString l;
      for(int col=0;col<r.width;col++) {
        l.append(QString::number(double_at(r.row + row, r.col + col)));
        if(col < r.width - 1) l.append('\t');
      }
      out.append(l);
    }
  }
  return out.join('\n');
}

QString rftablemodel::table_to_romraider() const {
  QStringList out;
  if(is_3d() == true) {
    out.append("[Table3D]");
  } else {
    out.append("[Table1D]");
  }
  out.append(column_string_list().join('\t'));
  for(int row=0;row<n_rows();row++) {
    QString l;
    l.append(row_name(row) + '\t');
    for(int col=0;col<n_columns();col++) {
      l.append(QString::number(double_at(row,col)));
      if(col < n_columns() - 1) l.append('\t');
    }
    out.append(l);
  }
  return out.join('\n');
}

double rftablemodel::lookup_3d_neighbour(const double &x, const double &y) const {
  if(n_columns() < 2 || n_rows() < 2) return 0.00;
  if(table_type() != TABLE3D) return 0.00;
  int r = row_index(y);
  if(r > 0) {
    if(y_axis_value(r) - y > y - y_axis_value(r-1)) r--;
  }
  int c = col_index(x);
  if(c > 0) {
    if(x_axis_value(c) - x > x - x_axis_value(c-1)) c--;
  }
  return double_at(r,c);
}

double rftablemodel::lowest_value() const {
  double lowest = __DBL_MAX__;
  for(int x=0;x<n_columns();x++) {
    for(int y=0;y<n_rows();x++) {
      if(is_null(y,x) == true) continue;
      double d = double_at(y,x);
      if(d < lowest) d = lowest;
    }
  }
  return lowest;
}

double rftablemodel::highest_value() const {
  double highest = __DBL_MIN__;
  for(int x=0;x<n_columns();x++) {
    for(int y=0;y<n_rows();x++) {
      if(is_null(y,x) == true) continue;
      double d = double_at(y,x);
      if(d > highest) d = highest;
    }
  }
  return highest;
}

double rftablemodel::lookup(const rflookupconfig &config, double x, double y) const {
  if(n_columns() < 2 || n_rows() < 2) return NAN;

  // edge filter for null can be done as a prefilter
  if(config.tail_type == rflookupconfig::TAIL_NULL) {
    if(x < x_axis_value(0)) return NAN;
    if(x > x_axis_value(n_columns() - 1)) return NAN;
    if(y > y_axis_value(n_rows() - 1)) return NAN;
    if(y < x_axis_value(0)) return NAN;
  }

  switch(config.lookup_type) {
  case rflookupconfig::LOOKUP_NEIGHBOUR:
    return lookup_3d_neighbour(x,y);
    break;
  case rflookupconfig::LOOKUP_LINEAR:
    return lookup_3d_linear(x,y,config.tail_type == rflookupconfig::TAIL_FLAT);
    break;

  }

  return NAN;
}

double rftablemodel::lookup_3d_linear(const double &x, const double &y, bool edge_filter) const {
  if(table_type() != TABLE3D) return 0.00;

  // determine column indexes
  int c2 = col_index(x);
  int c1;
  if(c2 == 0) {
      c2++;
      c1 = 0;
  } else {
    c1 = c2 - 1;
  }

  // determine row indexes
  int r2 = row_index(y);
  int r1;
  if(r2 == 0) {
    r2++;
    r1 = 0;
  } else {
    r1 = r2 - 1;
  }

  // determine layout and data
  double x1 = x_axis_value(c1);
  double x2 = x_axis_value(c2);
  double y1 = y_axis_value(r1);
  double y2 = y_axis_value(r2);
  double q11 = double_at(r1,c1);
  double q21 = double_at(r1,c2);
  double q12 = double_at(r2,c1);
  double q22 = double_at(r2,c2);

  // do not look up on nan values.
  if(q11 != q11 || q21 != q21 || q12 != q12 || q22 != q22) return NAN;

  /* x1    x    x2
     y1  q11    q21
     y    out
     y2  q12    q22
  */

  // edge filter routine creates a flat plane off the edge.
  if(edge_filter) {
    // left edge
    if(c1 == 0 && x < x_axis_value(0)) {
      x2 = x1;
      x1 = x;
      q21 = q11;
    }
    // right edge
    if(c2 == n_columns() - 1 && x > x_axis_value(n_columns() - 1)) {
      x1 = x2;
      x2 = x;
      q11 = q21;
    }
    // top edge
    if(r1 == 0 && y < y_axis_value(0)) {
      y2 = y1;
      y1 = y;
      q12 = q11;
    }
    // bottom edge
    if(r2 == n_rows() - 1 && y > y_axis_value(n_rows() - 1)) {
      y1 = y2;
      y2 = y;
      q21 = q22;
    }
  }

  if(x2 == x1 || y2 == y1) return 0.00;

  // grifted from https://helloacm.com/cc-function-to-compute-the-bilinear-interpolation/
  double x2x1, y2y1, x2x, y2y, yy1, xx1;
  x2x1 = x2 - x1;
  y2y1 = y2 - y1;
  x2x = x2 - x;
  y2y = y2 - y;
  yy1 = y - y1;
  xx1 = x - x1;
  return 1.0 / (x2x1 * y2y1) * (
        q11 * x2x * y2y +
        q21 * xx1 * y2y +
        q12 * x2x * yy1 +
        q22 * xx1 * yy1
        );
}

double rftablemodel::interp_linear(double x0, double x1, double y0, double y1, double x) {
  return y0+((x-x0)/(x1-x0))*(y1-y0);
}
