#include "module_table.h"
#include "ui_module_table.h"
#include "../dialogs/table_config_dialog.h"
#include "../dialogs/interpolation_dialog.h"
#include "muparser/include/muParser.h"
#include "dialogs/database_dialog.h"
#include <QClipboard>
#include <QDoubleValidator>
#include <QInputDialog>
#include <QDebug>

module_table::module_table() :
  module(nullptr),
  ui(new Ui::module_table) {
  ui->setupUi(this);
  tbl = new rftable();  

  ui->renderer->set_data(tbl);

  connect(this,&module::name_changed,ui->rename_btn,&QPushButton::setText);

  set_icon("3x3_grid_icon&16.png");
  set_name("Untitled Table");

  // configure line edit and cycle so the buttons are disabled.
  ui->lineEdit->clear();
  ui->undo_btn->setEnabled(false);
  ui->redo_btn->setEnabled(false);
  ui->calc_err->setHidden(true);

  connect(ui->renderer,&rftablewidget::cell_doubleclicked,this,&module_table::edit_item);
}

module_table::module_table(QVector<rfdatalog *> logs) : module_table() {
  this->logs = logs;
}

module_table::module_table(rftablemodel *data) : module_table() {
  if(data->table_type() == rftablemodel::DATALOG) return; // cannot make a table from a datalog.
  tbl->copy_data(*data);
}

module_table::~module_table() {
  while(undo_cache.isEmpty() == false) delete undo_cache.takeFirst();
  while(redo_cache.isEmpty() == false) delete redo_cache.takeFirst();
  delete ui;
  delete tbl;
}

void module_table::edit_item(int row, int col) {
  ui->lineEdit->setText(QString::number(get_data()->double_at(row,col)));
  ui->lineEdit->setFocus();
  ui->lineEdit->selectAll();
}

rftablemodel *module_table::get_data() {
  return tbl;
}

void module_table::set_table(rftable t) {
  tbl->copy_data(t);
  ui->renderer->set_data(tbl);
}

rftable module_table::get_table() const {
  rftable out;
  out.copy_data(*tbl);
  return out;
}

void module_table::on_layout_btn_clicked() {
  table_config_dialog d(logs);
  d.configure(tbl->x_axis(),tbl->y_axis());
  if(d.exec() == table_config_dialog::Accepted) {
    if(tbl->x_axis() == d.get_xaxis() && tbl->y_axis() == d.get_yaxis()) return; // bail if no real change.

    rftable::_layout_handler handler = rftable::CLEAR_ALL;
    table_interpolation iconfig = table_interpolation();

    // if we have data we should offer interpolation or deletion.
    if(tbl->contains_data()) {
      interpolation_dialog d;
      if(tbl->data_complete() == false) d.disable_interpolation();
      if(tbl->is_2d() == true) d.disable_interpolation(); // TEMP -- FIX ME WHEN 2d INTERP DONE
      if(d.exec() == interpolation_dialog::Rejected) return; // bail
      handler = d.selection;
      iconfig = d.config;
    }

    store();

    tbl->set_layout(d.get_xaxis(),d.get_yaxis(),handler,iconfig);

    if(custom_name.isEmpty()) set_name("Untitled (" + QString::number(tbl->n_columns()) + "x" + QString::number(tbl->n_rows()) + ")");
  }
}

void module_table::on_lineEdit_textChanged(const QString &arg1) {

}

void module_table::on_clr_btn_clicked() {
  //double value = entered_value();
  QVector <table_cell>cells = tbl->selected_cells();
  while(cells.isEmpty() == false) {
    table_cell c = cells.takeFirst();
    tbl->set_null(c.row,c.col);
  }
}

void module_table::on_lineEdit_returnPressed() {
   ui->set_btn->click();
}

void module_table::on_rename_btn_clicked() {
  QString s = QInputDialog::getText(nullptr,"Table name","Enter name for table.",QLineEdit::Normal,name());
  if(s.isEmpty() == false) set_name(s);
}

void module_table::on_set_btn_clicked() {
  QString equation = ui->lineEdit->text();

  mu::Parser parser;
  double n = 0;
  double x = 0;
  double y = 0;
  parser.DefineVar(QString("N").toStdWString(),&n);
  parser.DefineVar(QString("X").toStdWString(),&x);
  parser.DefineVar(QString("Y").toStdWString(),&y);
  parser.SetExpr(QString(equation).toStdWString());

  try {
    double d = parser.Eval();
  }

  catch (mu::Parser::exception_type &e) {
    ui->calc_err->setHidden(false);
    ui->calc_err->setText(QString::fromStdWString(e.GetMsg()));
    return;
  }

  ui->calc_err->setHidden(true);

  QVector <table_cell>cells = tbl->selected_cells();

  if(cells.isEmpty()) return;

  store(); // store for undo

  while(cells.isEmpty() == false) {
    table_cell c = cells.takeFirst();
    n = tbl->double_at(c.row,c.col);
    x = tbl->x_axis_value(c.col);
    y = tbl->y_axis_value(c.row);
    tbl->set(parser.Eval(),c.row,c.col);
  }
}

void module_table::store() {
  rftable *t = new rftable();
  t->copy_data(*tbl);
  undo_cache.append(t);
  ui->undo_btn->setEnabled(true);
  while(redo_cache.isEmpty() == false) delete redo_cache.takeFirst(); // drop redo cache on new storage.

}

module *module_table::clone() const {
  module_table *out = new module_table(tbl); // this copies the data.
  out->ui->renderer->import_data(ui->renderer->export_data()); // copy rendererer settings
  out->set_name(name() + " Copy");
  return out;
}

void module_table::on_undo_btn_clicked() {
  if(undo_cache.isEmpty()) return;
  rftable *t = undo_cache.takeLast();
  rftable *rt = new rftable();
  rt->copy_data(*tbl);
  redo_cache.append(rt);
  tbl->copy_data(*t);
  delete t;
  ui->redo_btn->setEnabled(true);
  ui->undo_btn->setEnabled(! undo_cache.isEmpty());
}

void module_table::on_redo_btn_clicked() {
  if(redo_cache.isEmpty()) return;
  rftable *rt = new rftable();
  rt->copy_data(*tbl);
  undo_cache.append(rt);
  rftable *t = redo_cache.takeLast();
  tbl->copy_data(*t);
  delete t;
  ui->undo_btn->setEnabled(true);
  ui->redo_btn->setEnabled(! redo_cache.isEmpty());
}

void module_table::on_db_btn_clicked() {
  dbdata db_out = tbl->export_data();
  db_out.name = name();
  db_out.set_subdata("RENDER",ui->renderer->export_data());
  database_dialog dialog("STOREDTABLE",db_out);
  if(dialog.exec() == database_dialog::Accepted) {
    dbdata db = dialog.selected_item;
    if(tbl->import_data(db) == false) {
      error("Could not import table.");
      return;
    } else {
      ui->renderer->import_data(db.subdata("RENDER"));
      set_name(db.name);
    }
  }
}
