#include "table_geometry.h"
#include <QStringList>
#include <QLocale>
#include "../widgets/axis_configurator_widget.h"

int table_axis::locate_index(const double &value, _locate_alignment align) const {
  // FIXME if we want more speed for this sorted list search we could do a btree search or something.

  switch(align) {
  case table_axis::ALIGN_LH:
  {
    int out = values.size()  - 1;
    while(out > 0 && values.at(out) >= value) out--;
    return out;
  }
    break;
  case table_axis::ALIGH_RH:
  {
    for(int x=0;x<values.size();x++) {
      if(values.at(x) > value) return x;
    }
    return values.size() - 1;
  }
    break;
  case table_axis::ALIGN_NN:
  {
    if(values.size() <= 1) return 0; // array size of at least 2.
    int l = locate_index(value,ALIGN_LH); // find left value
    if(l == values.size() - 1) return values.size() - 1;  // if left value is at end, neighbour is last element for sure.
    double median = (values.at(l) + values.at(l+1)) / 2;
    if(value < median) return l;
    return l + 1;
  }
    break;

  }

  return 0;
}

inline bool table_axis::value_in_range(const double &value) const {
  if(values.size() == 0) return false;
  if(value < values.first()) return false;
  if(value > values.last()) return false;
  return true;
}

double table_axis::at(int i) const {
  if(i < 0) return 0;
  if(i > count() - 1) return at(count() - 1);
  return values.at(i);
}

int table_axis::count() const {
  return values.count();
}

void table_axis::copy(const table_axis &a) {
  this->values = a.values;
  this->_precision = a._precision;
  this->name = a.name;
}

bool table_axis::operator==(const table_axis &other) {
  if(values.size() != other.values.size()) return false;
  if(precision() != other.precision()) return false;
  for(int x=0;x<values.size();x++) {
    if(values.at(x) != other.values.at(x)) return false;
  }
  return true;
}

bool table_axis::is_valid() const {
  if(values.isEmpty()) return false;
  if(_precision == -1) return false;
  if(is_sequential() == false) return false;
  return true;
}

void table_axis::clear() {
  values.clear();
  name.clear();
  _precision = -1;
}

bool table_axis::is_sequential() const {
  if(values.isEmpty()) return false;
  for(int x=1;x<values.size();x++) {
    if(values.at(x - 1) >= values.at(x)) return false;
  }
  return true;
}

void table_axis::calc_precision() {
  QStringList s;
  for(int x=0;x<values.size();x++) {
    s.append(QString::number(values.at(x),'f',QLocale::FloatingPointShortest));
  }
  _precision = axis_configurator_widget::calc_precision(s);
}

int table_axis::precision() const {
  return _precision;
}

double table_axis::interval() const {
  if(values.size() <= 1) return 0;
  double sum = 0;
  for(int x=1;x<values.size();x++) {
    sum += (values.at(x) - values.at(x - 1));
  }
  return sum / count();
}

QString table_axis::encode_values() const {
  // FORMAT:   *NAME|PRECISION|VALUE|VALUE|VALUE
  QString out;
  QString s = name;
  s.replace('|',' '); // filter out lines.
  s.prepend('*');
  out.append(s + "|");
  out.append(QString::number(precision()) + "|");
  for(int x=0;x<values.size();x++) {
    out.append(QString::number(values.at(x)));
    if(x < values.size() - 1) out.append('|');
  }
  return out;
}

bool table_axis::decode_values(const QString &s) {
  // FORMAT:   *NAME|PRECISION|VALUE|VALUE|VALUE
  values.clear();
  QStringList in = s.split('|');
  if(in.size() < 2) { // require at least a name and a precision.
    clear();
    return true;
  }
  QString name = in.takeFirst();
  if(name.startsWith('*') == false) return false;
  name.remove(0,1); // remove asterisk.
  this->name = name;
  _precision = in.takeFirst().toInt();
  while(in.isEmpty() == false) {
    bool ok;
    values.append(in.takeFirst().toDouble(&ok));
    if(ok == false) {
      values.clear();
      return false;
    }
  }
  return true;
}

bool table_region::contains(const int &row, const int &col) const {
  if(row < this->row) return false;
  if(col < this->col) return false;
  if(row > this->row + this->height - 1) return false;
  if(col > this->col + this->width - 1) return false;
  return true;
}

QVector<table_cell> table_region::cells() const {
  QVector <table_cell>out;
  for(int x=0;x<width;x++) {
    for(int y=0;y<height;y++) {
      out.append(table_cell(y + this->row,x + this->col));
    }
  }
  return out;
}

QString table_geometry::size_string() const {
  if(is_2d()) {
    return "(" + QString::number(y.count()) + ")";
  } else {
    return "(" + QString::number(x.count()) + "x" + QString::number(y.count()) + ")";
  }
}

void table_geometry::clear() {
  x.clear();
  y.clear();
  dbname.clear();
}

dbdata table_geometry::export_data() const {
  dbdata out;
  if(is_valid() == false) return out;
  out.name = dbname;
  out.set("IS2D",is_2d());
  out.set("YAXIS",y.encode_values());
  out.set("YAXISNAME",y.name);
  if(is_2d() == false) {
    out.set("XAXIS",x.encode_values());
    out.set("XAXISNAME",x.name);
  }
  return out;
}

bool table_geometry::import_data(const dbdata &data) {
  dbname = data.name;
  bool is_2d = data.value("IS2D").toBool();
  y.decode_values(data.value("YAXIS").toString());
  y.name = data.value("YAXISNAME").toString();
  if(is_2d == false) {
    x.decode_values(data.value("XAXIS").toString());
    x.name = data.value("XAXISNAME").toString();
  }
  return true;
}
