#include "dynamic_column.h"
#include "rfdatalog.h"
#include <QVector>
#include <QList>

bool dynamic_column::is_compatible(rfdatalog *log) const {
  if(equation_errors.isEmpty() == false) {
    return false;
  }
  if(equation.isEmpty() && variables.isEmpty()) return false;
  // ensure all data elements exist for this log.
  for(int x=0;x<variables.size();x++) {
    if(variables.at(x).is_compatible(log) == false) return false;
  }
  return true;
}

bool dynamic_column::uses_math() const {
  if(equation.isEmpty()) return false;
  return true;
}

void dynamic_column::set_equation(QString equation) {
  this->equation = equation;
  revision++;
  reload_calculator();
}

void dynamic_column::add_col_variable(const QString &col_name) {
  dynamic_col_variable d;
  d.col_type = dynamic_col_variable::COL_DIRECT;
  d.sources.clear();
  d.sources.append(col_name);
  variables.append(d);
  var_storage.append(0.00);
  revision++;
}

void dynamic_column::edit_col_variable(int index, const QString &col_name) {
  if(index < 0 || index > variable_count() - 1) return;
  dynamic_col_variable d;
  d.col_type = dynamic_col_variable::COL_DIRECT;
  d.sources.clear();
  d.sources.append(col_name);
  variables[index] = d;
  revision++;
}

void dynamic_column::add_lookup_variable(const rftable &table, const rflookupconfig &config) {
  dynamic_col_variable d;
  d.col_type = dynamic_col_variable::COL_LOOKUP;
  d.sources.clear();
  d.sources.append(table.x_axis_name());
  d.sources.append(table.y_axis_name());
  d.lookup.copy_data(table);
  d.lookup_config = config;
  variables.append(d);
  var_storage.append(0.00);
  revision++;
}

void dynamic_column::edit_lookup_variable(int index, const rftable &table, const rflookupconfig &config) {
  if(index < 0 || index > variable_count() - 1) return;
  dynamic_col_variable d;
  d.col_type = dynamic_col_variable::COL_LOOKUP;
  d.sources.clear();
  d.sources.append(table.x_axis_name());
  d.sources.append(table.y_axis_name());
  d.lookup.copy_data(table);
  d.lookup_config = config;
  variables[index] = d;
  revision++;
}

void dynamic_column::delete_variable(int index) {
  variables.removeAt(index);
  var_storage.removeAt(index);
  revision++;
}

dynamic_col_variable dynamic_column::variable_at(int x) const {
  return variables.at(x);
}

int dynamic_column::variable_count() const {
  return variables.size();
}

QString dynamic_column::var_name(int index) {
  static QString symbols = QStringLiteral("ABCDEFGHIJKLMNOPQRSTUVWabcdefghijklmnopqrstuvw");
  QString s;
  if(index < 0 || index > symbols.size() - 1) return QString();
  s.append(symbols.at(index));
  return s;
}

bool dynamic_column::reload_calculator() {
  equation_errors.clear();

  // config variables
  calc.ClearVar();
  var_storage.clear();
  var_storage.resize(variables.size());
  for(int x=0;x<variables.size();x++) {
    calc.DefineVar(var_name(x).toStdWString(),&var_storage[x]);
  }

  // set expr and trial run
  calc.SetExpr(equation.toStdWString());
  try
  {
    double d = calc.Eval();
  }
  catch (mu::Parser::exception_type &e)
  {
    equation_errors = QString::fromStdWString(e.GetMsg());
    return false;
  }
  return true;
}

double dynamic_column::value(rfdatalog *log, const int &row) const {
  for(int x=0;x<variables.size();x++) {
    var_storage[x] = variables.at(x).value(log,row);
  }
  if(uses_math() == false) { // add everything if no equation present
    double out = 0.00;
    for(int x=0;x<var_storage.size();x++) out += var_storage.at(x);
    return out;
  } else {
    return calc.Eval();
  }
}

dynamic_col_variable::dynamic_col_variable(const dynamic_col_variable &other) {
  sources = other.sources;
  col_type = other.col_type;
  lookup.copy_data(other.lookup);
  last_log = other.last_log;
  src_cache = other.src_cache;
  lookup_config = other.lookup_config;
}

dynamic_col_variable dynamic_col_variable::operator=(const dynamic_col_variable &other) {
  dynamic_col_variable out;
  out.sources = other.sources;
  out.col_type = other.col_type;
  out.lookup.copy_data(other.lookup);
  out.last_log = other.last_log;
  out.src_cache = other.src_cache;
  out.lookup_config = other.lookup_config;
  return out;
}

bool dynamic_col_variable::is_compatible(rfdatalog *log) const {
  for(int x=0;x<sources.size();x++) {
    if(log->index_of(sources.at(x)) == -1) return false;
  }
  return true;
}

double dynamic_col_variable::value(rfdatalog *log, const int &row) const {
  switch(col_type) {
  case dynamic_col_variable::COL_DIRECT:
    if(log != last_log) { // populate cache
      src_cache.clear();
      src_cache.append(log->index_of(sources.at(0)));
    }
    return log->double_at(row,src_cache.at(0));
    break;
  case dynamic_col_variable::COL_LOOKUP:
    if(log != last_log) { // populate cache
      src_cache.clear();
      src_cache.append(log->index_of(sources.at(0)));
      src_cache.append(log->index_of(sources.at(1)));
    }
    return lookup.lookup(lookup_config,log->double_at(row,src_cache.at(0)),log->double_at(row,src_cache.at(1)));
    break;
  }
  return true;
}

QString dynamic_col_variable::display_name() const {
  switch(col_type) {
  case dynamic_col_variable::COL_DIRECT:
    if(sources.size() != 1) return "ERR";
    return sources.at(0);
    break;
  case dynamic_col_variable::COL_LOOKUP:
    if(sources.size() != 2) return "ERR";
    return QString::number(lookup.n_columns()) + "x" +
        QString::number(lookup.n_rows()) + " vs " +
        sources.at(0) + " vs " + sources.at(1);
    break;
  }
  return QString();
}

dynamic_col_variable::~dynamic_col_variable() {

}

dbdata dynamic_col_variable::export_data() const {
  dbdata db_out;
  db_out.name = display_name();
  if(col_type == COL_DIRECT) {
    db_out.set("TYPE","D");
  } else {
    db_out.set("TYPE","L");
    db_out.set_subdata("TBL",lookup.export_data());
    db_out.set_subdata("LKP",lookup_config.encode());
  }
  db_out.set("SOURCES",sources);
  return db_out;
}

bool dynamic_col_variable::import_data(const dbdata &data) {
  QString t = data.value("TYPE").toString();
  if(t == "D") {
    col_type = COL_DIRECT;
  } else if(t == "L") {
    col_type = COL_LOOKUP;
    lookup.import_data(data.subdata("TBL"));
    lookup_config.decode(data.subdata("LKP"));
  } else return false;
  sources = data.value("SOURCES").toStringList();
  return true;
}

dbdata dynamic_column::export_data() const {
  dbdata db_out;
  db_out.set("NAME",name);
  db_out.set("N_VARS",variables.size());
  for(int x=0;x<variables.size();x++) {
    db_out.set_subdata("@VAR" + QString::number(x),variables.at(x).export_data());
  }
  db_out.set("EQUATION",equation);
  return db_out;
}

bool dynamic_column::import_data(const dbdata &data) {
  name = data.value("NAME").toString();
  variables.clear();
  for(int x=0;x<data.value("N_VARS").toInt();x++) {
    dynamic_col_variable v;
    if(v.import_data(data.subdata("@VAR" + QString::number(x)))) variables.append(v);
  }
  equation = data.value("EQUATION").toString();
  reload_calculator();
  return true;
}

dynamic_column *dynamic_column_list::ptr(int i) {
  if(i < 0 || i > count() - 1) {
    return nullptr;
  }
  return &data()[i];
}

dbdata dynamic_column_list::export_data() const {
  dbdata db_out;
  db_out.name = "DYNCOL";
  db_out.set("N",size());
  for(int x=0;x<size();x++) {
    db_out.set_subdata("@COL" + QString::number(x),at(x).export_data());
  }
  return db_out;
}

bool dynamic_column_list::import_data(const dbdata &data) {
  clear();
  for(int x=0;x<data.value("N").toInt();x++) {
    dynamic_column c;
    if(c.import_data(data.subdata("@COL" + QString::number(x)))) append(c);
  }
  return true;
}

