﻿#include <QPaintEvent>
#include <cmath>

#include <QFontMetrics>
#include <QPainter>
#include <QDebug>
#include <QFontDatabase>
#include <QApplication>
#include <QClipboard>
#include <QMenu>

#include <QElapsedTimer>

#include "rftablewidget.h"
#include "axis_configurator_widget.h"
#include "../datastructures/table_geometry.h"
#include "../dialogs/color_scale_dialog.h"
#include "config.h"

#define FONT_SIZE_SMALL 7
#define FONT_SIZE_REG 9
#define FONT_SIZE_LG 11

rftablewidget::rftablewidget(QWidget *parent) :
  QWidget(parent),hscrollbar(Qt::Horizontal,this),vscrollbar(Qt::Vertical,this) {

  margin = 3;

  QFont f = QFontDatabase::systemFont(QFontDatabase::FixedFont);
  f.setPointSize(f.pointSize());
  change_font(f);

  setFocusPolicy(Qt::StrongFocus);

  color_background = palette().window().color();
  color_cell_dark = color_background.darker(125);
  color_text = palette().text().color();
  color_cell_light = color_cell_dark.lighter(110);

  color_cell_selected = color_cell_light;
  color_cell_selected.setBlue(255);

  this->scale.set_fallback_color(color_cell_light);

  connect(&vscrollbar,&QScrollBar::valueChanged,this,&rftablewidget::vscrollbar_changed);
  connect(&hscrollbar,&QScrollBar::valueChanged,this,&rftablewidget::hscrollbar_changed);

  connect(&redraw_delay,&QTimer::timeout,this,&rftablewidget::_repaint);
  redraw_delay.setSingleShot(true);
  redraw_delay.setInterval(15); // slight redraw delay for UI events in between to happen.

  configure_actions();
}

void rftablewidget::configure_actions() {
  action_select_all = new QAction(QIcon(":/graphics/cc_16/3x3_grid_2_icon&16.png"),"Select All",this);
  action_select_all->setShortcut(QKeySequence::SelectAll);
  connect(action_select_all,&QAction::triggered,this,&rftablewidget::select_all);
  addAction(action_select_all);

  action_select_none = new QAction(QIcon(),"Select None",this);
  //action_copy->setShortcut(QKeySequence::Selec);
  connect(action_select_none,&QAction::triggered,this,&rftablewidget::select_none);
  addAction(action_select_none);

  action_copy = new QAction(QIcon(":/graphics/cc_16/clipboard_copy_icon&16.png"),"Copy",this);
  action_copy->setShortcut(QKeySequence::Open);
  connect(action_copy,&QAction::triggered,this,&rftablewidget::copy_to_clipboard);
  addAction(action_copy);

  action_cut = new QAction(QIcon(":/graphics/cc_16/clipboard_cut_icon&16.png"),"Cut",this);
  action_cut->setShortcut(QKeySequence::Cut);
  connect(action_cut,&QAction::triggered,this,&rftablewidget::cut_to_clipboard);
  addAction(action_cut);

  action_paste = new QAction(QIcon(":/graphics/cc_16/clipboard_past_icon&16.png"),"Paste",this);
  action_paste->setShortcut(QKeySequence::Paste);
  connect(action_paste,&QAction::triggered,this,&rftablewidget::paste_from_clipboard);
  addAction(action_paste);

  action_color = new QAction(QIcon(":/graphics/cc_16/brush_icon&16.png"),"Color Scale",this);
  //action_copy->setShortcut(QKeySequence::Paste);
  connect(action_color,&QAction::triggered,this,&rftablewidget::popup_color_scale_dialog);
  addAction(action_color);

#ifdef ROMRAIDER_TABLE_HACKS
  action_copy_table = new QAction(QIcon(":/graphics/cc_16/clipboard_copy_icon&16.png"),"Copy Table",this);
  connect(action_copy_table,&QAction::triggered,this,&rftablewidget::copy_table);
  addAction(action_copy_table);
#endif

  precision_actions.resize(6);
  precision_actions[0] = new QAction(QIcon(),"0",this);
  precision_actions[1] = new QAction(QIcon(),"0.0",this);
  precision_actions[2] = new QAction(QIcon(),"0.00",this);
  precision_actions[3] = new QAction(QIcon(),"0.000",this);
  precision_actions[4] = new QAction(QIcon(),"0.0000",this);
  precision_actions[5] = new QAction(QIcon(),"0.00000",this);
  connect(precision_actions[0],&QAction::triggered,this,&rftablewidget::set_precision0);
  connect(precision_actions[1],&QAction::triggered,this,&rftablewidget::set_precision1);
  connect(precision_actions[2],&QAction::triggered,this,&rftablewidget::set_precision2);
  connect(precision_actions[3],&QAction::triggered,this,&rftablewidget::set_precision3);
  connect(precision_actions[4],&QAction::triggered,this,&rftablewidget::set_precision4);
  connect(precision_actions[5],&QAction::triggered,this,&rftablewidget::set_precision5);
  for(int x=0;x<precision_actions.size();x++) {
    precision_actions.at(x)->setCheckable(true);
    addAction(precision_actions.at(x));
  }

  action_font_sm = new QAction(QIcon(),"Small",this);
  action_font_reg = new QAction(QIcon(),"Regular",this);
  action_font_lg = new QAction(QIcon(),"Large",this);
  action_font_sm->setCheckable(true);
  action_font_reg->setCheckable(true);
  action_font_lg->setCheckable(true);

  connect(action_font_sm,&QAction::triggered,this,&rftablewidget::set_font_sm);
  connect(action_font_reg,&QAction::triggered,this,&rftablewidget::set_font_reg);
  connect(action_font_lg,&QAction::triggered,this,&rftablewidget::set_font_lg);
  addAction(action_font_sm);
  addAction(action_font_reg);
  addAction(action_font_lg);

}

void rftablewidget::contextMenuEvent(QContextMenuEvent *event) {
  if(data == nullptr) {
    event->ignore();
    return;
  }
  QMenu *menu = create_menu();
  menu->popup(event->globalPos());
  event->accept();
}

QMenu *rftablewidget::create_menu() {
  QMenu *menu = new QMenu(this);
  menu->addAction(action_copy);
  menu->addAction(action_cut);
  menu->addAction(action_paste);
  menu->addAction(action_select_all);
  menu->addAction(action_select_none);
  menu->addAction(action_copy_table);
  menu->addSeparator();

  QMenu *precision_menu = menu->addMenu("Precision");
  for(int x=0;x<precision_actions.size();x++) {
    if(precision == x) {
      precision_actions.at(x)->setChecked(true);
    } else {
      precision_actions.at(x)->setChecked(false);
    }
    precision_menu->addAction(precision_actions.at(x));
  }

  QMenu *font_menu = menu->addMenu("Text Size");
  font_menu->addAction(action_font_sm);
  font_menu->addAction(action_font_reg);
  font_menu->addAction(action_font_lg);
  int fs = get_font_size();
  if(fs == FONT_SIZE_SMALL) {
    action_font_sm->setChecked(true);
    action_font_reg->setChecked(false);
    action_font_lg->setChecked(false);
  } else if(fs == FONT_SIZE_REG) {
    action_font_sm->setChecked(false);
    action_font_reg->setChecked(true);
    action_font_lg->setChecked(false);
  } else if(fs == FONT_SIZE_LG) {
    action_font_sm->setChecked(false);
    action_font_reg->setChecked(false);
    action_font_lg->setChecked(true);
  }

  menu->addSeparator();
  menu->addAction(action_color);
  menu->setAttribute(Qt::WA_DeleteOnClose);
  return menu;
}

void rftablewidget::set_precision0() {
  set_precision(0);
}

void rftablewidget::set_precision1() {
  set_precision(1);
}

void rftablewidget::set_precision2() {
  set_precision(2);
}

void rftablewidget::set_precision3() {
  set_precision(3);
}

void rftablewidget::set_precision4() {
  set_precision(4);
}

void rftablewidget::set_precision5() {
  set_precision(5);
}

void rftablewidget::set_font_sm() {
  set_font_size(FONT_SIZE_SMALL);
}

void rftablewidget::set_font_reg() {
  set_font_size(FONT_SIZE_REG);
}

void rftablewidget::set_font_lg() {
  set_font_size(FONT_SIZE_LG);
}

void rftablewidget::popup_color_scale_dialog() {
  if(data == nullptr) return;
  color_scale_dialog d(get_color_scale(),data->value_list());
  if(d.exec() == color_scale_dialog::Accepted) set_color_scale(true,d.scale);
}

void rftablewidget::set_color_scale(bool enable, color_scale scale) {
  if(enable) {
    this->scale = scale;
    this->scale.set_fallback_color(color_cell_light);
  } else {
    this->scale.clear();
  }
  delayed_redraw();
}

void rftablewidget::select_all() {
  if(data_valid() == false) return;
  data->select_all();
}

void rftablewidget::select_none() {
  if(data_valid() == false) return;
  data->clear_selection();
}

rftablewidget::~rftablewidget() {

}

void rftablewidget::wheelEvent(QWheelEvent *event) {
  QApplication::sendEvent(&vscrollbar,event);
}

int rftablewidget::row_at_point(const QPoint &p) const {
  for(int x = display_row;x<display_row + n_display_rows() + 1;x++) {
    if(row_rect(x - display_row + 1).contains(p) && x < data->n_rows()) {
      return x;
    }
  }
  return -1;
}

int rftablewidget::col_at_point(const QPoint &p) const {
  for(int x = col_offset;x<data->n_columns();x++) {
    if(col_rect(x).contains(p)) return x;
  }
  return -1;
}

int rftablewidget::col_header_at_point(const QPoint &p) const {
  if(row_rect(0).contains(p) == false) return -1;
  return col_at_point(p);
}

int rftablewidget::row_header_at_point(const QPoint &p) const {
  if(row_label_col_rect.contains(p) == false) return -1;
  return row_at_point(p);
}

bool rftablewidget::data_valid() const {
  if(data == nullptr) return false;
  if(data->valid() == false) return false;
  if(data->n_columns() == 0) return false;
  return true;
}

void rftablewidget::mouseDoubleClickEvent(QMouseEvent *event) {
  if(data_valid() == false) return;
  QPoint p = event->pos();
  int r = row_at_point(p);
  if(r == -1) {
    data->clear_selection();
    event->ignore();
    return;
  }

  int c = col_at_point(p);
  if(c == -1) {
    data->clear_selection();
    event->ignore();
    return;
  }

  data->select_single(r,c,true);
  emit cell_doubleclicked(r,c);
  event->accept();
}

void rftablewidget::keyPressEvent(QKeyEvent *event) {
  if(data == nullptr || data->valid() == false) {
    event->ignore();
  } else if(event->matches(QKeySequence::Copy)) {
    selection_to_clipboard();
    event->accept();
  } else if(event->matches(QKeySequence::SelectAll)) {
    select_all();
    event->accept();
  } else if(event->matches(QKeySequence::Cut) && data->readonly() == false) {
    selection_to_clipboard(true);
    event->accept();
  } else if(event->matches(QKeySequence::Paste) && data->readonly() == false) {
    paste_from_clipboard();
    event->accept();
  } else {
    event->ignore();
  }
}

void rftablewidget::selection_to_clipboard(bool cut) {
  QApplication::clipboard()->setText(data->selection_to_clipboard_text());

  //clear selection
  if(cut && data->readonly() == false) {
    QVector <table_cell>cells = data->selected_cells();
    while(cells.isEmpty() == false) {
      table_cell c = cells.takeFirst();
      data->set_null(c.row,c.col);
    }
  }
}

bool rftablewidget::paste_from_clipboard() {
  if(data == nullptr) return false;
  if(data->readonly()) return false;
  QStringList lines = QApplication::clipboard()->text().split('\n',Qt::SkipEmptyParts);

#ifdef ROMRAIDER_TABLE_HACKS
  // hacks for romraider tables
  if(lines.startsWith("[Selection3D]")) lines.takeFirst();
  if(lines.startsWith("[Selection1D]")) lines.takeFirst();
  bool whole_table_mode = false;
  if(lines.startsWith("[Table3D]")) {
    whole_table_mode = true;
    lines.takeFirst();
  }
#endif

  QVector<QStringList>rows;
  for(int x=0;x<lines.size();x++) rows.append(lines.at(x).split('\t'));
  int n_cols = -1;

  QPoint offset = data->selection_origin();
  int offset_x = offset.x();
  int offset_y = offset.y();

#ifdef ROMRAIDER_TABLE_HACKS
  if(whole_table_mode) {
    offset_x = 0;
    offset_y = 0;
    // trim x/y header data.
    if(rows.isEmpty() == false) rows.takeFirst(); // get rid of column header
    for(int x=0;x<rows.size();x++) {
      if(rows[x].isEmpty() == false) rows[x].takeFirst();
    }
  }
#endif

  // ensure all columns are the same size
  for(int r=0;r<rows.size();r++) {
    if(n_cols == -1) {
      n_cols = rows.at(r).size();
    } else if(rows.at(r).size() != n_cols) {
      return false;
    }
  }

  // set data
  for(int r=0;r<rows.size();r++) {
    for(int c=0;c<n_cols;c++) {
      bool ok;
      double in = rows.at(r).at(c).toDouble(&ok);
      if(ok == true) {
        data->set(in,offset_y + r,offset_x + c);
      } else {
        data->set_null(offset_y + r,offset_x + c);
      }
    }
  }
  return true;
}

void rftablewidget::copy_to_clipboard() {
  if(data == nullptr) return;
  if(data->selected_cells().count() == 0) return;
  QClipboard *clipboard = QApplication::clipboard();
  clipboard->setText(data->selection_to_clipboard_text());
}

void rftablewidget::copy_table() {
  if(data == nullptr) return;
  QClipboard *clipboard = QApplication::clipboard();
  clipboard->setText(data->table_to_romraider());
}

void rftablewidget::cut_to_clipboard() {
  if(data == nullptr) return;
  if(data->readonly() == true) return;
  if(data->selected_cells().count() == 0) return;
  QClipboard *clipboard = QApplication::clipboard();
  clipboard->setText(data->selection_to_clipboard_text());
  QVector <table_cell>cells = data->selected_cells();
  for(int x=0;x<cells.size();x++) {
    data->set_null(cells.at(x).row,cells.at(x).col);
  }
}

void rftablewidget::mousePressEvent(QMouseEvent *event) {
  if(data_valid() == false) return;
  QPoint p = event->pos();
  if(event->button() == Qt::LeftButton) {

    if(btn_rect.contains(p)) {
      QMenu *menu = create_menu();
      menu->popup(event->globalPosition().toPoint());
      event->accept();
      return;
    }

    int r = row_at_point(p);
    if(r == -1) {
      data->clear_selection();
      event->ignore();
      return;
    }

    int c = col_at_point(p);
    if(c == -1) {
      data->clear_selection();
      event->ignore();
      return;
    }

    if(event->modifiers().testFlag(Qt::ControlModifier) == true) {
      data->select_single(r,c,false);
    } else if(event->modifiers().testFlag(Qt::ShiftModifier) == true) {
      if(data->has_selection() == false) {
        data->select_single(r,c,true);
      } else {
        table_region old_sel = data->last_selection();
        table_region new_sel;

        if(c > old_sel.col) { // our clicked column is to the right of the old one
          new_sel.col = old_sel.col;
          new_sel.width = 1 + c - old_sel.col;
        } else { // our clicked column is to the left.
          new_sel.col = c;
          new_sel.width = 1 + old_sel.col - c;
        }

        if(r > old_sel.row) {
          new_sel.row = old_sel.row;
          new_sel.height = 1 + r - old_sel.row;
        } else {
          new_sel.row = r;
          new_sel.height = 1 + old_sel.row - r;
        }

        data->select_span(new_sel,true);
      }
    } else {
      data->select_single(r,c,true);
    }

    delayed_redraw();

    event->accept();
  }
  event->ignore();
}

void rftablewidget::delayed_redraw() {
  if(redraw_delay.isActive() == false) redraw_delay.start();
}

void rftablewidget::_repaint() {
  repaint(rect());
}

void rftablewidget::vscrollbar_changed(int value) {
  if(data_valid() == false) return;
  if(value < 0) value = 0;
  if(value > data->n_rows() - 1) value = data->n_rows() - 1;
  display_row = value;
  delayed_redraw();
}

void rftablewidget::hscrollbar_changed(int value) {
  if(data_valid() == false) return;
  if(value < 0) value = 0;
  if(value > data->n_columns() - 1) value = data->n_columns() - 1;
  col_offset = value;
  delayed_redraw();
}

void rftablewidget::set_font_size(int size) {
  QFont f = this->font();
  f.setPointSize(size);
  change_font(f);
}

void rftablewidget::set_precision(int precision) {
  this->precision = precision;
  data_changed();
}

int rftablewidget::get_precision() const {
  return this->precision;
}

int rftablewidget::get_font_size() const {
  return this->font().pointSize();
}

void rftablewidget::set_data(rftablemodel *data) {
  if(this->data != nullptr) {
    disconnect(data,&rftablemodel::data_changed,this,&rftablewidget::data_changed);
    disconnect(data,&rftablemodel::geometry_changed,this,&rftablewidget::geometry_changed);
    disconnect(data,&rftablemodel::selection_changed,this,&rftablewidget::selection_changed);
  }
  if(data == this->data) return;
  this->data = data;
  if(data->table_type() == rftablemodel::DATALOG) {
    data_numerical = false;
  } else {
    data_numerical = true;
  }

  geometry_changed(); // trigger full rebuild and redraw of table

  connect(data,&rftablemodel::data_changed,this,&rftablewidget::data_changed);
  connect(data,&rftablemodel::geometry_changed,this,&rftablewidget::geometry_changed);
  connect(data,&rftablemodel::selection_changed,this,&rftablewidget::selection_changed);

  action_cut->setEnabled(!data->readonly());
  action_paste->setEnabled(!data->readonly());
  action_color->setEnabled(data->table_type() != rftablemodel::DATALOG);
}

void rftablewidget::data_changed() {
  if(data == nullptr) return;
  // if data changed -- see if it affects geometry due to column size changes.
  if(data->auto_col_size()) {
    int c = calc_equal_col_size();
    if(c != col_size_cache) {
      col_size_cache = c;
      update_table_geometry();
    }
  }
  delayed_redraw();
}

void rftablewidget::geometry_changed() {
  col_size_cache = -1;
  update_widget_geometry(); // necessary right now just because of top and bottom areas.  fix me later.
  update_table_geometry();
  delayed_redraw();
}

void rftablewidget::selection_changed() {
  delayed_redraw();
}

void rftablewidget::update_widget_geometry() {
  if(data_valid() == false) return;

  QFontMetrics metrics(this->font());

  top_area = rect();
  if(data->x_axis_name().isEmpty()) {
    top_area.setHeight(0);
  } else {
    top_area.setHeight(metrics.height() + margin + margin);
  }

  bottom_area = rect();
  bottom_area.setTop(height() - 20);
  hscrollbar.setGeometry(bottom_area);

  // left area is as wide as a char except when there is no axis label.
  left_area = rect();
  left_area.setLeft(0);
  left_area.setTop(top_area.bottom());
  left_area.setBottom(bottom_area.top());
  if(data->y_axis_name().isEmpty()) {
    left_area.setWidth(0);
  } else {
    left_area.setWidth(metrics.height() + margin + margin);
  }

  right_area.setLeft(width() - 20);
  right_area.setRight(width());
  right_area.setTop(top_area.bottom());
  right_area.setBottom(rect().bottom());

  bottom_area.setRight(right_area.left());

  vscrollbar.setGeometry(right_area);

  // determine viewport
  table_area.setTop(top_area.bottom());
  table_area.setLeft(left_area.right());
  table_area.setRight(right_area.left());
  table_area.setBottom(bottom_area.top());
}

void rftablewidget::update_table_geometry() {
  if(data_valid() == false) {
    hscrollbar.setHidden(true);
    vscrollbar.setHidden(true);
    return;
  } else {
    hscrollbar.setHidden(false);
    vscrollbar.setHidden(false);
  }

  QFontMetrics metrics(this->font());

  cell_height = metrics.height() + margin + margin;

  // determine left column (row range) size.
  int row_label_size = 0;
  for(int x=0;x<data->n_rows();x++) {
    int s = data->row_name(x).size();
    if(s > row_label_size) row_label_size = s;
  }
  row_label_col_rect = table_area;
  row_label_col_rect.setWidth(metrics.boundingRect(QString(row_label_size,'_')).width() + margin + margin);

  // determine column sizing
  int cols = data->n_columns();
  col_size_map.resize(cols);

  QRect cursor = table_area;
  cursor.moveLeft(row_label_col_rect.right() + space_between_cells); // initial position

  if(data->auto_col_size()) {
    if(col_size_cache == -1) col_size_cache = calc_equal_col_size();
    cursor.setWidth(metrics.boundingRect(QString(col_size_cache,'_')).width() + margin + margin);
    for(int x=0;x<cols;x++) {
      col_size_map[x] = cursor;
      cursor.moveLeft(cursor.right() + space_between_cells + space_between_cells);
    }
  } else {
    for(int x=0;x<cols;x++) {
      cursor.setWidth(metrics.boundingRect(QString(data->col_custom_size(x),'_')).width() + margin + margin);
      col_size_map[x] = cursor;
      cursor.moveLeft(cursor.right() + space_between_cells + space_between_cells);
    }
  }

  int h = table_area.height();
  int w = table_area.width();
  row_label_col_rect.setHeight(h);

  int total_width = row_label_col_rect.width();
  for(int x=0;x<cols;x++) {
    col_size_map[x].setHeight(h);
    total_width += col_size_map[x].width();
  }
  // stretch if undersized.
  if(total_width < w) {
    stretch_width = ( ( w - total_width ) / col_size_map.size() ) - 1;
    hscrollbar.setRange(0,0);

    // scroll if oversized.
  } else {
    // find highest cell that, if placed at the start, would allow the entire table to be drawn.
    int acc = 0;
    int x = data->n_columns() - 1;
    while(x > 0) {
      acc += col_size_map.at(x).width() + space_between_cells + space_between_cells;
      if(acc >= w) {
        break;
      }
      x--;
    }
    hscrollbar.setRange(0,x + 1);
    stretch_width = 0;
  }
  // update vertical scrollbar
  vscrollbar.setRange(0,data->n_rows() - n_display_rows() + 1);
}

void rftablewidget::draw_row(QPainter *p, const int &row) {
  QRect r = row_rect(row + 1);
  QRect lr = row_label_col_rect;
  lr.setTop(r.top());
  lr.setHeight(r.height());
  draw_cell_static(p,lr,data->row_name(display_row + row));
  if(r.intersects(table_area) == false) return;
  for(int col = col_offset;col < data->n_columns();col++) {
    QRect cr = col_rect(col);
    cr.setTop(r.top());
    cr.setBottom(r.bottom());
    draw_cell_data(p,cr,display_row + row,col);
  }
}

QString rftablewidget::get_data(const int &row, const int &col) const {
  if(data->is_null(row,col) == true) return QString();
  if(data_numerical == false) {
    return data->string_at(row,col);
  } else {
    return QString::number(data->double_at(row,col),'f',precision);
  }
}

void rftablewidget::draw_cell_static(QPainter *p, QRect r, const QString &static_data) {
  if(r.intersects(table_area) == false) return; // do not draw off screen.
  p->fillRect(r,color_cell_dark);
  r.adjust(margin,margin,0-margin,0-margin);
  p->drawText(r,Qt::AlignRight | Qt::AlignVCenter,static_data);
}

void rftablewidget::draw_cell_data(QPainter *p, QRect r, const int &row, const int &col) {
  if(r.intersects(table_area) == false) return; // do not draw off screen.

  bool data_null = data->is_null(row,col);

  QColor bgcolor;

  // retrieve number data once
  double number_data = 0.00;
  if(data_numerical) number_data = data->double_at(row,col);

  if(data->is_selected(row,col)) {
    bgcolor = color_cell_selected;
  } else if(data_null) {
    bgcolor = color_cell_light;
  } else if(data_numerical == false) {
    bgcolor = color_cell_light;
  } else {
    bgcolor = scale.value(number_data);
  }

  // draw bg
  p->fillRect(r,bgcolor);

  // draw data
  if(data_null == false) {
    r.adjust(margin,margin,0-margin,0-margin);
    if(data_numerical) {
      p->drawText(r,Qt::AlignRight | Qt::AlignVCenter,QString::number(number_data,'f',precision));
    } else {
      p->drawText(r,Qt::AlignLeft | Qt::AlignVCenter,data->string_at(row,col));
    }
  }

}

int rftablewidget::calc_equal_col_size() {
  int out = 0;
  for(int x=0;x<data->n_columns();x++) {
    int cs = data->col_name(x).size();
    if(cs > out) out = cs;
    for(int y=0;y<data->n_rows();y++) {
      int s = get_data(y,x).size();
      if(s > out) out = s;
    }
  }
  return out;
}

void rftablewidget::draw_h_header(QPainter *p) {
  for(int col = col_offset;col < data->n_columns();col++) {
    draw_cell_static(p,col,0,data->col_name(col),color_cell_dark);
  }
  btn_rect = row_label_col_rect;
  btn_rect.setHeight(cell_height);
  QIcon(":/graphics/cc_16/align_just_icon&16.png").paint(p,btn_rect);
}

void rftablewidget::draw_background(QPainter *p, const QRect &r) {
  p->fillRect(r,color_background);
}

void rftablewidget::draw_background_dark(QPainter *p, const QRect &r) {
  p->fillRect(r,color_background.darker(109));
}

void rftablewidget::draw_cell_static(QPainter *p, const int &col, const int &row, const QString &s, const QColor &bgcolor) {
  draw_cell_static(p,cell_rect(col,row),s,bgcolor);
}

void rftablewidget::draw_cell_static(QPainter *p, QRect r, const QString &s, const QColor &bgcolor) {
  if(r.intersects(table_area) == false) return; // do not draw off screen.
  p->fillRect(r,bgcolor);
  r.adjust(margin,margin,0-margin,0-margin);
  p->drawText(r,Qt::AlignRight | Qt::AlignVCenter,s);
}

QRect rftablewidget::cell_rect(const int &col, const int &row) const {
  QRect r = col_rect(col);
  QRect x = row_rect(row);
  r.setTop(x.top());
  r.setHeight(x.height());
  return r;
}

QRect rftablewidget::col_rect(const int &col) const {
  QRect r = col_size_map.at(col);
  if(stretch_width > 0) {
    r.moveLeft(r.left() + (stretch_width * col) + space_between_cells);
    r.setRight(r.right() + stretch_width);
  } else {
    int sum = 0;
    for(int x=0;x<col_offset;x++) {
      sum += col_size_map.at(x).width();
    }
    r.moveLeft(r.left() - sum - (space_between_cells * col_offset) + space_between_cells);
  }
  return r;
}

QRect rftablewidget::row_rect(const int &row) const {
  int height = cell_height;
  int prefix_height = table_area.top() + ( height * row ) + (space_between_cells * row);
  return QRect(table_area.left(),prefix_height,table_area.width(),height);
}

QRect rftablewidget::row_label_rect(const int &row) const {
  QRect r = row_label_col_rect;
  QRect x = row_rect(row + 1);
  r.setTop(x.top());
  r.setHeight(x.height());
  return r;
}

int rftablewidget::n_display_rows() const {
  double height = cell_height + space_between_cells;
  double r = ( table_area.height() - space_between_cells) / height;
  return floor(r) - 1;
}

void rftablewidget::change_font(QFont f) {
  setFont(f);
  update_widget_geometry();
  data_changed();
}

void rftablewidget::paintEvent(QPaintEvent *event) {
  QPainter p(this);
  qreal inverseDPR = 1.0 / devicePixelRatio();
  p.scale(inverseDPR, inverseDPR);

  if(data_valid() == false || data->n_columns() != col_size_map.size()) {
    QRect r = this->rect();
    p.fillRect(r,QColor("#cccccc"));
    p.drawText(r,Qt::AlignCenter,"Table not configured.\nPlease define a layout to continue.");
    return;
  }

  if(event->rect().intersects(left_area)) {
    draw_background(&p,left_area);

    // rotate text
    p.save();
    p.translate(rect().bottomLeft());
    p.rotate(-90);
    QRect r(rect());
    r.setWidth(rect().height());
    r.setHeight(left_area.width());
    p.drawText(r,Qt::AlignCenter,data->y_axis_name());
    p.restore();
  }


  if(event->rect().intersects(top_area)) {
    draw_background(&p,top_area);
    p.drawText(top_area,Qt::AlignCenter,data->x_axis_name());
  }

  if(event->rect().intersects(table_area)) {
    p.setClipRect(table_area);

    draw_background(&p,table_area);
    draw_h_header(&p); // non-moving

    for(int row=0;row <= n_display_rows();row++) {
      if(row + display_row > data->n_rows() - 1) break; // prevent falling off end of table.
      draw_row(&p,row);
    }

    p.setClipping(false);
  }
}

void rftablewidget::resizeEvent(QResizeEvent *event) {
  update_widget_geometry();
  update_table_geometry();
}


QColor color_scale::value(const double &in) const {
  if(points.isEmpty()) return fallback_color;
  if(points.size() == 1) return points.at(0).c;
  if(in <= points.at(0).p) return points.at(0).c;
  if(in >= points.at(points.size() - 1).p) return points.at(points.size() - 1).c;
  for(int x=0;x<points.size() - 1;x++) {
    if(in > points.at(x).p && in < points.at(x + 1).p) {
      if(in == points.at(x).p) return points.at(x).c; // exact value
      int low_index = x;
      int high_index = x+1;
      double r = rftablemodel::interp_linear(points.at(low_index).p,points.at(high_index).p,1,2,in);
      QColor out = blend(points.at(low_index).c,points.at(high_index).c,r-1);
      if(out.isValid() == false) return fallback_color;
      return out;
    }
  }
  return fallback_color;
}

QColor color_scale::blend(const QColor &scale_color1, const QColor &scale_color2, double r) {
  if(r<0) r = 0;
  if(r>1) r = 1;
  QColor out(scale_color1.red() * (1-r) + scale_color2.red() * r,
      scale_color1.green() * (1-r) + scale_color2.green() * r,
      scale_color1.blue() * (1-r) + scale_color2.blue() *r);
  return out;
}

dbdata rftablewidget::export_data() const {
  dbdata d;
  d.set_subdata("COLORSCALE",scale.export_data());
  d.set("PRECISION",precision);
  d.set("FONT_SIZE",get_font_size());
  return d;
}

bool rftablewidget::import_data(const dbdata &data) {
  scale.import_data(data.subdata("COLORSCALE"));
  set_precision(data.value("PRECISION",2).toInt());
  set_font_size(data.value("FONT_SIZE",9).toInt());
  return true;
}

dbdata color_scale::export_data() const {
  dbdata d;
  QVariantList l;
  for(int x=0;x<points.size();x++) {
    l.append(points.at(x).c);
    l.append(points.at(x).p);
  }
  d.set("POINTS",l);
  return d;
}

bool color_scale::import_data(const dbdata &data) {
  clear();
  QVariantList l = data.value("POINTS").toList();
  for(int x=0;x<l.size();x += 2) {
    if(x+1 > l.size() - 1) {
      clear();
      return false;
    }
    point p;
    p.c = l.at(x).value<QColor>();;
    p.p = l.at(x+1).toDouble();
    points.append(p);
  }
  return true;
}

void color_scale::estimate(double min_value, double max_value) {
  clear();
  add_point(min_value,QColor(240,117,117));
  add_point((min_value + max_value) / 2,QColor(222,218,135));
  add_point(max_value,QColor(141,220,137));
}

void color_scale::clear() {
  points.clear();
}

void color_scale::add_point(const point &p) {
  if(points.isEmpty()) {
    points.append(p);
    return;
  }
  if(points.at(points.size() - 1).p < p.p) {
    points.append(p);
    return;
  }
  if(points.at(0).p > p.p) {
    points.prepend(p);
    return;
  }
  for(int x=1;x<points.size();x++) {
    if(points.at(x).p == p.p) {
      points[x].c = p.c;
      return;
    }
    if(points.at(x).p > p.p) {
      points.insert(x - 1,p);
      return;
    }
  }
}

void color_scale::add_point(double point, QColor color) {
  struct point p;
  p.c = color;
  p.p = point;
  add_point(p);
}

void color_scale::set_points(QVector<color_scale::point> points) {
  clear();
  for(int x=0;x<points.size();x++) add_point(points.at(x));
}

QVector<color_scale::point> color_scale::get_points() const {
  return points;
}
