#ifndef AFILTER_H
#define AFILTER_H

#include <QString>
#include "rfdatalog.h"
#include <QDebug>

//T1_COL, T2_STATIC, T2_VALUE, T2_COL, OPERATOR

class dbdata;

class afilter {
public:
  QString term1_colname;
  int term1_colindex = -1;

  bool term2_static = true;

  double term2_value = 0;
  QString term2_colname;
  int term2_colindex = -1;

  bool refactor(rfdatalog *log) {
    term1_colindex = log->index_of(term1_colname);
    if(term1_colindex == -1) return false;
    if(term2_static == false) {
      term2_colindex = log->index_of(term2_colname);
      if(term2_colindex == -1) return false;
    }
    return true;
  }

  dbdata export_data() const {
    dbdata out;
    out.set("T1_COL",term1_colname);
    out.set("T2_STATIC",term2_static);
    out.set("T2_VALUE",term2_value);
    out.set("T2_COL",term2_colname);
    out.set("OPERATOR",operator_symbol(operation));
    return out;
  };

  bool import_data(const dbdata &data) {
    term1_colname = data.value("T1_COL").toString();
    term2_static = data.value("T2_STATIC").toBool();
    term2_value = data.value("T2_VALUE").toDouble();
    term2_colname = data.value("T2_COL").toString();
    operation = symbol_map.value(data.value("OPERATOR").toString());
    return true;
  };

  bool is_valid() const {
    if(term1_colindex == -1) return false;
    if(term2_static == false && term2_colindex == -1) return false;
    return true;
  }

  enum operator_e {
    INVALID = -1,
    LESSTHAN = 0,
    LESSTHANOREQUAL = 1,
    GREATERTHAN = 2,
    GREATERTHANOREQUAL = 3,
    EQUAL = 4,
    NOTEQUAL = 5,
    MAX_DELTA = 6,
    MIN_DELTA = 7,
    DECREASING = 8,
    N_OPERATORS = 9
  };

  static QString operator_symbol(const operator_e &e) {
    switch(e) {
    case afilter::INVALID:
    case afilter::N_OPERATORS:
      return QStringLiteral("ERR");
      break;
    case afilter::LESSTHAN:
      return QStringLiteral("<");
      break;
    case afilter::LESSTHANOREQUAL:
      return QStringLiteral("<=");
      break;
    case afilter::GREATERTHAN:
      return QStringLiteral(">");
      break;
    case afilter::GREATERTHANOREQUAL:
      return QStringLiteral(">=");
      break;
    case afilter::EQUAL:
      return QStringLiteral("=");
      break;
    case afilter::NOTEQUAL:
      return QStringLiteral("!=");
      break;
    case afilter::MAX_DELTA:
      return QStringLiteral("MAX CHANGE");
      break;
    case afilter::MIN_DELTA:
      return QStringLiteral("MIN CHANGE");
      break;
    case afilter::DECREASING:
      return QStringLiteral("DECREASING");
      break;
    }
    return QString();
  }

  static QMap<QString,operator_e>symbol_map;
  static QMap<QString,operator_e>build_symbol_map() {
    QMap<QString,operator_e> out;
    for(int x=INVALID;x<N_OPERATORS;x++) {
      out[operator_symbol((operator_e)x)] = (operator_e)x;
    }
    return out;
  };

  operator_e operation = INVALID;

  bool matches(rftablemodel *d, const int &r) const {
    // return true if condition matches.
    double t1 = d->double_at(r,term1_colindex);
    double t2;
    if(term2_static) {
      t2 = term2_value;
    } else {
      t2 = d->double_at(r,term2_colindex);
    }
    switch(operation) {
    case afilter::INVALID:
    case afilter::N_OPERATORS:
      return false;
      break;
    case afilter::LESSTHAN:
      return t1 < t2;
      break;
    case afilter::LESSTHANOREQUAL:
      return t1 <= t2;
      break;
    case afilter::GREATERTHAN:
      return t1 > t2;
      break;
    case afilter::GREATERTHANOREQUAL:
      return t1 >= t2;
      break;
    case afilter::EQUAL:
      return t1 == t2;
      break;
    case afilter::NOTEQUAL:
      return t1 != t2;
      break;
    case afilter::MAX_DELTA:
      if(r < 1) return false;
    {
      double previous_value = d->double_at(r - 1,term1_colindex);
      if(previous_value < t1) { // value decreasing

      }
      if(t1 - previous_value >= t2) return false;
      return true;
    }
      break;
    case afilter::MIN_DELTA:
      if(r < 1) return false;
    {
      double previous_value = d->double_at(r - 1,term1_colindex);
      if(t1 - previous_value <= t2) return false;
      return true;
    }
      break;

    case afilter::DECREASING:
      if(r < 1) return false;
    {
      double previous_value = d->double_at(r - 1,term1_colindex);
      if(t1 < previous_value) return true;
      return false;
    }
      break;
    }
    return false;
  };
};



#endif // AFILTER_H
