#include "database.h"
#include "datastructures/rftablemodel.h"
#include "datastructures/afilter.h"
#include <QFile>
#include <QApplication>
#include <QSqlError>
#include <QErrorMessage>
#include <QDataStream>
#include <QDebug>

#define DB_VERSION 21

QMap<QString,afilter::operator_e> afilter::symbol_map = afilter::build_symbol_map();

QSqlDatabase database::db;
int database::profile_id = -1;
QString database::_profile_name = QString();

database::database() {

}

bool database::init() {
  if(db.isOpen() == true) return true; // prevent redundant config
  db = QSqlDatabase::addDatabase("QSQLITE");
  db.setDatabaseName(qApp->applicationDirPath() + "/" + "database.sqlite");
  if(db.open() == false) {
    err("Database Error!\n" +  db.lastError().text());
    return false;
  }
  configure_db();
  change_profile(get_config("LAST_PROFILE_ID",-1).toInt());
  return true;
}

void database::err(const QSqlQuery &q) {
  err(q.lastQuery() + "\n" + q.lastError().text());
}

QString database::ascii_query(const QString &query) {
  QSqlQuery q(db);
 if(q.exec(query) == false) {
   return q.lastError().text();
 } else {
   return ascii_table(q);
 }
}

void database::err(const QString &err_msg) {
  QErrorMessage m;
  m.showMessage(err_msg);
  m.exec();
}

bool database::create_std_table(const QString &table_name) {
  QSqlQuery q(db);
  if(q.exec("CREATE TABLE IF NOT EXISTS " + table_name + "( ID INTEGER PRIMARY KEY, NAME TEXT, "
            "PROFILE INT REFERENCES PROFILE(ID) ON DELETE CASCADE, "
            "VALUE TEXT )") == false) {
    err(q);
    return false;
  }
  return true;
}

bool database::create_db(bool upgrade) {
  Q_UNUSED(upgrade)
  db.transaction();
  QSqlQuery q(db);
  if(q.exec("CREATE TABLE IF NOT EXISTS CONFIG (KEY TEXT PRIMARY KEY, VALUE BLOB)") == false) {
    err(q);
    db.rollback();
    return false;
   }
  if(q.exec("CREATE TABLE IF NOT EXISTS PROFILE (ID INTEGER PRIMARY KEY, NAME TEXT)") == false) {
    err(q);
    db.rollback();
    return false;
  }
  if(create_std_table("GEOMETRY") == false) {
    db.rollback();
    return false;
  }
  if(create_std_table("ANALYSIS") == false) {
    db.rollback();
    return false;
  }
  if(create_std_table("GRAPH2D") == false) {
    db.rollback();
    return false;
  }
  if(create_std_table("STOREDTABLE") == false) {
    db.rollback();
    return false;
  }
  if(create_std_table("COLORSCALE") == false) {
    db.rollback();
    return false;
  }

  set_config("VERSION",DB_VERSION);
  if(profile_id == -1) {
    dbitem p;
    p.name = "DEFAULT";
    p.id = -1;
    set_config("LAST_PROFILE_ID",save_profile(p));
  }
  db.commit();
  return true;
}

bool database::upgrade_db(int old_version) {
  if(old_version < 19) {
    QSqlQuery q(db);
    q.prepare("DROP TABLE ANALYSIS");
    q.exec();
    q.prepare("DROP TABLE GEOMETRY");
    q.exec();
    q.prepare("DROP TABLE GRAPH2D");
    q.exec();
  }
  create_db(true);
  set_config("VERSION",DB_VERSION);
  return true;
}

bool database::change_profile(int profile_id) {
  QVector <dbitem> profiles = profile_list();
  QString name;
  for(int x=0;x<profiles.size();x++) { // slow search because lazy
    if(profiles.at(x).id == profile_id) {
      name = profiles.at(x).name;
      break;
    }
  }
  if(name.isEmpty()) {
    profile_id = profiles.at(0).id;
    name = profiles.at(0).name;
  }
  database::profile_id = profile_id;
  database::_profile_name = name;
  set_config("LAST_PROFILE_ID",profile_id);
  return true;
}

QString database::profile_name() {
  return _profile_name;
}

bool database::set_config(const QString &key, const QVariant &value) {
  QByteArray out;
  QDataStream s(&out,QIODevice::ReadWrite);
  s << value;
  QSqlQuery q(db);
  q.prepare("REPLACE INTO CONFIG (KEY,VALUE) VALUES(?,?)");
  q.addBindValue(key);
  q.addBindValue(out);
  if(q.exec() == false) {
    err(q);
    return false;
  }
  return true;
}

QVariant database::get_config(const QString &key, const QVariant &default_value) {
  db.transaction();
  QSqlQuery q(db);
  q.prepare("SELECT VALUE FROM CONFIG WHERE KEY = ?");
  q.addBindValue(key);
  if(q.exec() == false) {
    err(q);
    return QVariant();
  }
  if(q.next() == false) {
    if(set_config(key,default_value) == false) {
      db.rollback();
      return QVariant();
     } else {
      db.commit();
      return default_value;
    }
  }
  QByteArray in = q.value(0).toByteArray();
  QDataStream s(&in,QIODevice::ReadWrite);
  QVariant v;
  s >> v;
  db.commit();
  return v;
}

bool database::configure_db() {
  // see if we're configured, by existance of config table.
  bool configured = true;
  {
    QSqlQuery q(db);
    q.prepare("SELECT COUNT(*) FROM CONFIG");
    if(q.exec() == false) {
      configured = false;
    }
  }

  // create db if not configured yet.
  if(configured == false && create_db() == false) return false;

  int version = get_config("VERSION",-1).toInt();
  if(version < DB_VERSION) {
    upgrade_db(version);
  }

  return true;
}

QString database::ascii_table(QSqlQuery &q) {
  QString out;

  QVector <QStringList>query_data;
  QStringList query_header;

  for(int x=0;x<q.record().count();x++) {
    query_header.append(q.record().fieldName(x));
  }

  while(q.next() == true) {
    QStringList s;
    for(int x=0;x<query_header.size();x++) {
      s.append(q.value(x).toString());
    }
    query_data.append(s);
  }
  if(query_data.isEmpty()) {
    return "Empty result.";
  }

  QVector <int>col_print_width; // store print width of columns
  for(int x=0;x<query_data.at(0).size();x++) col_print_width.append(0); // init table at zero
  int col_print_spacing = 1; // default column spacing ...

  int n_columns = query_data[0].size();
  int n_rows = query_data.size();

  bool print_header = true;

  // need to determine max column sizes dynamically.... might be a bit slow
  for(int row=0;row<n_rows;row++) {
    for(int col=0;col<n_columns;col++) {
      int csz = query_data.at(row).at(col).size();
      if(col_print_width.at(col) < csz) col_print_width[col] = csz;
    }
  }

  // if printing header need to check that too ...
  if(print_header == true) {
    for(int x=0;x<n_columns;x++) {
      if(col_print_width.at(x) < query_header.at(x).size()) col_print_width[x] = query_header.at(x).size();
    }
  }

  // .. print the header
  if(print_header == true) {
    for(int x=0;x<n_columns;x++) {
      out.append(query_header.at(x));
      if(x < n_columns - 1) {
        int cs = query_header.at(x).size();
        for(int z=cs;z<col_print_width.at(x) + col_print_spacing;z++) out.append(' ');
      } else {
        out.append('\n');
      }
    }
  }

  // print dashed lines under table header
  if(print_header == true) {
    for(int x=0;x<n_columns;x++) {
      for(int z=0;z<col_print_width.at(x);z++) out.append('-'); // print dashes
      if(x < n_columns - 1) {
        for(int z=0;z<col_print_spacing;z++) out.append(' '); // pad if not at end
      } else {
        out.append('\n'); // linefeed at end
      }
    }
  }

  // .. print the table itself
  for(int row=0;row<n_rows;row++) {
    for(int col=0;col<n_columns;col++) {
      out.append(query_data.at(row).at(col));
      if(col < n_columns - 1) {
        // not sure if this is faster than leftJustified of qstring or not .
        int cs = query_data.at(row).at(col).size(); // current field size
        for(int z=cs;z<col_print_width.at(col) + col_print_spacing;z++) out.append(' '); // pad to end
      } else {
        out.append('\n');
      }
    }
  }

  return out;
}

QVector<dbitem> database::profile_list() {
  QVector<dbitem>out;
  QSqlQuery q(db);
  q.prepare("SELECT ID,NAME FROM PROFILE ORDER BY NAME");
  if(q.exec() == false) {
    err(q);
    return out;
  }
  while(q.next() == true) {
    dbitem p;
    p.id = q.value(0).toInt();
    p.name = q.value(1).toString();
    out.append(p);
  }
  return out;
}

QVector<dbitem> database::list(const QString &table_name) {
  QVector<dbitem>out;
  QSqlQuery q(db);
  q.prepare("SELECT ID,NAME FROM " + table_name + " ORDER BY NAME");
  if(q.exec() == false) {
    err(q);
    return out;
  }
  while(q.next() == true) {
    dbitem p;
    p.id = q.value(0).toInt();
    p.name = q.value(1).toString();
    out.append(p);
  }
  return out;
}

QVector <dbdata>database::all_data(const QString &table_name) {
  QVector<dbdata>out;
  QSqlQuery q(db);
  q.prepare("SELECT ID,NAME,VALUE FROM " + table_name + " ORDER BY NAME");
  if(q.exec() == false) {
    err(q);
    return out;
  }
  while(q.next() == true) {
    dbdata p;
    p.id = q.value(0).toInt();
    p.name = q.value(1).toString();
    p.from_string(q.value(2).toString(),table_name);
    if(p.is_valid()) out.append(p);
  }
  return out;
}

dbdata database::data(const QString &table_name, const int &id) {
  QSqlQuery q(db);
  q.prepare("SELECT NAME,VALUE FROM " + table_name + " WHERE ID = ?");
  q.addBindValue(id);
  if(q.exec() == false) {
    err(q);
    return dbdata();
  }
  if(q.next() == false) return dbdata();
  dbdata out;
  out.name = q.value(0).toString();
  out.from_string(q.value(1).toString(),table_name);
  return out;
}

int database::insert(const QString &table_name, const dbdata &data) {
  QSqlQuery q(db);
  q.prepare("INSERT INTO " + table_name + " (NAME,VALUE,PROFILE) VALUES(?,?,?)");
  q.addBindValue(data.name);
  q.addBindValue(data.to_string(table_name));
  q.addBindValue(profile_id);
  if(q.exec() == false) {
    err(q);
    return -1;
  }
  return q.lastInsertId().toInt();
}

int database::replace(const QString &table_name, const dbdata &data) {
  QSqlQuery q(db);
  q.prepare("UPDATE " + table_name + " SET NAME = ?, VALUE = ?, PROFILE = ?  WHERE ID = ?");
  q.addBindValue(data.name);
  q.addBindValue(data.to_string(table_name));
  q.addBindValue(profile_id);
  q.addBindValue(data.id);
  if(q.exec() == false) {
    err(q);
    return -1;
  }
  if(q.numRowsAffected() != 1) return -1;
  return data.id;
}

int database::drop(const QString &table_name, const dbdata &data) {
  QSqlQuery q(db);
  q.prepare("DELETE FROM " + table_name + " WHERE ID = ?");
  q.addBindValue(data.id);
  if(q.exec() == false) {
    err(q);
    return -1;
  }
  if(q.numRowsAffected() != 1) return -1;
  return data.id;
}

int database::remove(const QString &table_name, int id) {
  QSqlQuery q(db);
  q.prepare("DELETE FROM " + table_name + " WHERE ID = ?");
  q.addBindValue(id);
  if(q.exec() == false) {
    err(q);
    return -1;
  }
  return id;
}

int database::save_profile(const dbitem &p) {
  if(p.id == -1) {
    QSqlQuery q(db);
    q.prepare("INSERT INTO PROFILE (NAME) VALUES(?)");
    q.addBindValue(p.name);
    if(q.exec() == false) {
      err(q);
      return -1;
    }
    return q.lastInsertId().toInt();
  } else {
    QSqlQuery q(db);
    q.prepare("UPDATE PROFILE SET NAME = ? WHERE ID = ?");
    q.addBindValue(p.name);
    q.addBindValue(p.id);
    if(q.exec() == false) {
      err(q);
      return -1;
    }
    return p.id;
  }
}

bool dbdata::is_valid() const {
  return valid;
}

bool dbdata::from_string(const QString &data_in, const QString &table_name) {
  valid = false;
  if(data_in.isEmpty()) return false;
  QByteArray b = qUncompress(QByteArray::fromBase64(data_in.toLatin1()));
  // verify checksum
  if(b.size() < 3) return false;
  quint8 checksum_a = 0x01;
  for(int x=0;x<b.size() - 2;x++) {
    checksum_a += b.at(x);
  }
  quint8 checksum_b = checksum_a;
  for(int x=0;x<b.size() - 2;x++) {
    checksum_b +=  ( b.at(x) - (quint8)x );
  }
  if((quint8)b.at(b.size() - 2) != checksum_a || (quint8)b.at(b.size() - 1) != checksum_b)  return false;
  b.chop(2); // drop checksum for parsing
  // parse data
  QDataStream s(b);
  data.clear();
  QString header;
  s >> header;
  if(header != table_name) return false;
  this->table_name = header;
  while(s.atEnd() == false) {
    QString key;
    s >> key;
    QVariant value;
    s >> value;
    data[key] = value;
  }
  valid = true;
  return true;
}

QString dbdata::to_string(const QString &table_name) const {
  QByteArray b;
  QDataStream s(&b,QDataStream::ReadWrite);
  QStringList keys = data.keys();
  QString header = table_name;
  s << header;

  while(keys.isEmpty() == false) {
    QString key = keys.takeFirst();
    QVariant value = data.value(key);
    s << key;
    s << value;
  }
  // generate checksum
  quint8 checksum_a = 0x01;
  for(int x=0;x<b.size();x++) {
    checksum_a += b.at(x);
  }
  quint8 checksum_b = checksum_a;
  for(int x=0;x<b.size();x++) {
    checksum_b +=  ( b.at(x) - (quint8)x );
  }
  b.append(checksum_a);
  b.append(checksum_b);
  return qCompress(b).toBase64();
}

QByteArray dbdata::to_sub() const {
  QByteArray b;
  QDataStream s(&b,QDataStream::ReadWrite);
  QStringList keys = data.keys();

  while(keys.isEmpty() == false) {
    QString key = keys.takeFirst();
    QVariant value = data.value(key);
    s << key;
    s << value;
  }

  return b;
}

bool dbdata::from_sub(const QByteArray &b) {
  // parse data
  QDataStream s(b);
  data.clear();
  while(s.atEnd() == false) {
    QString key;
    s >> key;
    QVariant value;
    s >> value;
    data[key] = value;
  }
  valid = true;
  return true;
}

QVariant dbdata::value(const QString &key, const QVariant &default_value) const {
  return data.value(key,default_value);
}

dbdata dbdata::subdata(const QString &key) const {
  dbdata out;
  if(data.contains(key) == false) return out;
  out.from_sub(data[key].toByteArray());
  return out;
}

void dbdata::clear() {
  valid = false;
  data.clear();
  id = -1;
  name.clear();
}

QStringList dbdata::parameter_list() const {
  QStringList out;
  QStringList keys = data.keys();
  while(keys.isEmpty() == false) {
    QString key = keys.takeFirst();
    out.append(key + "=" + data[key].toString());
  }
  return out;
}

void dbdata::set(const QString &key, const QVariant &value) {
  data[key] = value;
  valid = true;
}

void dbdata::set_subdata(const QString &key, const dbdata &value) {
  data[key] = value.to_sub();
  valid = true;
}
