#include "rfanalysis.h"
#include "rfdatalog.h"
#include "regression.h"
#include <cmath>
#include <QDataStream>
#include <QDebug>
#include <QElapsedTimer>

#include "muparser/include/muParser.h"

using namespace mu;

void rfanalysis::testing() {

}


rfanalysis::rfanalysis() {

}

dbdata rfanalysis::export_data() const {
  dbdata out;
  out.set("IS2D",is_2d());
  out.set("YAXIS",yaxis.encode_values());
  out.set("YAXISNAME",yaxis.name);
  if(is_2d() == false) {
    out.set("XAXIS",xaxis.encode_values());
    out.set("XAXISNAME",xaxis.name);
  }
  QVariantList filter_encodings;
  for(int x=0;x<filters.size();x++) {
    afilter f = filters.at(x);
    dbdata fd = f.export_data();
    filter_encodings.append(fd.to_sub());
  }
  out.set("FILTERS",filter_encodings);
  out.set("METHOD",method);
  out.set("DISPLAY",display);
  out.set("LAGFILTER",lagfilter);
  out.set("MINCOUNT",mincount);
  out.set("DATACOL",_data_col);
  out.set("FILTER_EDGES",filter_edges);
  out.set("EQUATION",equation);
  return out;
}

bool rfanalysis::import_data(const dbdata &data) {
  yaxis.decode_values(data.value("YAXIS").toString());
  yaxis.name = data.value("YAXISNAME").toString();
  bool is2d = data.value("IS2D").toBool();
  if(is2d == false) {
    xaxis.decode_values(data.value("XAXIS").toString());
    xaxis.name = data.value("XAXISNAME").toString();
  } else {
    xaxis.clear();
  }
  filters.clear();
  QVariantList filter_encodings = data.value("FILTERS").toList();
  for(int x=0;x<filter_encodings.size();x++) {
    afilter a;
    dbdata d;
    d.from_sub(filter_encodings.at(x).toByteArray());
    a.import_data(d);
    filters.append(a);
  }
  method = (_analysis_method)data.value("METHOD").toInt();
  display = (display_t)data.value("DISPLAY").toInt();
  lagfilter = data.value("LAGFILTER").toDouble();
  mincount = data.value("MINCOUNT").toInt();
  _data_col = data.value("DATACOL").toString();
  filter_edges = data.value("FILTER_EDGES").toBool();
  equation = data.value("EQUATION").toString();
  clear_data(); // this resizes the data structures and triggers a table geometry update.
  return true;
}

rfanalysis::rfanalysis(const rfanalysis &other) : rftablemodel() {
  xaxis = other.xaxis;
  yaxis = other.yaxis;
  _data_col = other.data_col();
  filters = other.filters;
  clear_data();
}

bool rfanalysis::set_layout(const table_axis &xaxis, const table_axis &yaxis) {
  this->xaxis = xaxis;
  this->yaxis = yaxis;
  clear_data();
  return true;
}

bool rfanalysis::set_data_col(const QString &data_col) {
  _data_col = data_col;
  return true;
}

bool rfanalysis::set_filters(const QVector<afilter> &filters) {
  this->filters = filters;
  return true;
}

QVector<afilter> rfanalysis::get_filters() const {
  return filters;
}

rftablemodel::_table_type rfanalysis::table_type() const {
  //if(is_valid() == false) return NOTABLE;
  if(xaxis.values.isEmpty()) return TABLE2D;
  return TABLE3D;
}

bool rfanalysis::is_valid() const {
  if((is_3d() == true && xaxis.is_valid() == false) || yaxis.is_valid() == false) return false;
  if(data_col().isEmpty()) return false;
  if(table.isEmpty()) return false;
  return true;
}

bool rfanalysis::set_equation(const QString &equation) {
  errors.clear();
  this->equation.clear();
  mu::Parser parser;
  double var = 1.00;
  parser.DefineVar(QString("N").toStdWString(),&var);
  parser.DefineVar(QString("X").toStdWString(),&var);
  parser.DefineVar(QString("Y").toStdWString(),&var);
  parser.SetExpr(QString(equation).toStdWString());
  try
  {
    parser.Eval();
  }
  catch (mu::Parser::exception_type &e)
  {
    errors.append(QString::fromStdWString(e.GetMsg()));
    return false;
  }
  this->equation = equation;
  recalculate();
  return true;
}

QString rfanalysis::get_equation() const {
  return equation;
}

int rfanalysis::n_columns() const {
  if(is_2d()) return 1;
  return xaxis.count();
}

int rfanalysis::n_rows() const {
  return yaxis.count();
}

bool rfanalysis::is_null(const int &row, const int &col) const {
  if(table.isEmpty()) return true;
  if(display == rfanalysis::DISPLAY_CALC && equation.isEmpty()) return true;
  if(display == rfanalysis::DISPLAY_COUNT && table[col][row].count == 0) return true;
  if(table[col][row].count <= mincount) return true;
  return false;
}

QString rfanalysis::col_name(const int &col) const {
  if(is_2d()) return QString();
  return QString::number(xaxis.at(col),'f',xaxis.precision());
}

QString rfanalysis::row_name(const int &row) const {
  return QString::number(yaxis.at(row),'f',yaxis.precision());
}

QString rfanalysis::string_at(const int &row, const int &col) const {
  return "ERR";
}

double rfanalysis::double_at(const int &row, const int &col) const {
  if(_data_col.isEmpty()) return 0.00;
  if(is_null(row,col)) return 0.00;
  switch(display) {
  case rfanalysis::DISPLAY_MIN:
    return table[col][row].min;
    break;
  case rfanalysis::DISPLAY_MAX:
    return table[col][row].max;
    break;
  case rfanalysis::DISPLAY_AVG:
    return table[col][row].avg();
    break;
  case rfanalysis::DISPLAY_COUNT:
    return table[col][row].count;
    break;
  case rfanalysis::DISPLAY_CALC:
    return table[col][row].calculated;
    break;
  }
  return 0.00; // just in case i guess
}

double rfanalysis::max_value() const {
  return _max_value;
}

double rfanalysis::min_value() const {
  return _min_value;
}

void rfanalysis::set_view(const QString &view) {
  if(view == "AVG") {
    change_display(DISPLAY_AVG);
  } else if(view == "MIN") {
    change_display(DISPLAY_MIN);
  } else if(view == "MAX") {
    change_display(DISPLAY_MAX);
  } else if(view == "COUNT") {
    change_display(DISPLAY_COUNT);
  } else if(view == "CALC") {
    change_display(DISPLAY_CALC);
  }
}

void rfanalysis::change_display(display_t d) {
  if(display != d) {
    display = d;
    change_data();
  }
}

QString rfanalysis::current_view() const {
  switch(display) {
  case rfanalysis::DISPLAY_MIN:
    return "MIN";
    break;
  case rfanalysis::DISPLAY_MAX:
    return "MAX";
    break;
  case rfanalysis::DISPLAY_AVG:
    return "AVG";
    break;
  case rfanalysis::DISPLAY_COUNT:
    return "COUNT";
    break;
  case rfanalysis::DISPLAY_CALC:
    return "CALC";
    break;
  }
  return QString();
}

QStringList rfanalysis::get_views() const {
  return QStringList({"AVG","MIN","MAX","COUNT","CALC"});
}

void rfanalysis::clear_data() {
  table.clear();
  errors.clear();
  if(is_2d()) {
    table.resize(1);
  } else {
    table.resize(xaxis.count());
  }
  for(int z=0;z<table.size();z++) {
    table[z].resize(yaxis.count());
  }
  change_geometry();
}

bool rfanalysis::test_filters(rftablemodel *log, const int &row) const {
  for(int x=0;x<filters.size();x++) {
    if(filters.at(x).matches(log,row) == false) return false;
  }
  return true;
}

bool rfanalysis::refactor(rfdatalog *log) {
  for(int x=0;x<filters.size();x++) {
    afilter *f = &filters[x];
    if(f->refactor(log) == false) return false;
  }
  if(log->index_of(yaxis.name) == -1) return false;
  if(is_3d() && log->index_of(xaxis.name) == -1) return false;
  if(log->index_of(_data_col) == -1) return false;
  return true;
}

void rfanalysis::analyze(QVector <rfdatalog *>logs) {
  clear_data();
  _min_value = __DBL_MAX__;
  _max_value = __DBL_MIN__;
  if(_data_col.isEmpty() || yaxis.name.isEmpty() || (is_3d() && xaxis.name.isEmpty())) return;

  for(int x=0;x<logs.size();x++) {
    if(refactor(logs.at(x)) == false) {
      errors.append("Could not refactor for log " + logs.at(x)->name);
    } else {
      if(is_2d()) {
        switch(method) {
        case rfanalysis::NEAREST_NEIGHBOUR:
          analyze_2d_standard(logs.at(x));
          break;
        case rfanalysis::YLINEAR:
          analyze_2d_linear(logs.at(x),filter_edges);
          break;
        default:
          errors.append("Invalid analysis mode for 2d table.");
          return;
        }
      } else {
        switch (method) {
        case rfanalysis::BILINEAR:
          analyze_3d_linear(logs.at(x),filter_edges,regression_table::BILINEAR);
          break;
        case rfanalysis::NEAREST_NEIGHBOUR:
          analyze_3d_standard(logs.at(x),table_axis::ALIGN_NN);
          break;
        case rfanalysis::XLINEAR:
          analyze_3d_linear(logs.at(x),filter_edges,regression_table::XLINEAR);
          break;
        case rfanalysis::YLINEAR:
          analyze_3d_linear(logs.at(x),filter_edges,regression_table::YLINEAR);
          break;
        case rfanalysis::LEFTALIGN:
          analyze_3d_standard(logs.at(x),table_axis::ALIGN_LH);
          break;
        case rfanalysis::RIGHTALIGN:
          errors.append("Right alignment is not yet implemented because I have not had time.");
          break;
        case rfanalysis::UNDEFINED:
          break;
        }
      }
    }
  }

  recalculate();
  change_data();
}

void rfanalysis::recalculate() {
  if(is_valid() == false) return;
  if(equation.isEmpty() == false) {
    mu::Parser parser;
    double n = 0.00;
    double x = 0.00;
    double y = 0.00;
    parser.DefineVar(QString("N").toStdWString(),&n);
    parser.DefineVar(QString("X").toStdWString(),&x);
    parser.DefineVar(QString("Y").toStdWString(),&y);
    parser.SetExpr(QString(equation).toStdWString());
    for(int col=0;col<n_columns();col++) {
      for(int row=0;row<n_rows();row++) {
        a_cell *cell = &table[col][row];
        n = cell->avg();
        x = x_axis_value(col);
        y = y_axis_value(row);
        cell->calculated = parser.Eval();
      }
    }
  } else { // zero values
    for(int col=0;col<n_columns();col++) {
      for(int row=0;row<n_rows();row++) {
        table[col][row].calculated = 0.00;
      }
    }
  }
  change_data();
}

table_axis rfanalysis::x_axis() const {
  return xaxis;
}

double rfanalysis::x_axis_value(const int &col) const {
  if(is_2d()) return 0;
  return xaxis.at(col);
}

table_axis rfanalysis::y_axis() const {
  return yaxis;
}

double rfanalysis::y_axis_value(const int &row) const {
  return yaxis.at(row);
}

void rfanalysis::analyze_3d_standard(rfdatalog *log, table_axis::_locate_alignment alignment) {
  // get col indexes
  int xidx = log->col_index(xaxis.name);
  int yidx = log->col_index(yaxis.name);
  int zidx = log->col_index(_data_col);
  if(xidx == -1 || yidx == -1 || zidx == -1) return;
  int n_rows = log->n_rows();
  // iterate
  for(int row=0;row<n_rows;row++) {
    if(test_filters(log,row) == true) { // filter unwanted data
      // locate data point
      int x = xaxis.locate_index(log->double_at(row,xidx),alignment);
      int y = yaxis.locate_index(log->double_at(row,yidx),alignment);
      if(x == -1 || y == -1) continue;

      int r = row;
      if(lagfilter != 0.00) {
        // calculate for lag here
        r = log->seek(r,lagfilter);
        if(r == -1)  continue; // out of range due to lag filter.
      }

      double d = log->double_at(r,zidx);
      table[x][y].add_data(d);
      if(d < _min_value) _min_value = d;
      if(d > _max_value) _max_value = d;
    }
  }
  change_data();
}

void rfanalysis::analyze_3d_linear(rfdatalog *log, bool filter_edges, regression_table::_analysis_mode mode) {
  // get col indexes
  int xidx = log->col_index(xaxis.name);
  int yidx = log->col_index(yaxis.name);
  int zidx = log->col_index(_data_col);
  if(xidx == -1 || yidx == -1 || zidx == -1) return;

  int n_rows = log->n_rows();

  //QVector <double>xaxis = x_axis().values;
  //QVector <double>yaxis = y_axis().values;

  if(xaxis.count() < 2 || yaxis.count() < 2) return; // requires a 2x2 minimum table size.

  regression_table reg(xaxis,yaxis,mode); // create new regression table

  // iterate
  for(int row=0;row<n_rows;row++) {
    if(test_filters(log,row) == false) continue;

    double x = log->double_at(row,xidx);
    double y = log->double_at(row,yidx);

    // filter edges
    if(filter_edges) {
      if(x < xaxis.at(0) || y < yaxis.at(0) || x > xaxis.at(xaxis.count() - 1) || y > yaxis.at(yaxis.count() - 1)) continue;
    }

    // find the data point as with lag filtering
    int r = row;
    if(lagfilter != 0.00) {
      // calculate for lag here
      r = log->seek(r,lagfilter);
      if(r == -1)  continue; // out of range due to lag filter.
    }
    double value = log->double_at(r,zidx);

    // store global analysis min max
    if(value < _min_value) _min_value = value;
    if(value > _max_value) _max_value = value;

    reg.add_data(x,y,value);

  }

  reg.calculate(); // slow -- notify user ?

  for(int x=0;x<this->n_columns();x++) {
    for(int y=0;y<this->n_rows();y++) {
      if(reg.is_null(x,y) == true) {
        table[x][y].set_null();
      } else {
        table[x][y].set_static_data(reg.at(x,y),reg.count_at(x,y),reg.min_at(x,y),reg.max_at(x,y));
      }
    }
  }

  change_data();
}

void rfanalysis::analyze_2d_linear(rfdatalog *log, bool filter_edges) {
  // get col indexes
  int yidx = log->col_index(yaxis.name);
  int zidx = log->col_index(_data_col);
  if(yidx == -1 || zidx == -1) return;

  int n_rows = log->n_rows();

  //QVector <double>xaxis = x_axis().values;
  //QVector <double>yaxis = y_axis().values;

  if(xaxis.count() < 2 || yaxis.count() < 2) return; // requires a 2x2 minimum table size.

  regression_table reg(yaxis); // create new regression table

  // iterate
  for(int row=0;row<n_rows;row++) {
    if(test_filters(log,row) == false) continue;

    double y = log->double_at(row,yidx);

    // filter edges
    if(filter_edges) {
      if(y < yaxis.at(0) || y > yaxis.at(yaxis.count() - 1)) continue;
    }

    // find the data point as with lag filtering
    int r = row;
    if(lagfilter != 0.00) {
      // calculate for lag here
      r = log->seek(r,lagfilter);
      if(r == -1)  continue; // out of range due to lag filter.
    }
    double value = log->double_at(r,zidx);

    // store global analysis min max
    if(value < _min_value) _min_value = value;
    if(value > _max_value) _max_value = value;

    reg.add_data(y,value);

  }

  reg.calculate(); // slow -- notify user ?

  for(int y=0;y<this->n_rows();y++) {
    if(reg.is_null(y) == true) {
      table[0][y].set_null();
    } else {
      table[0][y].set_static_data(reg.at(y),reg.count_at(y),reg.min_at(y),reg.max_at(y));
    }
  }

  change_data();
}

void rfanalysis::analyze_2d_standard(rfdatalog *log) {
  int yidx = log->col_index(yaxis.name);
  int zidx = log->col_index(_data_col);
  if(yidx == -1 || zidx == -1) return;
  for(int row=0;row<log->n_rows();row++) {
    if(test_filters(log,row) == true) { // filter unwanted data
      // locate data point
      int y = yaxis.locate_index(log->double_at(row,yidx));
      if(y == -1) continue;
      int r = row;
      if(lagfilter != 0.00) {
        // calculate for lag here
        r = r + log->seek(r,lagfilter);
        if(r == -1)  continue; // out of range due to lag filter.
      }
      double d = log->double_at(r,zidx);
      table[0][y].add_data(d);
      if(d < _min_value) d = _min_value;
      if(d > _max_value) d = _max_value;
    }
  }
  change_data();
}

void a_cell::set_static_data(const double &value, const double &count, const double &min, const double &max) {
  avgsum = value;
  static_avg = true;
  this->count = count;
  this->min = min;
  this->max = max;
}

void a_cell::add_data(const double &value) {
  static_avg = false;
  if(count == 0) {
    min = value;
    max = value;
  } else if(value < min) {
    min = value;
  } else if(value > max) {
    max = value;
  }
  count++;
  avgsum += value;
}

double a_cell::avg() const {
  if(static_avg) return avgsum;
  if(count == 0) return 0; // avoid div by 0
  return avgsum / (double)count;
}

void a_cell::set_null() {
  min = 0;
  max = 0;
  count = 0;
  static_avg = false;
  avgsum = 0;
}

