#include "rfmathtable.h"

rfmathtable::rfmathtable(QVector<rftablemodel *> tables) {
  this->tables = tables;

  table_data.resize(tables.size());
  table_symbols.resize(tables.size());

  static QString symbols = QStringLiteral("ABCDEFGHIJKLMNOPQRSTUVWabcdefghijklmnopqrstuvw");

  if(tables.size() > symbols.size()) {
    errors.append("TABLE COUNT TOO LARGE");
    return;
  }

  for(int x=0;x<tables.size();x++) {
    connect(tables[x],&rftablemodel::data_changed,this,&rfmathtable::parent_data_changed);
    table_data[x] = 0.00;
    table_symbols[x] = symbols.at(x);
    QString sym;
    sym.append(symbols.at(x));
    calc.DefineVar(sym.toStdWString(),&table_data[x]);
  }

  calc.DefineVar(QString("X").toStdWString(),&X);
  calc.DefineVar(QString("Y").toStdWString(),&Y);

  copy_layout(tables[0]); // default to first table layout.

  change_data();
}

bool rfmathtable::set_equation(const QString &eq) {
  errors.clear();

  calc.SetExpr(eq.toStdWString());

  try
  {
    double d = calc.Eval();
  }
  catch (mu::Parser::exception_type &e)
  {
    errors.append(QString::fromStdWString(e.GetMsg()));
    equation_valid = false;
    change_data();
    return false;
  }

  equation_valid = true;
  change_data();
  return true;
}

void rfmathtable::parent_data_changed() {
  change_data();
}

double rfmathtable::double_at(const int &row, const int &col) const {
  if(is_null(row,col) == true) return 0.00;
  if(equation_valid == false) return 0.00;

  X = xaxis.values.at(col);
  Y = yaxis.values.at(row);

  for(int x=0;x<tables.size();x++) table_data[x] = lookup(X,Y,tables[x]);

  return calc.Eval();
}

bool rfmathtable::is_null(const int &row, const int &col) const {
  if(equation_valid == false) return true;
  if(filter_null) {
    double x = x_axis_value(col);
    double y = y_axis_value(row);
    for(int tbl=0;tbl<tables.count();tbl++) {
      rftablemodel *m = tables[tbl];
      if(x < m->x_axis_value(0) || x > m->x_axis_value(m->n_columns() - 1)) return true;
      if(y < m->y_axis_value(0) || y > m->y_axis_value(m->n_rows() - 1)) return true;
      int ix = m->col_index(x);
      int iy = m->row_index(y);
      if(ix == -1) return true;
      if(iy == -1) return true;
      if(m->is_null(iy,ix)) return true;
    }
  }
  return false;
}

double rfmathtable::lookup(double x, double y, rftablemodel *m) const {
  switch(interp_type) {
  case rfmathtable::INTERP_LINEAR:
    return m->lookup_3d_linear(x,y);
    break;
  case rfmathtable::INTERP_NEIGHBOUR:
    return m->lookup_3d_neighbour(x,y);
    break;
  }
  return 0.00;
}

QString rfmathtable::string_at(const int &row, const int &col) const {
  return QString();
}

bool rfmathtable::valid() const {
  return table_type() != NOTABLE; // see table_type()
}

rftablemodel::_table_type rfmathtable::table_type() const {
  if(tables.isEmpty()) return NOTABLE; // if no tables, then we are not a table ourselves.
  // if all table types are equal we are good otherwise not.  compare all to the first table.
  for(int x=1;x<tables.size();x++) {
    if(tables[0]->table_type() != tables[x]->table_type()) return NOTABLE;
  }
  if(tables[0]->table_type() == DATALOG) return NOTABLE;
  return tables[0]->table_type();
}

void rfmathtable::set_view(const QString &view) {

}

table_axis rfmathtable::x_axis() const {
  return xaxis;
}

table_axis rfmathtable::y_axis() const {
  return yaxis;
}

bool rfmathtable::readonly() const {
  return true;
}

bool rfmathtable::auto_col_size() const {
  return true;
}

QString rfmathtable::col_name(const int &col) const {
  return QString::number(xaxis.at(col),'f',xaxis.precision());
}

QString rfmathtable::row_name(const int &row) const {
  return QString::number(yaxis.at(row),'f',yaxis.precision());
}

int rfmathtable::n_columns() const {
  if(table_type() == TABLE2D) return 1;
  return xaxis.count();
}

void rfmathtable::set_null_filter(bool enable) {
  filter_null = enable;
  change_data();
}

void rfmathtable::set_interp_type(rfmathtable::interp_type_t t) {
  interp_type = t;
  change_data();
}

void rfmathtable::copy_layout(rftablemodel *m) {
  if(table_type() == TABLE3D) {
    xaxis = m->x_axis();
  } else {
    xaxis.clear();
  }
  yaxis = m->y_axis();
  change_geometry();
}

void rfmathtable::copy_layout(int index) {
  copy_layout(tables.at(index));
}

QStringList rfmathtable::get_labels() const {
  QStringList out;
  for(int x=0;x<tables.size();x++) {
    QString s;
    s.append(table_symbols.at(x));
    out.append(s);
  }
  return out;
}

void rfmathtable::set_layout(table_geometry g) {
  xaxis = g.x;
  yaxis = g.y;
  change_geometry();
}


