#include "rftable.h"
#include <QDebug>

rftable::rftable() : rftablemodel() {

}

rftable::rftable(const rftablemodel &other) : rftablemodel() {
  copy_data(other);
}

rftable::rftable(const rftable &other) : rftable() {
  copy_data(other);
}

bool rftable::copy_layout(const rftablemodel &other) {
  set_layout(other.x_axis(),other.y_axis());
  change_geometry();
  change_data();
  return true;
}

bool rftable::copy_data(const rftablemodel &other) {
  if(layout_is_identical(other) == false) {
    if(copy_layout(other) == false) return false;
  }
  clear();
  for(int c=0;c<other.n_columns();c++) {
    for(int r=0;r<other.n_rows();r++) {
      if(other.is_null(r,c) == false) {
        set(other.double_at(r,c),r,c);
      } else {
        set_null(r,c);
      }
    }
  }
  change_data();
  return true;
}

dbdata rftable::export_data() const {
  dbdata out;
  out.table_name = "STOREDTABLE";
  out.set("2D",is_2d());
  QVariantList d;
  for(int x=0;x<data.size();x++) {
    if(data_null.at(x) == true) {
      d.append((double)NAN);
    } else {
      d.append((double)data.at(x));
    }
  }
  out.set("DAT",d);
  out.set("XAXIS",xaxis.encode_values());
  out.set("YAXIS",yaxis.encode_values());
  return out;
}

bool rftable::import_data(const dbdata &data) {
  table_axis xaxis;
  if(xaxis.decode_values(data.value("XAXIS").toString()) == false) return false;
  table_axis yaxis;
  if(yaxis.decode_values(data.value("YAXIS").toString()) == false) return false;
  set_layout(xaxis,yaxis,CLEAR_ALL);
  QVariantList l = data.value("DAT").toList();
  if(this->data.size() != l.size()) {
    clear();
    return false;
  }
  for(int x=0;x<l.size();x++) {
    double d = l.at(x).toDouble();
    if(d != d) {
      this->data[x] = 0.00;
      this->data_null[x] = true;
    } else {
      this->data[x] = d;
      this->data_null[x] = false;
    }
  }
  geometry_changed();
  return true;
}

table_axis rftable::x_axis() const {
  return xaxis;
}

table_axis rftable::y_axis() const {
  return yaxis;
}

void rftable::set_layout(table_axis xaxis, table_axis yaxis, _layout_handler handler, table_interpolation config) {
  switch(handler) {
  case rftable::INTERPOLATE:
    if(is_3d()) interpolate_3d(xaxis,yaxis,config);
    break;
  case rftable::CLEAR_ALL:
    this->xaxis = xaxis;
    this->yaxis = yaxis;
    resize();
    clear();
    break;
  case rftable::CLEAR_NONMATCHING:
    // IMPLEMENT ME
    break;
  case rftable::OVERWRITE:
  {
    rftable t;
    t.copy_data(*this);
    this->xaxis = xaxis;
    this->yaxis = yaxis;
    resize();
    clear();
    for(int c=0;c<t.n_columns();c++) {
      for(int r=0;r<t.n_rows();r++) {
        if(t.is_null(r,c) == false) set(t.double_at(r,c),r,c);
      }
    }
  }
    break;

  }
  change_data();
  change_geometry();
}

int rftable::data_index(const int &row, const int &col) const {
  return col + (row * n_columns());
}

bool rftable::layout_is_identical(const rftablemodel &other) {
  if(table_type() != other.table_type()) return false;
  if(other.n_columns() != n_columns()) return false;
  if(other.n_rows() != n_rows()) return false;
  if(other.x_axis().values != x_axis().values) return false;
  if(other.y_axis().values != y_axis().values) return false;
  return true;
}

int rftable::n_columns() const {
  if(is_2d()) return 1;
  return xaxis.count();
}

int rftable::n_rows() const {
  return yaxis.count();
}

bool rftable::is_null(const int &row, const int &col) const {
  if(in_bounds(row,col) == false) return true;
  return data_null.at(data_index(row,col));
}

QString rftable::col_name(const int &col) const {
  if(is_2d()) return QString();
  return QString::number(xaxis.values.at(col),'f',xaxis.precision());
}

QString rftable::row_name(const int &row) const {
  return QString::number(yaxis.values.at(row),'f',yaxis.precision());
}

QString rftable::string_at(const int &row, const int &col) const {
  if(in_bounds(row,col) == false) return QString();
  return "ERR";
}

double rftable::double_at(const int &row, const int &col) const {
  if(in_bounds(row,col) == false) return 0.00;
  if(is_null(row,col)) return 0.00;
  return data.at(data_index(row,col));
}

bool rftable::set(const double &d, const int &row, const int &col) {
  if(in_bounds(row,col) == false) return false;
  data[data_index(row,col)] = d;
  data_null[data_index(row,col)] = false;
  change_data();
  return true;
}

bool rftable::set(const QString &s, const int &row, const int &col) {
  return false;
}

bool rftable::set_null(const int &row, const int &col) {
  if(in_bounds(row,col) == false) return false;
  data[data_index(row,col)] = 0.00;
  data_null[data_index(row,col)] = true;
  change_data();
  return true;
}

bool rftable::in_bounds(const int &row, const int &col) const {
  if(row < 0 || row > n_rows() - 1) return false;
  if(col < 0 || col > n_columns() - 1) return false;
  return true;
}

bool rftable::contains_data() const {
  // if contains any non null data return true.
  for(int x=0;x<data_null.size();x++) {
    if(data_null.at(x) == false) return true;
  }
  return false;
}

bool rftable::data_complete() const {
  // if contains any null data return false;
  for(int x=0;x<data_null.size();x++) {
    if(data_null.at(x) == true) return false;
  }
  return true;
}

void rftable::resize() {
  data.resize(n_columns() * n_rows());
  data_null.resize(n_columns() * n_rows());
  change_geometry();
  change_data();
}

void rftable::interpolate_3d(table_axis xaxis, table_axis yaxis, table_interpolation config) {
  // copy this table
  rftable t;
  t.copy_data(*this);

  // overwrite and clear our layout
  set_layout(xaxis,yaxis);

  if(config.method == table_interpolation::LINEAR) {
    // recreate our data from the other table cell by cell using the standard linear lookup.
    for(int c=0;c<n_columns();c++) {
      for(int r=0;r<n_rows();r++) {
        set(t.lookup_3d_linear(this->xaxis.values.at(c),this->yaxis.values.at(r)),r,c);
      }
    }
  } else if(config.method == table_interpolation::SPLINE) {

  } else if(config.method == table_interpolation::NEIGHBOUR) {
    for(int c=0;c<n_columns();c++) {
      for(int r=0;r<n_rows();r++) {
        set(t.lookup_3d_neighbour(x_axis_value(c),y_axis_value(r)),r,c);
      }
    }
  } else {
    // not implemented
  }

  if(config.tail == table_interpolation::ZERO) {
    QRectF r;
    r.setTop(y_axis_value(0));
    r.setBottom(y_axis_value(n_rows() - 1));
    r.setLeft(x_axis_value(0));
    r.setRight(x_axis_value(n_columns() - 1));
    for(int x=0;x<n_columns();x++) {
      for(int y=0;y<n_rows();y++) {
        QPointF p(x_axis_value(x),y_axis_value(y));
        if(r.contains(p) == false) {
            set_null(y,x);
        }
      }
    }
  }

  change_geometry();
  change_data();
}

void rftable::clear() {
  data.fill(0.00);
  data_null.fill(true);
  change_data();
}
