#include "import_xdf_dialog.h"
#include "ui_import_xdf_dialog.h"
#include "database.h"
#include "../modules/module.h"
#include <QFileDialog>

#include <QXmlStreamReader>
#include <QDebug>
#include <QMap>

import_xdf_dialog::import_xdf_dialog(QWidget *parent) :
  QDialog(parent),
  ui(new Ui::import_xdf_dialog) {
  ui->setupUi(this);
  ui->ok_btn->setEnabled(false);
}


void import_xdf_dialog::parse_xdf() {
  tables.clear();
  table_geometry table;
  QChar current_axis = ' ';
  QVector <double>axis_values;
  bool in_axis = false;

  while (xdf.atEnd() == false) {

    xdf.readNext();

    QString name(xdf.name().toString());

    if(xdf.name() == QLatin1String("XDFTABLE")) {
      if(xdf.isStartElement()) {
        table.clear(); // start new table
        axis_values.clear();
      } else {
        if(table.y.values.isEmpty() || table.y.values.size() == 1) {
          table.y = table.x;
          table.x.clear();
        }
        if(table.x.values.size() == 1) table.x.clear();
        table.x.calc_precision();
        table.y.calc_precision();
        if(table.is_valid()) tables.append(table); // append table if valid.
      }
      continue;
    }

    if(xdf.name() == QLatin1String("title")) {
      if(xdf.isStartElement()) {
        table.dbname = xdf.readElementText().trimmed();
      }
      continue;
    }

    if(xdf.name() == QLatin1String("XDFAXIS")) {
      if(xdf.isStartElement()) {
        current_axis = xdf.attributes().value("id").at(0);
        in_axis = true;
      } else {
        if(current_axis == 'x') {
          table.x.values = axis_values;
        } else if(current_axis == 'y') {
          table.y.values = axis_values;
        }
        current_axis = ' ';
        axis_values.clear();
        in_axis = false;
      }
      continue;
    }

    if(xdf.name() == QLatin1String("units")) {
      if(xdf.isStartElement() && current_axis != ' ') {
        if(current_axis == 'x') {
          table.x.name = xdf.readElementText().trimmed();
        } else if(current_axis == 'y') {
          table.y.name = xdf.readElementText().trimmed();
        }
      } else {
        current_axis = ' ';
      }
      continue;
    }

    // invalidate link axises
    if(xdf.name() == QLatin1String("embedinfo")) {
      if(xdf.isStartElement() && xdf.attributes().value("type").toInt() > 0) {
        axis_values.clear();
        current_axis = ' ';
      }
      continue;
    }

    if(xdf.name() == QLatin1String("LABEL") && xdf.isStartElement() && in_axis) {
      bool ok;
      double v = xdf.attributes().value("value").trimmed().toDouble(&ok);
      if(ok == true) { // a conversion error invalidates this axis
        axis_values.append(v);
      } else {
        axis_values.clear();
        current_axis = ' ';
        in_axis = false;
      }
      continue;
    }

  }
}

import_xdf_dialog::~import_xdf_dialog() {
  delete ui;
}


void import_xdf_dialog::on_xdf_btn_clicked() {
  QString last_dir = database::get_config("LAST_XDF_DIR").toString();
  QString filename = QFileDialog::getOpenFileName(nullptr,"Select definition file",last_dir,
                                                  "TunerPro (*.xdf)",nullptr,QFileDialog::DontUseNativeDialog);
  if(filename.isEmpty()) return;

  QFile f(filename);
  if(f.open(QFile::ReadOnly) == false) {
    module::error("Could not open file: " + f.errorString());
    return;
  }

  QFileInfo i(filename);
  database::set_config("LAST_XDF_DIR",i.dir().path());
  QString basename = i.baseName();
  ui->xdf_btn->setText(basename);

  xdf.addData(f.readAll());
  parse_xdf();
  if (xdf.hasError()) {
    return;
  }

  ui->parameter_list->clear();
  for(int x=0;x<tables.size();x++) {
    QListWidgetItem *i = new QListWidgetItem;
    QString s = tables.at(x).dbname + " "  + tables.at(x).size_string();
    i->setText(s);
    i->setData(Qt::UserRole,x);
    ui->parameter_list->addItem(i);
  }

  return;
}

void import_xdf_dialog::on_parameter_list_itemSelectionChanged() {
  if(ui->parameter_list->selectedItems().size() == 1) {
    ui->ok_btn->setEnabled(true);
    selected_table = tables.at(ui->parameter_list->selectedItems().at(0)->data(Qt::UserRole).toInt());
  } else {
    ui->ok_btn->setEnabled(false);
  }
}

void import_xdf_dialog::on_ok_btn_clicked() {
    accept();
}

void import_xdf_dialog::on_cancel_btn_clicked() {
    reject();
}

void import_xdf_dialog::on_parameter_list_itemDoubleClicked(QListWidgetItem *item) {
    selected_table = tables.at(item->data(Qt::UserRole).toInt());
    accept();
}

void import_xdf_dialog::on_filter_box_textEdited(const QString &arg1) {
  QStringList search_terms = arg1.split(' ',Qt::SkipEmptyParts);
  int count = ui->parameter_list->count();
  bool first = false;
  ui->parameter_list->clearSelection();
  for(int x=0;x<count;x++) {
    QListWidgetItem *i = ui->parameter_list->item(x);
    i->setHidden(! matches_search(search_terms,i->text()));
    if(first == false && i->isHidden() == false) {//select first item
      i->setSelected(true);
      first = true;
    }
  }
}

bool import_xdf_dialog::matches_search(const QStringList &search, const QString &text) {
  if(search.isEmpty()) return true;
  for(int y=0;y<search.size();y++) {
    if(text.contains(search.at(y),Qt::CaseInsensitive) == false) {
      return false;
    }
  }
  return true;
}
