#include "ui_module_grapher2d.h"
#include "module_grapher2d.h"
#include <QVBoxLayout>
#include "../dialogs/chart_element_dialog.h"
#include <QDataStream>
#include <QInputDialog>
#include "database.h"
#include "../widgets/data_selector_widget.h"
#include "../dialogs/database_dialog.h"

module_grapher2d::module_grapher2d(module *parent) : ui(new Ui::module_grapher2d) {
  ui->setupUi(this);

  parent_module = parent;
  if(parent->module_type() != module::MODULE_DATALOG)  {
    setEnabled(false);
    return; //failsafe
  }

  tbl = (rfdatalog*)(parent->get_data());
  connect(tbl,&rfdatalog::data_changed,this,&module_grapher2d::source_data_changed);

  QApplication::setOverrideCursor(Qt::WaitCursor);

  set_name(parent->name() + " 2D Graph");
  set_icon("chart_line_2_icon&16.png");

  plotter = ui->plotter;

  // https://www.qcustomplot.com/index.php/tutorials/specialcases/axistags



  cursor = new QCPItemLine(plotter);
  cursor->setPen(QColor(Qt::black));

  clear();

  QApplication::restoreOverrideCursor();
}

void module_grapher2d::clear() {
  invalidate_data();
  invalidate_time_axis();
  plotter->clearGraphs();
  plotter->xAxis->setVisible(true); // use for time axis.
  plotter->xAxis2->setVisible(false);
  plotter->yAxis->setVisible(false);
  plotter->yAxis2->setVisible(false);
  plotter->setFont(QFontDatabase::systemFont(QFontDatabase::FixedFont));
  plotter->setBackground(palette().window().color());
  plotter->setOpenGl(true);

  plotter->setInteraction(QCP::iRangeDrag, true);
  plotter->setInteraction(QCP::iRangeZoom, true);
  plotter->axisRect()->setRangeDrag(Qt::Horizontal);
  plotter->axisRect()->setRangeZoom(Qt::Horizontal);

  for(int x=plotter->graphCount() - 1; x >= 0 ; x--){
    QCPAxis *a = plotter->graph(x)->valueAxis();
    plotter->removeGraph(x);
    delete a;
  }

  populate_time_axis();
}

void module_grapher2d::move_cursor(double y_value) {
  if(cursor == nullptr) return;
  cursor->start->setCoords(y_value,0);
  cursor->end->setCoords(y_value,1);
  plotter->replot(QCustomPlot::rpQueuedRefresh);
}

void module_grapher2d::source_data_changed() {
  invalidate_time_axis();
  invalidate_data();
  populate_data();
}

dbdata module_grapher2d::export_data() const {
  dbdata out;
  out.table_name = "GRAPH2D";
  int count = plotter->graphCount();
  if(count == 0) return out;
  out.set("COUNT",count);
  for(int x=0;x<plotter->graphCount();x++) {
    QCPGraph *g = plotter->graph(x);
    QCPAxis *a = g->valueAxis();
    dbdata d;
    d.set("N",x);
    d.set("LABEL",a->label());
    d.set("RANGE_LOWER",a->range().lower);
    d.set("RANGE_HIGHER",a->range().upper);
    d.set("COLOR",a->labelColor());
    out.set_subdata("GRAPH" + QString::number(x),d);
  }
  return out;
}

bool module_grapher2d::import_data(const dbdata &in) {
  clear();
  invalidate_data();
  int count = in.value("COUNT").toInt();
  for(int x=0;x<count;x++) {
    dbdata d = in.subdata("GRAPH" + QString::number(x));
    if(d.is_valid() == false) return false;
    int graph_number = d.value("N").toInt();
    if(graph_number != x) return false;
    //qDebug() << d.value("LABEL").toString() << d.value("RANGE_LOWER").toDouble() << d.value("RANGE_UPPER").toDouble();
    QCPGraph *g = plotter->addGraph(plotter->xAxis,plotter->axisRect()->addAxis(QCPAxis::atLeft));
    QCPAxis *a = g->valueAxis();
    a->setLabel(d.value("LABEL").toString());
    a->setRange(d.value("RANGE_LOWER").toDouble(),d.value("RANGE_HIGHER").toDouble());
    QColor color = d.value("COLOR").value<QColor>();
    a->setLabelColor(color);
    a->setTickLabelColor(color);
    a->setTickPen(QPen(color));
    g->setPen(QPen(QBrush(color),1));
  }
  populate_data();
  return true;
}

module *module_grapher2d::clone() const {
  module_grapher2d *out = new module_grapher2d(parent_module);
  out->copy_children((module*)this);
  out->import_data(export_data());
  return out;
}


void module_grapher2d::configure() {
  chart_element_dialog *d = new chart_element_dialog(plotter,tbl);
  connect(d,&chart_element_dialog::accepted,this,&module_grapher2d::populate_data);
  connect(d,&chart_element_dialog::finished,d,&chart_element_dialog::deleteLater);
  d->setModal(true);
  d->open();
}

void module_grapher2d::invalidate_data() {
  data_load_cache.clear();
}

void module_grapher2d::invalidate_time_axis() {
  keys.clear();
}

void module_grapher2d::populate_time_axis() {
  // either get the time axis or generate a static one by row index.
  if(tbl->time_col != -1) {
    keys = tbl->col_vector(tbl->time_col);
    plotter->xAxis->setLabel(tbl->col_name(tbl->time_col) + " : " + tbl->name);
    plotter->xAxis->setRange(0,30);
  } else {
    double z  = 0;
    for(int x=0;x<tbl->n_rows();x++) {
      keys.append(z);
      z++;
    }
    plotter->xAxis->setLabel(QString("Row#") + " : " + tbl->name);
    plotter->xAxis->setRange(0,1000);
  }
}

void module_grapher2d::populate_data() {
  QApplication::setOverrideCursor(Qt::WaitCursor);
  if(keys.isEmpty()) populate_time_axis();
  for(int x=0;x<plotter->graphCount();x++) {
    QApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
    QCPGraph *g = plotter->graph(x);
    QCPAxis *a = g->valueAxis();
    QString col_name = a->label();
    if(data_load_cache.value(g) == col_name) continue; // skip already loaded data.
    int i = tbl->col_index(col_name);
    if(i != -1) {
      g->data()->clear();
      g->addData(keys,tbl->col_vector(i),true);
      data_load_cache[g] = col_name;
    }
  }
  plotter->replot(QCustomPlot::rpQueuedRefresh);
  QApplication::restoreOverrideCursor();
}

void module_grapher2d::on_data_btn_clicked() {
  configure();
}

void module_grapher2d::on_database_btn_clicked() {
  dbdata db_out = export_data();
  database_dialog dialog("GRAPH2D",db_out);
  if(dialog.exec() == database_dialog::Accepted) {
    dbdata db = dialog.selected_item;
    if(import_data(db) == false) {
      plotter->clearGraphs();
      error("Could not decode analysis.");
      return;
    }
  }
}
