#include "datalog_load_dialog.h"
#include "ui_datalog_load_dialog.h"

#include <QMessageBox>
#include "../datastructures/rfdatalog.h"
#include "database.h"

datalog_load_dialog::datalog_load_dialog(QString filename, QWidget *parent) :
  QDialog(parent),
  ui(new Ui::datalog_load_dialog) {
  ui->setupUi(this);
  this->filename = filename;
  ui->ok_btn->setEnabled(false);
  QString last = database::get_config("LAST_DATALOG_FORMAT",QString()).toString();
  if(last.isEmpty() == false) {
    decode(last);
  }
  log = new rfdatalog;
  ui->renderer->set_data(log);
  refresh_log();
  // don't ask again if log identical.
  //if(last_col_list.isEmpty() == false && col_list == last_col_list) accept();
}

datalog_load_dialog::~datalog_load_dialog() {
  delete ui;
  delete log;
}

void datalog_load_dialog::refresh_log() {
  if(loading) return;
  config_datalog(log,true);
  if(log->load_file(filename) == false) {
    ui->status->setText(log->parse_errors.join('\n'));
    ui->ok_btn->setEnabled(false);
  } else {
    ui->ok_btn->setEnabled(true);
    ui->status->setText("Load OK");
    ui->timeref_btn->set_columns(log);
    this->col_list = log->column_string_list();
  }
  ui->renderer->geometry_changed();
}

void datalog_load_dialog::error(const QString &err_msg) {
  QMessageBox::warning(nullptr,"Datalog Error",err_msg);
}

void datalog_load_dialog::config_datalog(rfdatalog *log, bool trial) {
  int line_limit = -1;

  char seperator = ',';
  if(ui->tsv_box->isChecked()) seperator = '\t';

  int header_row = ui->header_line_box->value();
  int data_line = ui->data_line_box->value();

  if(trial == true) line_limit = data_line + 20;

  QString time_col = ui->timeref_btn->col_name;

  bool shittycsv = ui->shitty_datalog->isChecked();
  bool boolean = ui->boolean_translate_btn->isChecked();

  log->configure_log(seperator,header_row,data_line,shittycsv,boolean,line_limit,time_col);
}

void datalog_load_dialog::on_cancel_btn_clicked() {
  reject();
}

void datalog_load_dialog::on_ok_btn_clicked() {
  database::set_config("LAST_DATALOG_FORMAT",encode());
  accept();
}

void datalog_load_dialog::on_csv_box_toggled(bool checked) {
  if(checked) refresh_log();
}

void datalog_load_dialog::on_tsv_box_toggled(bool checked) {
  if(checked) refresh_log();
}

void datalog_load_dialog::on_header_line_box_valueChanged(int arg1) {
  //ui->data_line_box->setMinimum(arg1);
  refresh_log();
}

void datalog_load_dialog::on_data_line_box_valueChanged(int arg1) {
  //ui->header_line_box->setMaximum(arg1);
  refresh_log();
}

void datalog_load_dialog::on_boolean_translate_btn_toggled(bool checked) {
  refresh_log();
}

void datalog_load_dialog::on_shitty_datalog_toggled(bool checked) {
  refresh_log();
}

bool datalog_load_dialog::decode(const QString &in) {
  dbdata d;
  if(d.from_string(in,"DATALOGCONFIG") == false) return false;

  last_col_list = d.value("COL_LIST").toStringList();
  quint8 seperator = d.value("SEP").toInt();
  int header_row = d.value("HEADER_ROW").toInt();
  int data_line = d.value("DATA_LINE").toInt();
  QString time_col = d.value("TIME_COL").toString();
  bool shittycsv = d.value("SHITTY_CSV").toBool();
  bool boolean = d.value("BOOLEAN").toBool();

  loading = true;

  ui->csv_box->setChecked(seperator == ',');
  ui->header_line_box->setValue(header_row);
  ui->data_line_box->setValue(data_line);
  ui->timeref_btn->set_column(time_col);
  ui->shitty_datalog->setChecked(shittycsv);
  ui->boolean_translate_btn->setChecked(boolean);

  loading = false;

  return true;
}

QString datalog_load_dialog::encode() const {
  char seperator = ',';
  if(ui->tsv_box->isChecked()) seperator = '\t';

  quint8 header_row = ui->header_line_box->value();
  quint8 data_line = ui->data_line_box->value();

  QString time_col = ui->timeref_btn->col_name;

  bool shittycsv = ui->shitty_datalog->isChecked();
  bool boolean = ui->boolean_translate_btn->isChecked();

  dbdata d;
  d.set("COL_LIST",col_list);
  d.set("SEP",(int)seperator);
  d.set("HEADER_ROW",header_row);
  d.set("DATA_LINE",data_line);
  d.set("TIME_COL",time_col);
  d.set("SHITTY_CSV",shittycsv);
  d.set("BOOLEAN",boolean);

  return d.to_string("DATALOGCONFIG");
}
