#include "datalog_extended.h"
#include "ui_datalog_extended.h"
#include "datalog.h"

#include <QDebug>
#include <QFileDialog>
#include <QProgressDialog>
#include <QMessageBox>
#include <QFile>

datalog_extended::datalog_extended(QWidget *parent) :
  QWidget(parent),
  ui(new Ui::datalog_extended) {
  ui->setupUi(this);
  def = nullptr;
  current_display_packet = nullptr;
  lvl = datalog_element::VIEW_BASIC;
  n_errors = 0;
  patch_level = 0x00;
  update_pktage(nullptr);
}

datalog_extended::~datalog_extended() {
  delete ui;
}

unsigned int datalog_extended::n_elements(datalog_definition *def, datalog_element::VIEW_LEVEL lvl) {
  datalog_element *e = def->get_first_element();
  unsigned int x = 0;
  while(e != nullptr) {
    if(e->level >= lvl &&
       patch_level >= e->min_patch &&
       patch_level <= e->max_patch) x++;
    e = e->next;
  }
  return x;
}

void datalog_extended::refresh_display() {
  if(def == nullptr) return;
  datalog_element *x = def->get_first_element();
  static QString error_cache = QString(" ");
  int row_number = 0;
  while(x != nullptr) {
    if(x->level >= lvl && patch_level >= x->min_patch && patch_level <= x->max_patch) {
      QString element_string = x->get_string(current_display_packet);
      if(! ui->data_display->item(row_number,1) || // item exists
         element_string.compare(ui->data_display->item(row_number,1)->text()) != 0) { // is different
        QTableWidgetItem *table_cell = new QTableWidgetItem(element_string);
        // do hilighting
        if(x->is_bool() && current_display_packet != nullptr &&
           x->get_bool(current_display_packet) == true) { // is binary value
          table_cell->setBackgroundColor(QColor(80,80,255));
        } else { // is numeric value
          if(x->is_high(current_display_packet) == true) { // value high
            table_cell->setBackgroundColor(QColor(80,255,80));
          } else if(x->is_low(current_display_packet) == true) { // value low
            table_cell->setBackgroundColor(QColor(255,80,80));
          }
        }
        ui->data_display->setItem(row_number,1,table_cell); // replace item
      }
      row_number++;
    }
    x = x->next;
  }
  if(n_errors != 0) {
    QString err_str = def->err_str(current_display_packet,patch_level);
    if(err_str.isEmpty() == true) err_str = "No Errors";
    err_str.append("\n" + QString::number(n_errors) + " Errors Monitored");
    if(error_cache.compare(err_str) != 0) {
      ui->err_display->setPlainText(err_str);
      error_cache = err_str;
    }
  }
}

void datalog_extended::new_packet_recieved(datalog_packet *packet) {
  if(packet == nullptr) { // clear
    current_display_packet = nullptr;
    refresh_display();
    return;
  }
  if(packet->def == def) { // belongs to us
    current_display_packet = packet;
  } else { // doesn't belong to us
    current_display_packet = packet->get_prev(def->get_device(),def->get_msg_number());
  }
  if(this->isVisible() == true) {
    refresh_display();
    update_pktage(packet);
  }
}

void datalog_extended::update_pktage(datalog_packet *latest_pkt) {
  if(latest_pkt == nullptr || current_display_packet == nullptr) {
    ui->display_pktage->setText("NO DATA");
    return;
  } else {
    float timestamp_diff = latest_pkt->timestamp - current_display_packet->timestamp;
    ui->display_pktage->setText(QString::number(timestamp_diff / 1000,'f',1) + "sec");
  }
}

void datalog_extended::new_patch_level(int patch_level_in) {
  patch_level = patch_level_in;
  reconfigure();
}

void datalog_extended::display_config_changed() {
  refresh_display();
}

void datalog_extended::set_definition(datalog_definition *def_in) {
  def = def_in; // upd definition ptr
  if(def->get_device() == 0xF4 && def->get_msg_number() == 0x00) ui->acquire_btn->setChecked(true);
  reconfigure();
}

void datalog_extended::reconfigure() {
  if(def == nullptr) return;
  // enable/disable controls
  if(def->get_device() == 0xE4 && patch_level < 0x03) { // e-side pre patch
    ui->acquire_btn->setEnabled(false);
    ui->get_once_btn->setEnabled(false);
    ui->acquire_btn->setChecked(false);
  } else {
    ui->acquire_btn->setEnabled(true);
    ui->get_once_btn->setEnabled(true);
  }

  // config columns
  ui->data_display->setColumnWidth(0,320);
  // config rows
  ui->data_display->setRowCount(n_elements(def,lvl));

  int row_number = 0;
  datalog_element *x = def->get_first_element();
  while(x != nullptr) {
    if(x->level >= lvl && x->matches_patchlvl(patch_level)) {
      QTableWidgetItem *description_element = new QTableWidgetItem(x->full_title());
      ui->data_display->setItem(row_number,0,description_element);
      row_number++;
    }
    x = x->next;
  }
  if(row_number == 0) {
    ui->data_display->setRowCount(1);
    QTableWidgetItem *empty_table = new QTableWidgetItem("No Definitions Available");
    empty_table->setBackgroundColor(QColor(255,80,80));
    ui->data_display->setItem(0,0,empty_table);
  }
  ui->err_display->clear();
  n_errors = def->n_errors(patch_level);
  if(n_errors == 0) ui->err_display->setPlainText("No errors available in this message.");
  refresh_display();
}

void datalog_extended::showEvent(QShowEvent *event) {
  (void)event;
  refresh_display();
  if(current_display_packet != nullptr) update_pktage(current_display_packet->get_latest());
}

void datalog_extended::on_filter_slider_valueChanged(int value) {
  if(value == 1) {
    lvl = datalog_element::VIEW_BASIC;
  } else if(value == 2) {
    lvl = datalog_element::VIEW_EXTENDED;
  } else if(value == 3) {
    lvl = datalog_element::VIEW_HACKER;
  }
  reconfigure();
}

void datalog_extended::on_get_once_btn_clicked() {
  def->acq.set(true);
}

void datalog_extended::on_acquire_btn_toggled(bool checked) {
  def->loop.set(checked);
  def->acq.set(checked);
  ui->get_once_btn->setEnabled(!checked);
}

void datalog_extended::on_btn_exportcsv_clicked() {
  if(current_display_packet == nullptr) return;
  // get filename
  QFileDialog dialog;
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setDefaultSuffix("csv");
  dialog.setNameFilter("Comma Seperated File (*.csv)");
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  eehack_settings config;
  dialog.setDirectory(config.get_savepath());
  QStringList file_names;
  QString filename;
  if(dialog.exec()) {
    file_names = dialog.selectedFiles();
    filename = file_names.takeFirst();
  } else {
    return;
  }
  if(filename.isEmpty() == true || filename.isNull() == true) return;
  // check it out
  QFile f(filename);
  f.open(QIODevice::WriteOnly|QIODevice::Text);
  if(f.isWritable() == false) {
    error_dialog("Could not open file for writing.");
    return;
  }
  QTextStream data_out(&f);
  // write header
  datalog_element *e = def->get_first_element();
  bool is_first_element = true;
  while(e != nullptr) {
    if(e->level >= lvl && patch_level >= e->min_patch && patch_level <= e->max_patch) {
      if(is_first_element == false) data_out << ",";
      is_first_element = false;
      data_out << e->long_name;
      if(e->uom.isEmpty() == false) { // print uom
        data_out << "(";
        data_out << e->uom;
        data_out << ")";
      }
    }
    e = e->next;
  }
  data_out << "\n";
  // write data
  datalog_packet *pkt = current_display_packet->log->oldest_packet();
  while(pkt != nullptr) { // iterate packets
    if(pkt->def == def) { // associated
      is_first_element = true;
      datalog_element *x = def->get_first_element();
      while(x != nullptr) { // iterate elements
        if(x->level >= lvl && patch_level >= x->min_patch && patch_level <= x->max_patch) {
          if(is_first_element == false) data_out << ",";
          is_first_element = false;
          data_out << x->get_string(pkt,true);
        }
        x = x->next;
      }
    }
    data_out << "\n";
    pkt = pkt->get_next();
  }
}

void datalog_extended::error_dialog(QString msg) {
  QMessageBox msgBox;
  msgBox.setText("EEHack: An error has occurred!");
  msgBox.setInformativeText(msg);
  msgBox.setIcon(QMessageBox::Critical);
  msgBox.setStandardButtons(QMessageBox::Ok);
  msgBox.setDefaultButton(QMessageBox::Ok);
  msgBox.exec();
}


void datalog_extended::on_rate_slider_valueChanged(int value) {
  def->frequency.set(value);
}
