#include <QApplication>
#include <QThread>
#include <QProgressDialog>
#include <QGraphicsEllipseItem>
#include "analyzer.h"
#include "ui_analyzer.h"
#include "datastream.h"
#include "datalog.h"
#include "config.h"
#include "common.h"
#include "bin_file.h"

analyzer_ui::analyzer_ui(datalog *log_in, QWidget *parent) :
  QWidget(parent),
  ui(new Ui::analyzer) {
  ui->setupUi(this);
  // -------------
  anl = new analysis; // new analysis object
  log = log_in;
  ui->afr_ve_table->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  ui->afr_ve_table->verticalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  ui->afr_maf_table->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  ui->anl_blm_chart->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  ui->anl_blm_chart->verticalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  ui->afr_pe_table->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  knock_scene = new QGraphicsScene();
}

void analyzer_ui::resizeEvent(QResizeEvent* event) {
   QWidget::resizeEvent(event);
   draw_results_knock();
}

analyzer_ui::~analyzer_ui() {
  delete anl;
  delete ui;
}

void analyzer_ui::on_interface_loaded() {
  // preset slider displays
  on_anl_mintime_slider_valueChanged(ui->anl_mintime_slider->value());
  on_anl_mincooltmp_slider_valueChanged(ui->anl_mincooltmp_slider->value());
  on_anl_afrcounts_slider_valueChanged(ui->anl_afrcounts_slider->value());
  // throw slider defaults into analyzer
  anl->min_afrcounts = ui->anl_afrcounts_slider->value();
  anl->min_cooltmp = ui->anl_mincooltmp_slider->value();
  anl->min_timestamp = ui->anl_mintime_slider->value();
  anl->use_integrator = ui->setting_anl_int->isChecked();
  anl->anl_filter_idleanddecel = ui->setting_anl_extracells->isChecked();
  anl->wbpercent = ui->setting_display_wb_percent->isChecked();
  anl->worst_case_blm = ui->setting_afrblmavg->isChecked();
  // configure knock display
  ui->anl_knockchart->setScene(knock_scene);
  draw_results_knock(); // draw blank chart
  wideband_config_updated(); // enables/disables wideband
  if(config.get_analyze_wideband() == true) { // remember wb/nb setting
    ui->afr_input_wide->setChecked(true);
  } else {
    ui->afr_input_narrow->setChecked(true);
  }
  label_anl_graphs();
}

//--------------------------------------------

void analyzer_ui::on_analyzeDataBtn_clicked() {
  // routine to check if any data exists ...
  if(ui->anl_all_btn->isChecked() == false && log->is_data() == false) { // if only one log and it's empty
    emit info_dialog("No recorded data to analyze.\nEither record some data, or use the load button on the datalogger page and select 'analyze all logs'.");
    return;
  } else if(ui->anl_all_btn->isChecked() == true) { // if multiple logs
    // search for any data
    bool any_data = false;
    datalog *search_log = log->get_first();
    while(search_log != nullptr) {
      if(search_log->is_data() == true) {
        any_data = true;
        break;
      }
      search_log = search_log->get_next();
    }
    if(any_data == false) {
      emit info_dialog("No data to analyze in any loaded log.\nUse the LOAD button on the datalogger page to load log data.");
      return;
    }
  }
  //-------------
  anl->reset_analyzer();
  ui->analyzeDataBtn->setEnabled(false);
  //-------------
  datalog *anl_log;
  if(ui->anl_all_btn->isChecked() == true) {
    anl_log = log->get_first();
  } else {
    anl_log = log;
  }
  //-------------
  // delete lock all logs
  datalog *lock_log = log->get_first();
  while(lock_log != nullptr) {
    lock_log->del_lock.lock();
    lock_log = lock_log->get_next();
  }
  //-------------
  QProgressDialog progress("Analyzing data...","ABORT",0,1);
  progress.setMinimumDuration(0);
  progress.setWindowModality(Qt::WindowModal);
  progress.setAutoClose(false);
  //-------------
  while(anl_log != nullptr) {
    // analyze
    if(anl_log->is_data() == true) {
      progress.setRange(0,anl_log->get_pkt_count());
      datalog_packet *x = anl_log->oldest_packet();
      while(x->is_last_packet() == false) {
        anl->analyze_segment(x);
        x = x->get_next();
        progress.setValue(x->sequence);
        if(progress.wasCanceled()) {
          goto ANL_CANCEL;
        }
      }
    }
    // advance
    if(ui->anl_all_btn->isChecked() == true) {
      anl_log = anl_log->get_next();
    } else {
      anl_log = nullptr; // breaks loop
    }
  }
  //-------------
  ANL_CANCEL:
  //-------------
  draw_analysis_results(); // set fields
  progress.cancel();
  //-------------
  ui->analyzeDataBtn->setEnabled(true);
  //-------------
  // clear delete locks
  // delete lock all logs
  lock_log = log->get_first();
  while(lock_log != nullptr) {
    lock_log->del_lock.unlock();
    lock_log = lock_log->get_next();
  }
  //-------------
  QApplication::restoreOverrideCursor();
}

void analyzer_ui::wideband_config_updated() {
  if(config.get_wideband_enable() == true) {
    ui->afr_input_wide->setEnabled(true);
  } else {
    ui->afr_input_wide->setEnabled(false);
    ui->afr_input_narrow->setChecked(true);
    config.set_analyze_wideband(false);
  }
}

void analyzer_ui::def_parse_err_in(QString str) {
  emit error_dialog(str);
}

void analyzer_ui::on_anl_mincooltmp_slider_valueChanged(int value) {
   ui->anl_mincooltmp_display->setText(QString("%1").arg(value));
   anl->min_cooltmp = value;
}

void analyzer_ui::on_anl_mintime_slider_valueChanged(int value) {
  ui->anl_mintime_display->setText(QString("%1").arg(value));
  anl->min_timestamp = value;
}

void analyzer_ui::on_anl_afrcounts_slider_valueChanged(int value) {
  ui->anl_afrcounts_display->setText(QString("%1").arg(value));
  anl->min_afrcounts = value;
}

void analyzer_ui::on_afr_input_narrow_toggled(bool checked) {
  if(checked == false) return;
  ui->grp_nbfilter->setEnabled(true);
  ui->group_wbfilter->setEnabled(false);
  anl->wideband = !checked;
}

void analyzer_ui::on_afr_input_wide_toggled(bool checked) {
  if(checked == false) return;
  ui->grp_nbfilter->setEnabled(false);
  ui->group_wbfilter->setEnabled(true);
  anl->wideband = checked;
  config.set_analyze_wideband(true);
}

//--------------------------------------------

fcell::fcell() {
  low = 0;
  high = 0;
  average_accumulator = 0;
  count = 0;
}

void fcell::clear() {
  low = 0;
  high = 0;
  average_accumulator = 0;
  count = 0;
}

float fcell::avg() {
  if(count > 1) {
    return average_accumulator / count;
  } else if(count < 0) {
    return 0;
  } else {
    return average_accumulator;
  }
}

void fcell::add_data(float value) {
  // analyze
  if(count == 0) { // first record
    high = value;
    low = value;
  } else {
    if(value > high) high = value;
    if(value < low) low = value;
  }
  average_accumulator += value;
  count += 1;
}

void fcell::add_data(float value, float min, float max) {
  if(value > min && value < max) add_data(value);
}

//--------------------------------------------

analysis::analysis() {
  min_cooltmp = 50;
  min_timestamp = 60;
  worst_case_blm = false;
  wideband = false;
  wbcl = false;
  wbpercent = false;
  reset_analyzer();
  ve_map_list = {20,25,30,35,40,45,50,55,60,65,70,75,80,85,90,95,100};
  ve_rpm_list = {400,600,800,1000,1200,1400,1600,1800,2000,2200,2400,2600,2800,3000,3200,3400,3600,3800,4000,4500,5000,5500,6000,6500,7000};
}

void analyzer_ui::on_setting_anl_int_toggled(bool checked) {
  anl->use_integrator = checked;
}

void analyzer_ui::on_setting_anl_extracells_toggled(bool checked) {
  anl->anl_filter_idleanddecel = checked;
}

void analyzer_ui::on_setting_afrblmavg_toggled(bool checked) {
  anl->worst_case_blm = checked;
}

void analyzer_ui::on_setting_wb_cl_toggled(bool checked) {
  anl->wbcl = checked;
}

void analyzer_ui::on_setting_enh_accuracy_toggled(bool checked) {
   anl->enhanced_accuracy = checked;
}

void analyzer_ui::on_setting_display_wb_percent_toggled(bool checked) {
   anl->wbpercent = checked;
}

void analysis::reset_analyzer() {
  lo2.clear();
  ro2.clear();
  knock_list.clear();
  idle_trim[0].clear();
  idle_trim[1].clear();
  idle_map.clear();
  idle_maf.clear();
  lcrosscount = 0; rcrosscount = 0;
  lstate = 0; rstate = 0;
  rstuck = 0; lstuck = 0;
  total_knock_events = 0;
  total_knock_counts = 0;
  for(int x=0;x<ve_rpm_list.size();x++) {
    for(int y=0;y<ve_map_list.size();y++) {
      ve_trim[x][y].clear();
    }
  }
  for(int x=0;x<70;x++) {
    maf_trim[x].clear();
  }
  for(int rpm_axis=0;rpm_axis<28;rpm_axis++) {
    for(int x=0;x<3;x++) pe_fuel[rpm_axis][x].clear();
  }
  for(int x=0;x<19;x++) {
    for(int y=0;y<4;y++) {
      blm[x][y].clear();
    }
  }
}

bool analysis::packet_valid(datalog_packet *p) {
  int message_number = p->get_msgnumber();
  if(message_number == 0x00) return true;
  if(message_number == 0x05 && p->log->info.patch_version.get() > 0x00) return true;
  return false;
}

void analysis::analyze_segment(datalog_packet *p) {
  if(packet_valid(p) == false) return; // skip packets with no valid data.

  parse_record = p;

  // bail if out of global thresholds
  if(p->get_float("COOLTMP") < min_cooltmp) return;
  if(parse_record->timestamp < min_timestamp * 1000) return;
  if(p->get_float("RPM") < 200) return; // eng probably not running

  // run various analyzers
  analyze_o2();
  if(wideband == true) {
    analyze_wb_afr();
  } else {
    analyze_nb_afr();
  }
  analyze_idle();
  analyze_pe();
  analyze_knock();
  analyze_blm();
}

//--------------------------------------------

void analyzer_ui::draw_results_o2() {
  //------LH
  ui->anl_o2record_l->setText(QString("%1").arg(anl->lo2.count));
  ui->anl_peakmv_l->setText(QString("%1mv").arg(anl->lo2.high));
  ui->anl_lowmv_l->setText(QString("%1mv").arg(anl->lo2.low));
  ui->anl_avgmv_l->setText(QString("%1mv").arg(anl->lo2.avg()));
  ui->anl_crosscount_l->setText(QString("%1").arg(anl->lcrosscount));
  //------RH
  ui->anl_peakmv_r->setText(QString("%1mv").arg(anl->ro2.high));
  ui->anl_lowmv_r->setText(QString("%1mv").arg(anl->ro2.low));
  ui->anl_avgmv_r->setText(QString("%1mv").arg(anl->ro2.avg()));
  ui->anl_crosscount_r->setText(QString("%1").arg(anl->rcrosscount));
}

//--------------------------------------------

void analysis::analyze_pe() {
  // filtering
  if(parse_record->get_bool("PE") == false) return;

  // get vars
  int current_rpm = parse_record->get_int("RPM");
  float current_map = parse_record->get_float("MAP");
  int rpm_axis = select_cell(7000,28,current_rpm);
  int map_cell = 0;
  if(current_map < 70) {
    map_cell = 0;
  } else if(current_map > 90) {
    map_cell = 2;
  } else { // (70-90)
    map_cell = 1;
  }

  float current_trim;
  if(config.get_wideband_enable() == false) { // regardless of nb/wb setting, use wideband in pe
    current_trim = ( parse_record->get_float("LO2") + parse_record->get_float("RO2") ) / 2;
  } else {
    current_trim = parse_record->get_float(config.get_wb_str());
  }

  pe_fuel[rpm_axis][map_cell].add_data(current_trim);
}

//--------------------------------------------

void analysis::analyze_idle() {
  if(parse_record->get_int("BLMCELL") != 16) return;
  if(wideband == true) {
    if(wbcl != parse_record->get_bool("CL")) return;
    idle_trim[0].add_data(parse_record->get_float(config.get_wb_str()));
    idle_trim[1].add_data(parse_record->get_float(config.get_wb_str()));
    idle_map.add_data(parse_record->get_float("MAP"));
    idle_maf.add_data(parse_record->get_float("MAF"),1,99999);
  } else {
    if(parse_record->get_bool("CL") == false) return;
    idle_trim[0].add_data(parse_record->get_int("LBLM"));
    idle_trim[1].add_data(parse_record->get_int("RBLM"));
    idle_map.add_data(parse_record->get_float("MAP"));
    idle_maf.add_data(parse_record->get_float("MAF"),1,99999);
  }

}

//--------------------------------------------

void analysis::analyze_blm() {
  if(parse_record->get_bool("CL") == false) return;
  if(parse_record->get_bool("PE") == true) return;
  int blmcell = clamp(parse_record->get_int("BLMCELL"),0,18);
  blm[blmcell][0].add_data(parse_record->get_float("RPM"));
  blm[blmcell][1].add_data(parse_record->get_float("MAP"));
  blm[blmcell][2].add_data(parse_record->get_int("LBLM"));
  blm[blmcell][3].add_data(parse_record->get_int("RBLM"));
}

//--------------------------------------------

void analysis::analyze_nb_afr() {
  //filtering
  if(parse_record->get_bool("CL") == false) return;
  if(parse_record->get_bool("PE") == true) return;
  if(parse_record->get_bool("BLM") == false) return;
  if(parse_record->get_int("BLMCELL") == 18) return;
  if(anl_filter_idleanddecel == true) {
    if(parse_record->get_int("BLMCELL") >= 16) return;
    if(parse_record->get_float("TPS") <= 1) return; // no tps
    if(parse_record->get_float("LBPW") <= 1) return; // no lbpw
    if(parse_record->get_float("RBPW") <= 1) return; // no rbpw
  }
  // get vars
  float current_blm;
  int lblm = parse_record->get_int("LBLM");
  int rblm = parse_record->get_int("RBLM");
  if(worst_case_blm == true) {
    if(lblm > rblm) { // left higher
      current_blm = lblm;
    } else { // right higher
      current_blm = rblm;
    }
  } else {
    current_blm = clamp(( lblm + rblm ) / 2,0,255);
  }
  float current_int = clamp(( parse_record->get_int("LINT") + parse_record->get_int("RINT") ) / 2,0,255);
  if(enhanced_accuracy == true) {
    if(current_int < 112 || current_int > 140) return;
  }
  float trim = current_blm;
  if(use_integrator == true) {
    trim += ( current_int - 128 );
    trim = clamp((float)trim,(float)0,(float)255);
  }
  int current_rpm = parse_record->get_int("RPM");
  int current_map = parse_record->get_int("MAP");
  // ve_trim
  int rpm_axis = select_cell(ve_rpm_list,current_rpm);
  int map_axis = select_cell(ve_map_list,current_map);
  ve_trim[rpm_axis][map_axis].add_data(trim);
  // maf
  int current_afgs = parse_record->get_int("MAF");
  if(current_afgs < 1) return; // no airflow
  int afgs_axis = select_cell(300,70,current_afgs);
  maf_trim[afgs_axis].add_data(trim);
}

void analysis::analyze_wb_afr() {
  //filtering
  if(parse_record->get_bool("PE") == true) return;
  if(wbcl == true) {
    if(parse_record->get_bool("CL") == false) return;
    if(parse_record->get_bool("BLM") == false) return;
  } else {
    if(parse_record->get_bool("CL") == true) return;
    if(parse_record->get_bool("BLM") == true) return;
  }
  if(parse_record->get_float("LBPW") < 0.01 || parse_record->get_float("RBPW") < 0.01) return; // dfco
  // get vars
  float current_afr = parse_record->get_float(config.get_wb_str());
  float current_target;
  current_target = parse_record->get_float("AFR");
  if(wbpercent == true) {
    if(current_target < 1) return; // no afr target data in percentage mode, err!
    current_afr = current_afr / current_target;
  }
  int current_rpm = parse_record->get_int("RPM");
  int current_map = parse_record->get_int("MAP");
  // ve_trim[14][10]
  int rpm_axis = select_cell(ve_rpm_list,current_rpm);
  int map_axis = select_cell(ve_map_list,current_map);
  ve_trim[rpm_axis][map_axis].add_data(current_afr);
  // maf
  int current_afgs = parse_record->get_int("MAF");
  if(current_afgs < 1) return; // no airflow
  int afgs_axis = select_cell(300,70,current_afgs);
  maf_trim[afgs_axis].add_data(current_afr);
}

//--------------------------------------------

int analysis::select_cell(int range, int cells, float data) {
  float interval = range / cells;
  int cell = floor((data/(float)range)*(float(range)/interval));
  if(cell > cells - 1) return cells - 1;
  if(cell < 0) return 0;
  return cell;
}

int analysis::select_cell(QVector<int> list, float data) {
  int last = 0;
  for(int x=0;x<list.size();x++) {
    if(data >= last && data <= list.at(x)) return x;
    last = list.at(x);
  }
  return list.size() - 1;
}

//--------------------------------------------

void analysis::analyze_knock() {
  datalog_packet *prev = parse_record->get_prev(parse_record->get_device(),parse_record->get_msgnumber());
  if(prev == nullptr) return; // no prev data
  int prev_knock = prev->get_int("KCOUNT");
  timestamp_t prev_timestamp = prev->timestamp;
  int current_knock = parse_record->get_int("KCOUNT");
  timestamp_t current_timestamp = parse_record->timestamp;
  if(current_knock > prev_knock &&
     current_timestamp - prev_timestamp < 500) { // reject large rec gap
    if(total_knock_events >= 2000) return; // bail if tons of records
    knockevent_t e;
    e.kr = parse_record->get_float("KR");
    e.spark = parse_record->get_int("SPKADV");
    e.wot = parse_record->get_bool("PE");
    e.rpm = parse_record->get_int("RPM");
    e.map = parse_record->get_int("MAP");
    e.interval = current_timestamp - prev_timestamp;
    e.timestamp = current_timestamp;
    e.count = current_knock - prev_knock;
    knock_list.append(e);
    total_knock_events++;
    total_knock_counts += ( current_knock - prev_knock );
  }
}


//--------------------------------------------

void analysis::analyze_o2() {
  // filtering
  if(parse_record->get_bool("CL") == false) return;
  if(parse_record->get_bool("PE") == true) return;
  if(parse_record->get_bool("BLM") == false) return;

  int lo2v = parse_record->get_int("LO2");
  int ro2v = parse_record->get_int("RO2");
  lo2.add_data(lo2v);
  ro2.add_data(ro2v);
  // crosscounts
  if(lo2v > 600) { // left o2 high
    if(lstate == 0) { // state was low
      lstate = 1; // set state high
      lcrosscount++; // increment cross count
    }
  } else if(lo2v < 400) { // left o2 low
    if(lstate == 1) { // state was high
      lstate = 0; // set state low
      lcrosscount++;
    }
  } else {
    lstuck++;
  }
  if(ro2v > 600) { // r o2 high
    if(rstate == 0) { // state was low
      rstate = 1; // set state high
      rcrosscount++; // increment cross count
    }
  } else if(ro2v < 400) { // r o2 low
    if(rstate == 1) { // state was high
      rstate = 0; // set state low
      rcrosscount++;
    }
  } else {
    rstuck++;
  }
}

//--------------------------------------------

void analyzer_ui::label_anl_graphs() {
  for(int x=0;x<70;x++) {
    ui->afr_maf_table->setItem(x,0,
          new QTableWidgetItem(QString("%2-%1").arg((x + 1) * 5).arg(((x + 1) * 5)-4)));
  }
  for(int x=0;x<28;x++) {
    ui->afr_pe_table->setItem(x,0,
          new QTableWidgetItem(QString("%2-%1").arg((x + 1) * 250).arg(((x + 1) * 250)-250)));
  }
}

void analyzer_ui::draw_analysis_results() {
  draw_results_o2();
  // knock is drawn live already ...
  ui->total_knock_counts->setText(QString("%1").arg(anl->total_knock_counts));
  ui->total_knock_events->setText(QString("%1").arg(anl->total_knock_events));
  if(anl->wideband == true) {
    draw_results_wb_afr();
  } else {
    draw_results_nb_afr();
  }
  draw_results_pe();
  draw_results_idle();
  draw_results_blm();
  draw_results_knock();
}

void analyzer_ui::draw_results_knock() {
  knock_scene->clear();

  int width = ui->anl_knockchart->width();
  int height = ui->anl_knockchart->height();

  draw_knock_grid(width,height);

  if(anl == nullptr) return;

  for(int x=0;x<anl->knock_list.size();x++) add_knock_point(width,height,anl->knock_list.at(x));
}

void analyzer_ui::draw_knock_grid(int width, int height) {
  float rpm_range = 8000;
  float map_range = 120;
  int n_map_cells = 12;
  int n_rpm_cells = 16;
  for(int x=1;x<n_map_cells;x++) { // draw vertical lines
    qreal xpos = ((float)width/n_map_cells)*(float)x;
    qreal h = height - 1;
    knock_scene->addLine(xpos,1,xpos,h);
    int current_map = ( map_range / n_map_cells ) * x;
    QGraphicsSimpleTextItem *label = knock_scene->addSimpleText(QString::number(current_map));
    label->setPos(xpos + 2,1);
  }
  for(int x=1;x<n_rpm_cells;x++) { // hz lines
    qreal ypos = ((float)height/n_rpm_cells)*(float)x;
    qreal w = width - 1;
    knock_scene->addLine(1,ypos,w,ypos);
    int current_rpm = ( rpm_range / n_rpm_cells ) * x;
    QGraphicsSimpleTextItem *label = knock_scene->addSimpleText(QString::number(current_rpm));
    label->setPos(2,ypos);
  }
}

void analyzer_ui::add_knock_point(int width, int height, knockevent_t e) {
  float rpm_range = 8000;
  float map_range = 120;
  float circle_size = clamp_d((float)e.count / (float)ANALYZER_KNOCK_COUNT_DIVISOR,
                            ANALYZER_KNOCK_MIN_CIRCLE * 2,
                            ANALYZER_KNOCK_MAX_CIRCLE * 2);
  float scale_x = width * ((float)e.map / map_range);
  float scale_y = height * ((float)e.rpm / rpm_range);
  qreal x = clamp_d(scale_x - (circle_size / 2),1,width - 1);
  qreal y = clamp_d(scale_y - (circle_size / 2),1,height - 1);
  qreal w = clamp_d(circle_size,1,width - circle_size - 1);
  qreal h = clamp_d(circle_size,1,height - circle_size - 1);
  QGraphicsEllipseItem *point = knock_scene->addEllipse(x,y,w,h);

  // color according to power enrichment status .
  if(e.wot == true) {
    point->setBrush(QColor(200,0,0,180));
  } else {
    point->setBrush(QColor(0,200,0,180));
  }

  // long tooltip ...
  point->setToolTip(QString("Knock Event Details\nIncrement: %1/msec\nRPM: %2\nMAP: %3 KPA\nSPARK: %4deg\nKR: %5\nTIMESTAMP: %6").
                arg(e.count/e.interval).arg(e.rpm).arg(e.map).arg(e.spark).arg(e.kr).arg((float)e.timestamp / 1000));
}

QString analyzer_ui::percent_to_str(double x) {
  if(config.get_low_precision() == true) {
    if(x < DISCARD_TRIM && x > -DISCARD_TRIM && config.get_low_percent_ok() == true) {
      return QStringLiteral(" 0%");
    }
    if(x > 0) return QString("+%1%").arg((double)x,0,'f',0);
    if(x < 0) return QString("%1%").arg((double)x,0,'f',0); // already has minus sign ...
    return QStringLiteral(" 0%");
  } else { // one decimal place
    if(x < DISCARD_TRIM && x > -DISCARD_TRIM && config.get_low_percent_ok() == true) {
      return QStringLiteral(" 0.0");
    }
    if(x > 0) return QString("+%1").arg((double)x,0,'f',1);
    if(x < 0) return QString("%1").arg((double)x,0,'f',1); // already has minus sign ...
    return QStringLiteral(" 0.0");
  }
}

float analyzer_ui::trim_to_percent(int trim) {
  return (((float)trim / 128.00 * 100.00)-100.00);
}

void analyzer_ui::draw_results_idle() {
  if(anl->idle_trim[0].count > anl->min_afrcounts) {
    if(ui->setting_display_blm_percent->isChecked() == true && anl->wideband == false) {
      ui->anl_idle_lh->setText(percent_to_str(trim_to_percent(anl->idle_trim[0].avg())));
    } else {
      ui->anl_idle_lh->setText(QString::number(anl->idle_trim[0].avg(),'f',1));
    }
  } else {
    ui->anl_idle_lh->setText("---");
  }
  if(anl->idle_trim[1].count > anl->min_afrcounts) {
    if(ui->setting_display_blm_percent->isChecked() == true && anl->wideband == false) {
      ui->anl_idle_rh->setText(percent_to_str(trim_to_percent(anl->idle_trim[1].avg())));
    } else {
      ui->anl_idle_rh->setText(QString::number(anl->idle_trim[1].avg(),'f',1));
    }
  } else {
    ui->anl_idle_rh->setText("---");
  }
  if(anl->idle_map.count > anl->min_afrcounts) {
    ui->anl_idle_map->setText(QString::number(anl->idle_map.avg(),'f',1));
  } else {
    ui->anl_idle_map->setText("---");
  }
  if(anl->idle_maf.count > anl->min_afrcounts) {
    ui->anl_idle_maf->setText(QString::number(anl->idle_maf.avg(),'f',1));
  } else {
    ui->anl_idle_maf->setText("---");
  }
}

void analyzer_ui::draw_results_nb_afr() {
  bool display_percentage = ui->setting_display_blm_percent->isChecked();
  for(int map_axis=0;map_axis<anl->ve_map_list.size();map_axis++) {
    for(int rpm_axis=0;rpm_axis<anl->ve_rpm_list.size();rpm_axis++) {
      float trim = anl->ve_trim[rpm_axis][map_axis].avg();
      float trim_percent = (((double)trim / 128.0 * 100.0)-100.0);
      if(anl->ve_trim[rpm_axis][map_axis].count >= anl->min_afrcounts) {
        if(display_percentage == true) {
          ui->afr_ve_table->setItem(rpm_axis,map_axis, new QTableWidgetItem(percent_to_str(trim_percent)));
        } else {
          ui->afr_ve_table->setItem(rpm_axis,map_axis, new QTableWidgetItem(QString::number(trim,'f',0)));
        }
        if(trim > 134) ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::green);
        if(trim < 122)  ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::red);
      } else {
        ui->afr_ve_table->setItem(rpm_axis,map_axis, new QTableWidgetItem(QString(" ")));
        ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::lightGray);
      }
    }
  }
  for(int maf_axis=0;maf_axis<70;maf_axis++) {
    float trim = anl->maf_trim[maf_axis].avg();
    float trim_percent = (((double)trim / 128.0 * 100.0)-100.0);
    if(anl->maf_trim[maf_axis].count >= anl->min_afrcounts) {
      if(display_percentage == true) {
        ui->afr_maf_table->setItem(maf_axis,1, new QTableWidgetItem(percent_to_str(trim_percent)));
      } else {
        ui->afr_maf_table->setItem(maf_axis,1,new QTableWidgetItem(QString::number(trim,'f',0)));
      }
      if(trim > 134) ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::green);
      if(trim < 122) ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::red);
    } else {
      ui->afr_maf_table->setItem(maf_axis,1, new QTableWidgetItem(QString(" ")));
      ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::darkGray);
    }

  }
}

void analyzer_ui::draw_results_wb_afr() {
  for(int map_axis=0;map_axis<anl->ve_map_list.size();map_axis++) {
    for(int rpm_axis=0;rpm_axis<anl->ve_rpm_list.size();rpm_axis++) {
      float afr = anl->ve_trim[rpm_axis][map_axis].avg();
      float percent = clamp((float)((anl->ve_trim[rpm_axis][map_axis].avg() * 100.00) - 100.00),-99.00,99.00);
      // draw VE
      if(anl->ve_trim[rpm_axis][map_axis].count >= anl->min_afrcounts) {
        if(anl->wbpercent == false) {
          ui->afr_ve_table->setItem(rpm_axis,map_axis, new QTableWidgetItem(QString::number((float)afr,'f',1)));
          if(afr > 15) {
            ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::green);
          } else if(afr < 12) {
            ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::red);
          }
        } else { // percentage mode
          ui->afr_ve_table->setItem(rpm_axis,map_axis, new QTableWidgetItem(percent_to_str(percent)));
          if(percent < -5) ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::red);
          if(percent > 5) ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::green);
        }
      } else {
        ui->afr_ve_table->setItem(rpm_axis,map_axis, new QTableWidgetItem(QString(" ")));
        ui->afr_ve_table->item(rpm_axis,map_axis)->setBackgroundColor(Qt::darkGray);
      }
    }
  }
  // draw MAF
  for(int maf_axis=0;maf_axis<70;maf_axis++) {
    float afr = anl->maf_trim[maf_axis].avg();
    float percent = clamp((float)((afr * 100.00) - 100.00),-99.00,99.00);
    if(anl->maf_trim[maf_axis].count >= anl->min_afrcounts) {
      if(anl->wbpercent == false) { // raw mode
        ui->afr_maf_table->setItem(maf_axis,1,new QTableWidgetItem(QString::number((float)afr,'f',1)));
        if(afr > 15) {
          ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::green);
        } else if(afr < 12) {
          ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::red);
        }
      } else { // percent mode
        ui->afr_maf_table->setItem(maf_axis,1, new QTableWidgetItem(percent_to_str(percent)));
        if(afr < -5) ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::red);
        if(afr > 5) ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::green);
      }
    } else {
      ui->afr_maf_table->setItem(maf_axis,1, new QTableWidgetItem(QString(" ")));
      ui->afr_maf_table->item(maf_axis,1)->setBackgroundColor(Qt::darkGray);
    }
  }
}

void analyzer_ui::draw_results_blm() {
  for(int x=0;x<19;x++) {
    // cell number and count
    ui->anl_blm_chart->setItem(x,0,
                               new QTableWidgetItem(QString("%1 (%2)").
                                                    arg(x).
                                                    arg(anl->blm[x][0].count)));
    for(int y=0;y<2;y++) { //rpm/map
      if(anl->blm[x][0].count < 1) {
        ui->anl_blm_chart->setItem(x,y + 1,new QTableWidgetItem(QString(" ")));
      } else {
       ui->anl_blm_chart->setItem(x,y + 1,
                                 new QTableWidgetItem(
                                   QString("%1 - %2").
                                   arg(anl->blm[x][y].low,0,'f',0).
                                   arg(anl->blm[x][y].high,0,'f',0)));
      }
    }
    // blm averages
    if(anl->blm[x][0].count < 1) { // not enough data
      ui->anl_blm_chart->setItem(x,3,new QTableWidgetItem(QString(" ")));
    } else {
      // lblm
      ui->anl_blm_chart->setItem(x,3,
                                new QTableWidgetItem(QString("%1").arg(anl->blm[x][2].avg(),0,'f',0)));
      // rblm
      ui->anl_blm_chart->setItem(x,4,
                                new QTableWidgetItem(QString("%1").arg(anl->blm[x][3].avg(),0,'f',0)));
    }
  }
}

void analyzer_ui::draw_results_pe() {
  for(int rpm_axis=0;rpm_axis<28;rpm_axis++) {
    for(int map_cell=0;map_cell<3;map_cell++) {
      float wbfuel = anl->pe_fuel[rpm_axis][map_cell].avg();
      if(config.get_wideband_enable() == false) { // use narrowband mv
        if(anl->pe_fuel[rpm_axis][map_cell].count < 3) { // not enough data
          ui->afr_pe_table->setItem(rpm_axis,map_cell + 1,new QTableWidgetItem(QString(" ")));
          ui->afr_pe_table->item(rpm_axis,map_cell + 1)->setBackgroundColor(Qt::darkGray);
          continue;
        } // else valid data
        ui->afr_pe_table->setItem(rpm_axis,map_cell + 1,
                                  new QTableWidgetItem(QString::number(wbfuel,'f',0)));
        if(wbfuel > 825) {
          ui->afr_pe_table->item(rpm_axis,map_cell + 1)->setBackgroundColor(Qt::green);
        } else if(wbfuel <= 825) {
          ui->afr_pe_table->item(rpm_axis,map_cell + 1)->setBackgroundColor(Qt::red);
        }
      } else { // wideband afr
        if(anl->pe_fuel[rpm_axis][map_cell].count < 4) { // not enough data
          ui->afr_pe_table->setItem(rpm_axis,map_cell + 1,new QTableWidgetItem(QString(" ")));
          ui->afr_pe_table->item(rpm_axis,map_cell + 1)->setBackgroundColor(Qt::darkGray);
          continue;
        } // else valid data
        ui->afr_pe_table->setItem(rpm_axis,map_cell + 1,new QTableWidgetItem(QString::number(wbfuel,'f',1) + ":1"));
          if(wbfuel > 14) ui->afr_pe_table->item(rpm_axis,map_cell + 1)->setBackgroundColor(Qt::green);
          if(wbfuel < 12) ui->afr_pe_table->item(rpm_axis,map_cell + 1)->setBackgroundColor(Qt::red);
      }
    }
  }
}


