#include <QApplication>
#include <QtDebug>

#include "datastream.h"
#include "stdlib.h"
#include "datalog.h"
#include "config.h"
#include "common.h"

void datastream_control::m4_comm_spark(int advance, int absolute) {
  m4_lock.lock();
  if(advance == 0) { /* no advance, disable */
    clrbit(construct_mode4[13],0); /* disable spark control */
    clrbit(construct_mode4[13],2); /* clear mode (redundant ...) */
    clrbit(construct_mode4[13],1); /* clear advret (redundant ...) */
    construct_mode4[14] = 0; /* clear adv */

  } else { /* enable advance */
    setbit(construct_mode4[13],0); /* set spark control enable bit */

    /* WARNING absolute mode is kinda fucking dangerous, dont use it */
    if(absolute == 1) { /* configure for absolute spark */
      clrbit(construct_mode4[13],2); /* abs mode */
      construct_mode4[14] = clamp(advance,0,70); /* set advance */
      /* don't bother with retard/advance, abs negative spark is useless */

    } else { /* configure for delta spark */
      setbit(construct_mode4[13],2); /* delta mode */
      if(advance > 0) {
        clrbit(construct_mode4[13],1); /* advance */
        construct_mode4[14] = clamp(advance,0,70);
      } else {
        setbit(construct_mode4[13],1); /* retard */
        construct_mode4[14] = clamp(( 1 - advance ),0,70);
      }
    }
  }
  m4_updated.request();
  m4_lock.unlock();
}

int datastream_control::m4_get_spark() {
  m4_lock.lock();
  int adv = 0;
  if(getbit(0,construct_mode4[13]) == false) { // disabled
    m4_lock.unlock();
    return 0;
  }
  if(getbit(2,construct_mode4[13]) == false) { // absmode
    m4_lock.unlock();
    return 0;
  }
  if(getbit(1,construct_mode4[13]) == false) { // advance
    adv = construct_mode4[14];
  } else { // retard
    adv = 1 - (int)construct_mode4[14];
  }
  m4_lock.unlock();
  return adv;
}

void datastream_control::m4_comm_idle(int rpm, int mode) {
  m4_lock.lock();
  if(rpm == 0) { /* disable idle command */
    clrbit(construct_mode4[9],4); /* clear iac enable */
    clrbit(construct_mode4[9],5); /* clear steps/rpm */
    construct_mode4[11] = 0x00;
  } else if(mode == 1) { /* RPM mode */
    setbit(construct_mode4[9],4); /* set iac enable */
    setbit(construct_mode4[9],5); /* set rpm mode */
    construct_mode4[11] = rpm * 0.08; /* set rpm */
  } else { /* steps mode */
    setbit(construct_mode4[9],4); /* set iac enable */
    clrbit(construct_mode4[9],5); /* set rpm mode */
    construct_mode4[11] = rpm;
  }
  m4_updated.request();
  m4_lock.unlock();
}

void datastream_control::m4_drop_cyl(int cyl) {
  m4_lock.lock();
  if(cyl == 0) { /* cyl 0 = disable */
    clrbit(construct_mode4[9],7); /* disable cut */
    construct_mode4[12] = 0x00; /* disable cyl */
  } else {
    clrbit(construct_mode4[9],6); // disable afr
    construct_mode4[12] =  cylnum_to_m4ref(cyl);
    setbit(construct_mode4[9],7); /* enable cut */
  }
  m4_updated.request();
  m4_lock.unlock();
}

byte datastream_control::cylnum_to_m4ref(int cyl) {
  int clamped_cyl = clamp(cyl,1,8); // clamp input to 1-8
  static byte cyl_conv[8] = { 0, // 1
                              7, // 2
                              3, // 3
                              2, // 4
                              5, // 5
                              4, // 6
                              6, // 7
                              1 }; // 8
  byte cyl_cancel = cyl_conv[clamped_cyl - 1];
  return cyl_cancel;
}

void datastream_control::m4_comm_afr(byte afr) {
  m4_lock.lock();
  if(afr > 0) {
    clrbit(construct_mode4[9],7); // disable cyl drop
    setbit(construct_mode4[9],6);
    construct_mode4[12] = m4_process_afr(afr);
  } else {
    clrbit(construct_mode4[9],6);
    construct_mode4[12] = 0x00;
  }
  m4_updated.request();
  m4_lock.unlock();
}

void datastream_control::m4_comm_linepressure(bool enable, byte pressure) {
  m4_lock.lock();
  if(enable == true) {
    setbit(construct_mode4[9],0);
    construct_mode4[10] = pressure;
  } else {
    clrbit(construct_mode4[9],0);
    construct_mode4[10] = 0x00;
  }
  m4_updated.request();
  m4_lock.unlock();
}

void datastream_control::m4_comm_egr(bool enable, byte pressure) {
  m4_lock.lock();
  if(enable == true) {
    setbit(construct_mode4[9],2);
    construct_mode4[10] = pressure;
  } else {
    clrbit(construct_mode4[9],2);
    construct_mode4[10] = 0x00;
  }
  m4_updated.request();
  m4_lock.unlock();
}

void datastream_control::m4_reset_blm(bool set = true) {
  m4_lock.lock();
  if(set == true) {
    setbit(construct_mode4[3],5);
  } else {
    clrbit(construct_mode4[3],5);
  }
  m4_updated.request();
  m4_lock.unlock();
}

void datastream_control::m4_clear_blm_reset() {
  clrbit(construct_mode4[3],5);
}

void datastream_control::m4_force_gear(int gear) {
  m4_lock.lock();
  if(gear == 0) { // disable
    clrbit(construct_mode4[7],2);
    clrbit(construct_mode4[7],3);
    clrbit(construct_mode4[8],2);
    clrbit(construct_mode4[8],3);
  } else if(gear >= 1 && gear <= 4) {
    setbit(construct_mode4[7],2);
    setbit(construct_mode4[7],3);
    switch(gear) {
    case 1:
      setbit(construct_mode4[8],2);
      setbit(construct_mode4[8],3);
      break;
    case 2:
      clrbit(construct_mode4[8],2);
      setbit(construct_mode4[8],3);
      break;
    case 4:
      setbit(construct_mode4[8],2);
      clrbit(construct_mode4[8],3);
      break;
    case 3:
      clrbit(construct_mode4[8],2);
      clrbit(construct_mode4[8],3);
      break;
    }
  }
  m4_updated.request();
  m4_lock.unlock();
}

void datastream_control::m4_actuator(int enable_byte, int enable_bit,
                             int actuator_byte, int actuator_bit,
                             bool set, bool enable = true) {
  m4_lock.lock();
  if(enable == false) { // auto
    clrbit(construct_mode4[enable_byte],enable_bit);
    clrbit(construct_mode4[actuator_byte],actuator_bit);
  } else if(set == true) { // force on
    setbit(construct_mode4[enable_byte],enable_bit);
    setbit(construct_mode4[actuator_byte],actuator_bit);
  } else { // force off
    setbit(construct_mode4[enable_byte],enable_bit);
    clrbit(construct_mode4[actuator_byte],actuator_bit);
  }
  m4_updated.request();
  m4_lock.unlock();
}

void datastream_control::m4_mil(bool set) {
  m4_actuator(5,2,6,2,set,set);
}

void datastream_control::m4_force_cl(bool set, bool enable = true) {
  m4_actuator(7,6,8,6,set,enable);
}

void datastream_control::m4_fan(int n, bool set, bool enable) {
  int i = 2; // fan 1
  if(n == 2) i = 3; // fan 2
  m4_actuator(3,i,4,i,set,enable);
}

void datastream_control::m4_ac(bool set, bool enable = true) {
  m4_actuator(3,0,4,0,set,enable);
}

void datastream_control::m4_tcc(bool set, bool enable = true) {
  m4_actuator(7,0,8,0,set,enable);
  m4_actuator(7,1,8,1,set,enable);
}

void datastream_control::m4_air(bool set, bool enable = true) {
  m4_actuator(3,1,4,1,set,enable);
}

void datastream_control::m4_ccp(bool set, bool enable = true) {
  m4_actuator(3,4,4,4,set,enable);
}

void datastream_control::m4_force_blm(bool set, bool enable = true) {
  m4_actuator(7,5,8,5,set,enable);
}

byte datastream_control::m4_process_afr(int afr) {
  return afr;
}

void datastream_control::m4_comm_init() {
  m4_lock.lock();
  /* zero entire string */
  int x;
  for(x=3;x<16;x++) construct_mode4[x] = 0x00;
  /* add prefix */
  construct_mode4[BDEVICE] = PRIMARY_ECM; /* ecm address */
  construct_mode4[BMODE] = 0x04; /* mode byte */
  m4_updated.request();
  m4_lock.unlock();
}

QString datastream_control::m4_getpkt() {
  m4_lock.lock();
  QByteArray t;
  t.setRawData((char *)construct_mode4 +3,12);
  QString out = t.toHex().toUpper();
  m4_lock.unlock();
  return out;
}

bool datastream_control::m4_get_request(byte *buf) {
  m4_lock.lock();
  if(m4_updated.getclear() == true) {
    for(int x=0;x<16;x++) buf[x] = construct_mode4[x];
    m4_clear_blm_reset();
    m4_lock.unlock();
    return true;
  } else {
    m4_lock.unlock();
    return false;
  }
}

void datastream_control::m4_get_raw(byte *buf) {
  m4_lock.lock();
  for(int x=0;x<16;x++) buf[x] = construct_mode4[x];
  m4_lock.unlock();
}

void datastream_control::m4_set_raw(byte *buf) {
  m4_lock.lock();
  for(int x=0;x<16;x++) construct_mode4[x] = buf[x];
  m4_updated.request();
  m4_lock.unlock();
}
