#include <QDebug>
#include "datalog_definition.h"
#include "common.h"

bool datalog_definition::is_empty() {
  if(element == nullptr) return true;
  return false;
}

datalog_element *datalog_definition::attach_element(datalog_element *e) {
  e->next = nullptr;
  if(is_empty() == true) { // first element
    element = e;
  } else {
    datalog_element *last = get_last_element();
    last->next = e;
  }
  return e;
}

datalog_element *datalog_definition::get_first_element() {
  return element;
}

datalog_definition::datalog_definition() {
  element = nullptr;
  next_def = nullptr;
  n_passes = 0;
}

datalog_element *datalog_definition::get_last_element() {
  datalog_element *x = element;
  while(x->next != nullptr) x = x->next;
  return x;
}

void datalog_definition::parsing_error(QString) {
  //emit fatal_error("Definition Parser Error!\n" + str + "at line" + QString::number(line));
}

bool datalog_definition::load_definition_line(QString line_in) {
  if(line_in.isNull()) return false; // EOF
  //------------
  QStringList field_in = line_in.split(",");
  if(field_in.count() < 1) return true;
  if(field_in.at(0).startsWith("#") == true) return true;
  if(field_in.count() != DEF_FORMAT_NFIELDS) {
    parseerr = "Wrong number of fields";
    return false;
  }
  bool res;

  // --- offset
  int offset = field_in.at(DEF_FORMAT_FIELD_OFFSET).toInt(&res,0);
  if(offset < 0x00 || offset >= 0xFF || res == false) {
    parseerr = "Offset field error";
    return false;
  }
  // FIXME should check msg length too ....

  datalog_element *new_element = new datalog_element(offset,this); // new element (remember to delete)

  // --- short name
  QString short_name_in = field_in.at(DEF_FORMAT_FIELD_SHORTNAME);
  if(short_name_in.size() < 2 || short_name_in.size() >= MAX_SHORTNAME_LENGTH) {
    delete new_element;
    parseerr = "Short name too long or short";
    return false;
  }

  strcpy(new_element->short_name, short_name_in.toStdString().c_str());

  // --- long name
  new_element->long_name = field_in.at(DEF_FORMAT_FIELD_LONGNAME);
  if(new_element->long_name.size() < 1) {
    new_element->long_name = new_element->short_name;
  } else if(new_element->long_name.size() >= 0xFF) {
      parseerr = "Long name is too long";
      delete new_element;
      return false;
  }

  // --- UOM
  new_element->uom = field_in.at(DEF_FORMAT_FIELD_UOM);
  if(new_element->uom.size() < 1) {
    new_element->uom.clear(); // no UOM
  }

  // --- LEVEL
  if(field_in.at(DEF_FORMAT_FIELD_VIEWLEVEL).toUpper() == "HACKER") {
    new_element->level = datalog_element::VIEW_HACKER;
  } else if(field_in.at(DEF_FORMAT_FIELD_VIEWLEVEL).toUpper() == "EXTENDED") {
    new_element->level = datalog_element::VIEW_EXTENDED;
  } else if(field_in.at(DEF_FORMAT_FIELD_VIEWLEVEL).toUpper() == "BASIC") {
    new_element->level = datalog_element::VIEW_BASIC;
  } else if(field_in.at(DEF_FORMAT_FIELD_VIEWLEVEL).toUpper() == "ERROR") {
    new_element->level = datalog_element::VIEW_ERROR;
  } else if(field_in.at(DEF_FORMAT_FIELD_VIEWLEVEL).toUpper() == "VERROR") {
    new_element->level = datalog_element::VIEW_VERROR;
  } else if(field_in.at(DEF_FORMAT_FIELD_VIEWLEVEL).toUpper() == "HIDDEN") {
    new_element->level = datalog_element::VIEW_HIDDEN;
  } else {
    parseerr = "View level error (Requires HACKER/EXTENDED/BASIC/ERROR/VERROR/HIDDEN)";
    delete new_element;
    return false;
  }

  // --- SIZE
  new_element->size = field_in.at(DEF_FORMAT_FIELD_SIZE).toInt(&res,0);
  if(new_element->size < 0 || ( new_element->size > 1 && new_element->size % 8 != 0) || res == false) {
    parseerr = "Size field error, must either be 1 or a multiple of 8";
    delete new_element;
    return false;
  }

  // --- PRECISION
  new_element->precision = field_in.at(DEF_FORMAT_FIELD_PRECISION).toInt(&res,0);
  if((new_element->size == 1 && (new_element->precision < 0 || new_element->precision > 7)) || res == false) {
    parseerr = "Precision field error";
    delete new_element;
    return false;
  }

  // --- MINIMUM PATCH
  if(field_in.at(DEF_FORMAT_FIELD_MINPATCH).size() < 1) {
    new_element->min_patch = 0x00;
  } else {
    new_element->min_patch = field_in.at(DEF_FORMAT_FIELD_MINPATCH).toInt(&res,0);
    if(res == false) {
      parseerr = "Minpatch field error";
      delete new_element;
      return false;
    }
  }

  // --- MAXIMUM PATCH
  if(field_in.at(DEF_FORMAT_FIELD_MAXPATCH).size() < 1) {
    new_element->max_patch = 0xff;
  } else {
    new_element->max_patch = field_in.at(DEF_FORMAT_FIELD_MAXPATCH).toInt(&res,0);
    if(res == false) {
      parseerr = "Maxpatch field error";
      delete new_element;
      return false;
    }
  }

  // --- MULTIPLIER
  if(field_in.at(DEF_FORMAT_FIELD_MULT).size() < 1) {
    new_element->eq_mult = 1.00;
  } else {
    new_element->eq_mult = field_in.at(DEF_FORMAT_FIELD_MULT).toDouble(&res);
    if(res == false) {
      parseerr = "Multiplier field error";
      delete new_element;
      return false;
    }
    if(new_element->eq_mult == 0) new_element->eq_mult = 1; // zero multiplier is pointless
  }

  // --- ADDER
  if(field_in.at(DEF_FORMAT_FIELD_ADD).size() < 1) {
    new_element->eq_add = 0.00;
  } else {
    new_element->eq_add = field_in.at(DEF_FORMAT_FIELD_ADD).toDouble(&res);
    if(res == false) {
      parseerr = "Addition field error";
      delete new_element;
      return false;
    }
  }

  // --- SPECIALIZED FIELDS
  if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).size() < 1) {
    new_element->special = DEF_SPECIAL_NONE;
  } else {
    if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "TRIM") {
      new_element->special = DEF_SPECIAL_TRIM;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "CELCIUS") {
      new_element->special = DEF_SPECIAL_CELCIUS;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "AVG") {
      new_element->special = DEF_SPECIAL_AVERAGE;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "ASCII") {
      new_element->special = DEF_SPECIAL_ASCII;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "MPH") {
      new_element->special = DEF_SPECIAL_MPH;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "KPA") {
      new_element->special = DEF_SPECIAL_KPA;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "RPMMINOR") {
      new_element->special = DEF_SPECIAL_RPMMINOR;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "SHIFTPOS") {
      new_element->special = DEF_SPECIAL_SHIFTPOS;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "WIDEBAND") {
      new_element->special = DEF_SPECIAL_WIDEBAND;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "SIGNED") {
      new_element->special = DEF_SPECIAL_SIGNED;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "RESEARCH") {
      new_element->special = DEF_SPECIAL_RESEARCH;
    } else if(field_in.at(DEF_FORMAT_FIELD_SPECIAL).toUpper() == "ECONOMY") {
      new_element->special = DEF_SPECIAL_ECONOMY;
    } else {
      parseerr = "Special field invalid specifier";
      delete new_element;
      return false;
    }
  }

  if(field_in.at(DEF_FORMAT_FIELD_WARNLOW).size() > 0) {
    new_element->warn_low = field_in.at(DEF_FORMAT_FIELD_WARNLOW).toDouble(&res);
    if(res == false) {
      parseerr = "Low threshold error";
      delete new_element;
      return false;
    }
  }

  if(field_in.at(DEF_FORMAT_FIELD_WARNHIGH).size() > 0) {
    new_element->warn_high = field_in.at(DEF_FORMAT_FIELD_WARNHIGH).toDouble(&res);
    if(res == false) {
      parseerr = "High threshold error";
      delete new_element;
      return false;
    }
  }

  if(field_in.at(DEF_FORMAT_FIELD_RANGEHIGH).size() > 0) {
    new_element->range_max = field_in.at(DEF_FORMAT_FIELD_RANGEHIGH).toDouble(&res);
    if(res == false) {
      parseerr = "High range error";
      delete new_element;
      return false;
    }
  } else {
    new_element->calc_high_range();
  }

  if(field_in.at(DEF_FORMAT_FIELD_RANGELOW).size() > 0) {
    new_element->range_min = field_in.at(DEF_FORMAT_FIELD_RANGELOW).toDouble(&res);
    if(res == false) {
      parseerr = "Low range error";
      delete new_element;
      return false;
    }
  } else {
    new_element->calc_low_range();
  }

  // -- DEF OK
  attach_element(new_element);
  return true;
}

bool datalog_definition::setup(QString line_in) {
  if(line_in.isNull()) return false; // EOF
  //------------
  QStringList field_in = line_in.split(",");
  if(field_in.count() < 1) return true;
  if(field_in.at(0).startsWith("#") == true) return true;
  if(field_in.count() < DEF_HEADER_FIELD_NFIELDS) {
    parseerr = "Wrong number of fields";
    return false;
  }
  bool res;

  // --- device
  device_addr = field_in.at(DEF_HEADER_FIELD_DEVICE).toInt(&res,0);
  if(res == false) {
    parseerr = "Device field error";
    return false;
  }

  // --- msg no
  msg_number = field_in.at(DEF_HEADER_FIELD_MSGNUMBER).toInt(&res,0);
  if(res == false) {
    parseerr = "Message number error";
    return false;
  }

  // --- name
  QString short_name_in = field_in.at(DEF_HEADER_FIELD_COMMENT);
  if(short_name_in.size() > 1) {
     msg_desc = short_name_in;
  }

  return true;
}

datalog_element *datalog_definition::element_by_index(int i) {
  datalog_element *x = element;
  int y = 0;
  while(y < i) {
    x = x->next;
    if(x == nullptr) return nullptr; // end of list
    y++;
  }
  return x;
}

datalog_element *datalog_definition::element_by_name(const char *name) {
  if(name == nullptr) return nullptr;
  datalog_element *x = element;
  while(x != nullptr) {
    if(strings_are_equal(name,x->short_name) == true) return x;
    x = x->next;
  }
  qDebug() << "Can't find element: " << name << "in definition" << get_msg_number();
  return nullptr;
}

bool datalog_definition::element_exists(const char *name) {
  if(element_by_name(name) == nullptr) return false;
  return true;
}

byte datalog_definition::get_msg_number() {
  return msg_number;
}

byte datalog_definition::get_device() {
  return device_addr;
}

//---------

byte datalog_definition::any_acq_enabled() {
  datalog_definition *x = this;
  do {
    if(x->acq.get() == true && x->is_empty() == false) return true;
    x = x->next_def;
  } while(x != this);
  return false;
}

datalog_definition *datalog_definition::next_acq_def() {
  datalog_definition *x = next_def;
  do {
    if(x->acq.get() == true && x->is_empty() == false) {
      if(x->n_passes >= x->frequency.get()) {
        x->n_passes = 0;
        return x;
      } else {
        x->n_passes++; // skip
      }
    }
    x = x->next_def;
  } while(x != next_def);
  return nullptr;
}

datalog_definition *datalog_definition::get_msg_number(byte device, byte msg) {
  datalog_definition *x = next_def;
  do {
    if(x->get_device() == device && x->get_msg_number() == msg) return x;
    x = x->next_def;
  } while(x != next_def);
  return nullptr;
}

//---------

bool datalog_definition::err_elements_exist(byte patchlvl) {
  if(n_errors(patchlvl) == 0) return false;
  return true;
}

bool datalog_definition::errors_occurred(datalog_packet *p, byte patchlvl) {
  if(p == nullptr) return false;
  datalog_element *x = element;
  while(x != nullptr) {
    if(x->matches_patchlvl(patchlvl) == true) {
      if(x->level == datalog_element::VIEW_ERROR && x->get_bool(p) == true) return true;
      if(config.get_vehicle_type() == VEHICLE_Y &&
         x->level == datalog_element::VIEW_VERROR &&
         x->get_bool(p) == true) return true;
    }
    x = x->next;
  }
  return false;
}

QString datalog_definition::err_str(datalog_packet *p, byte patchlvl) {
  if(p == nullptr) return "No Data";
  datalog_element *x = element;
  QString err_out;
  while(x != nullptr) {
    if(x->matches_patchlvl(patchlvl) == true) {
      if(x->level == datalog_element::VIEW_ERROR ||
         (config.get_vehicle_type() == VEHICLE_Y &&
          x->level == datalog_element::VIEW_VERROR)) {
        if(x->get_bool(p) == true) { // error active
          err_out += QString(x->short_name);
          err_out += "(";
          err_out += x->long_name;
          err_out += ")\n";
        }
      }
    }
    x = x->next;
  }
  if(err_out.isEmpty() == true) {
    return "No Errors";
  } else {
    return err_out;
  }
}

unsigned int datalog_definition::n_errors(byte patchlvl) {
  datalog_element *x = element;
  unsigned int err_count = 0;
  while(x != nullptr) {
    if(x->matches_patchlvl(patchlvl) == true) {
      if(x->level == datalog_element::VIEW_ERROR || x->level == datalog_element::VIEW_VERROR) {
        err_count++;
      }
    }
    x = x->next;
  }
  return err_count;
}

//---------

void datalog_definition::config_wideband(float adder, float multiplier, bool enable) {
  if(is_empty() == true) return;
  datalog_element *x = element;
  while(x != nullptr) {
    if(x->special == DEF_SPECIAL_WIDEBAND) {
      x->eq_mult = multiplier * MULT_LINEAR_5V;
      x->eq_add = adder;
      if(enable == true) {
        x->level = datalog_element::VIEW_BASIC;
      } else {
        x->level = datalog_element::VIEW_HIDDEN;
      }
    }
    x = x->next;
  }
}

//---------

