#include <QMainWindow>
#include <QMessageBox>
#include <QFileDialog>
#include <QProgressDialog>
#include <QDesktopServices>
#include <QCloseEvent>
#include <QDebug>
#include <QDesktopWidget>

#include "datalog_window.h"
#include "config.h"
#include "ui_autospark.h"
#include "debuglog_window.h"
#include "bin_file.h"
#include "knock_warning.h"
#include "about_window.h"
#include "mainwindow.h"
#include "graphing.h"
#include "settings.h"

MainWindow::MainWindow(QWidget *parent) :
  QMainWindow (parent),
  ui(new Ui::MainWindow) {
  loading_complete = false;
  ui->setupUi(this);
  //resize(QDesktopWidget().availableGeometry(this).size() * 0.75);
  //----- VARIABLES
  error_count = 0;
}

void MainWindow::on_interface_loaded() {
   set_loading_status("Loading...");

  //----- INSTANCES OF WINDOWS AND CTRL STRUCTURES
  def = new datalog_definition[MAX_MODE1_MESSAGES];
  debugwin = new debuglog_window;
  log = new datalog(RECORDED_DATA_LOGNAME,nullptr,nullptr); // initial datalog, for recording
  control = new datastream_control();
  about_win = new about_window;
  settings_window = new settings_editor(control,def);
  mode4controller = new controller(control,log,0);
  notes = new notepad;
  datalog_win = new datalog_window(control,log,def);
  test_window = new autotests(control);
  raw_command_window = new raw_command(control);
  analyzer_win = new analyzer_ui(log);
  update = new update_notify(this);
  memio_win = new memory_ui(0,control);

  control->def = def; // cp definition

  // The order of these modules can be important......

  //---- SIGNALS FROM MODE4 CONTROLLER
  connect(mode4controller,SIGNAL(send_new_note(QString)),notes,SLOT(add_new_note(QString)));

  //---- SIGNALS FROM LAUNCHER (THIS WINDOW)
  connect(this,SIGNAL(enable_port_selection(bool)),settings_window,SLOT(enable_port_selection(bool)));
  connect(this,SIGNAL(recheck_version()),update,SLOT(recheck_version()));

  //---- SIGNALS FROM DATALOG WINDOW
  connect(datalog_win,SIGNAL(add_new_note(QString)),notes,SLOT(add_new_note(QString)));
  connect(datalog_win,SIGNAL(display_info_dialog(QString)),this,SLOT(recieve_info_msg(QString)));
  connect(datalog_win,SIGNAL(display_error_dialog(QString)),this,SLOT(recieve_critical_error_msg(QString)));

  //---- SIGNALS FROM SETTINGS WINDOW
  connect(settings_window,SIGNAL(update_recheck_requested()),update,SLOT(recheck_version()));
  connect(settings_window,SIGNAL(display_config_changed()),datalog_win,SLOT(display_config_changed()));
  connect(settings_window,SIGNAL(display_config_changed()),analyzer_win,SLOT(wideband_config_updated()));

  //---- SIGNALS FROM TESTS WINDOW
  connect(test_window,SIGNAL(info_dialog(QString)),this,SLOT(recieve_info_msg(QString)));
  connect(test_window,SIGNAL(error_dialog(QString)),this,SLOT(recieve_critical_error_msg(QString)));
  connect(test_window,SIGNAL(send_new_note(QString)),notes,SLOT(add_new_note(QString)));

  //---- SIGNALS FROM ANALYZER
  connect(analyzer_win,SIGNAL(info_dialog(QString)),this,SLOT(recieve_info_msg(QString)));
  connect(analyzer_win,SIGNAL(error_dialog(QString)),this,SLOT(recieve_critical_error_msg(QString)));

  //---- SIGNALS FROM VERSION CHECKER
  connect(update,SIGNAL(update_network_status(QString)),settings_window,SLOT(get_update_status(QString)));
  connect(update,SIGNAL(new_version_recieved(int)),this,SLOT(check_version(int)));

  //---- LOAD DEFINITION (failure here is fatal)
  load_datastream_defs();

  //----- ON_INTERFACE_LOADED STUFF
  settings_window->on_interface_loaded();
  raw_command_window->on_interface_loaded();
  datalog_win->on_interface_loaded();
  analyzer_win->on_interface_loaded();
  mode4controller->on_interface_loaded();

  //---- CHECK FOR NEW VERSION
  ui->versionDisplay->setText(current_version_string());
  eehack_settings config;
  if(config.get_check_update() == true) emit recheck_version();

  //---- INITIAL DEBUG INFO
  init_debug_info();

  // this can happen last, as by design, nothing ever signals the acqthread
  prepare_acqthread();

  // add datalog to main window
  ui->frame_layout->addWidget(datalog_win);

  // enable interface
  loading_complete = true;

  // welcome screen
  display_welcome_screen();
}

void MainWindow::display_welcome_screen() {
  eehack_settings config;
  if(config.get_first_run() == true) { // first time run
    QMessageBox::about(this,"Welcome to EEHack",WELCOME_STR);
    on_btn_config_clicked(); // show settings window
  }
}

void MainWindow::set_loading_status(QString str) {
  ui->display_connstatus->setText(str);
  QApplication::processEvents();
}

void MainWindow::prepare_acqthread() {
  if(control == nullptr || log->get_first() == nullptr) {
    bail_error("Can't start ACQ thread due to out-of-order operations!");
  }

  // create object
  acqthread = new datastream(control,log->get_first(),def); // set up acq object

  //----- SIGNALS FROM ACQTHREAD
  connect(acqthread,SIGNAL(new_packet_ready(datalog_packet*)),mode4controller,SLOT(recieve_new_packet(datalog_packet*)));
  connect(acqthread,SIGNAL(display_critical_error(QString)),this,SLOT(recieve_critical_error_msg(QString)));
  connect(acqthread,SIGNAL(new_packet_ready(datalog_packet*)),datalog_win,SLOT(recieve_new_packet(datalog_packet*)));
  connect(acqthread,SIGNAL(update_pktrate(float)),datalog_win,SLOT(recieve_new_pktrate(float)));
  connect(acqthread,SIGNAL(connection_state_changed(int)),datalog_win,SLOT(recieve_connection_state(int)));
  connect(acqthread,SIGNAL(updated_ecm_info()),datalog_win,SLOT(recieve_new_ecm_info()));
  connect(acqthread,SIGNAL(connection_state_changed(int)),this,SLOT(recieve_connection_state(int)));
  connect(acqthread,SIGNAL(error_occurred()),this,SLOT(add_fail_count()));
  connect(acqthread,SIGNAL(invalid_serial_port()),this,SLOT(recieve_invalid_serial_port()));
  connect(acqthread,SIGNAL(new_cylbalance_result(int,int,float)),test_window,SLOT(display_cylbalance_result(int,int,float)));
  connect(acqthread,SIGNAL(new_cylbalance_progress(QString,int)),test_window,SLOT(display_cylbalance_progress(QString,int)));
  connect(acqthread,SIGNAL(custom_log_append(QString)),raw_command_window,SLOT(new_custom_log_entry(QString)));
  connect(acqthread,SIGNAL(connection_state_changed(int)),raw_command_window,SLOT(recieve_connection_state(int)));
  connect(acqthread,SIGNAL(acq_event_display(QString)),this,SLOT(display_acqstatus(QString)));
  connect(acqthread,SIGNAL(reset_error_count()),this,SLOT(reset_error_count()));
  connect(acqthread,SIGNAL(custom_a_copied()),raw_command_window,SLOT(custom_a_copied()));
  connect(acqthread,SIGNAL(custom_b_copied()),raw_command_window,SLOT(custom_b_copied()));
  connect(acqthread,SIGNAL(update_blm_cells(QByteArray)),memio_win,SLOT(display_blm_cells(QByteArray)));
  connect(acqthread,SIGNAL(connection_state_changed(int)),memio_win,SLOT(connection_state_changed(int)));

  //---- LAUNCH ACQ THREAD
  QTimer::singleShot(0,this,SLOT(run_acq_thread()));
}

void MainWindow::run_acq_thread() {
  acqthread->start(); // start acq persistent thread
}

void MainWindow::closeEvent(QCloseEvent *x) {
  eehack_settings s;
  if(s.get_warn_notes() == true && notes->handle_unsaved_data() == false) { // 'cancel' clicked, don't exit!
    x->ignore();
    return;
  }
  if(s.get_warn_log() == true && datalog_win->handle_unsaved_data() == false) { // 'cancel' clicked, don't exit!
    x->ignore();
    return;
  }
  // we need to do things in a certain order since some things may still be active.

  emit close_graphs();
  kill_acqthread(); // this stops new datalog events from being pushed to the datalog window.
  delete datalog_win; // then it's okay to kill the datalog window.
  // the rest probably aren't important to kill in any order ...
  delete settings_window;
  delete mode4controller;
  delete notes;
  delete debugwin;
  delete test_window;
  delete raw_command_window;
  //delete ui;
  x->accept();
}

MainWindow::~MainWindow() {
  delete ui;
  // this doesn't really get called.  closeevent should be called instead.
}

void MainWindow::display_acqstatus(QString str) {
  ui->display_acqevent->setText(str);
}

void MainWindow::on_connectButton_clicked() {
  if(control->is_connected() == true || control->connection_state.get() == STATE_CONNECTING) {
    control->connection.set(false);
    return;
  } else if(control->connection_state.get() == STATE_DISCONNECTING) {
    return;
  } else { // connect
    control->connection.set(true);
    return;
  }
}

void MainWindow::recieve_invalid_serial_port() {
  settings_window->show();
}

void MainWindow::recieve_connection_state(int c) {
  switch(c) {
  case STATE_CONNECTED:
    set_connstatus(QStringLiteral("Connected"));
    ui->connectButton->setText("Disconnect");
    ui->connectButton->setEnabled(true);
    emit enable_port_selection(false);
    break;
  case STATE_CONNECTING:
    set_connstatus(QStringLiteral("Connecting..."));
    ui->connectButton->setText("Disconnect");
    ui->connectButton->setEnabled(true);
    emit enable_port_selection(false);
    break;
  case STATE_DISCONNECTING:
    set_connstatus(QStringLiteral("Disconnecting..."));
    ui->connectButton->setText("Connect");
    ui->connectButton->setEnabled(false);
    emit enable_port_selection(false);
    break;
  case STATE_DISCONNECTED:
    set_connstatus(QStringLiteral("Disconnected."));
    ui->connectButton->setText("Connect");
    ui->connectButton->setEnabled(true);
    emit enable_port_selection(true);
    break;
  }
}

void MainWindow::add_fail_count() {
  error_count++;
  display_error_count();
}

void MainWindow::reset_error_count() {
  error_count = 0;
  display_error_count();
}

void MainWindow::display_error_count() {
  ui->display_errors->setText(QString::number(error_count));
}

void MainWindow::set_connstatus(QString status) {
  ui->display_connstatus->setText(status);
}

void MainWindow::recieve_critical_error_msg(QString msg) {
  error_dialog(msg);
}

void MainWindow::recieve_fatal_error_msg(QString msg) {
  error_dialog(msg);
  exit(1);
}

void MainWindow::recieve_info_msg(QString msg) {
  info_dialog(msg);
}

void MainWindow::bail_error(QString msg) {
  error_dialog(msg);
  exit(1);
}

void MainWindow::error_dialog(QString msg) {
  QMessageBox msgBox;
  msgBox.setText("EEHack: An error has occurred!");
  msgBox.setInformativeText(msg);
  msgBox.setIcon(QMessageBox::Critical);
  msgBox.setStandardButtons(QMessageBox::Ok);
  msgBox.setDefaultButton(QMessageBox::Ok);
  msgBox.exec();
}

void MainWindow::info_dialog(QString q) {
  QMessageBox msgBox;
  msgBox.setText(q);
  msgBox.exec();
}

void MainWindow::kill_acqthread() {
  control->quit.request();
  int timeout = 0;
  while(control->quit.get() == true) { // use this as an 'im dead' flag
    timeout += 50;
    QThread::msleep(50);
    if(timeout > 5000) exit(1);
  }
}

void MainWindow::init_debug_info() {
  debuglog_controller debug;
  if(QElapsedTimer::isMonotonic() == true) {
    debug.print_verbose("SYSTEM: Using monotonic reference clock.");
  } else {
    debug.print_verbose("SYSTEM: Using non-monotonic reference clock.");
  }
  int current_version = VERSION_CODE;
  debug.print("SYSTEM: EEHack Version " + QString::number(convert_version(current_version),'f',2));
  debug.print("SYSTEM: Current OS : " + QSysInfo::prettyProductName());
  debug.print("SYSTEM: Built with THROTTLE_MS=" + QString::number(THROTTLE_MS));
}

//------VERSIONING

float MainWindow::convert_version(int v) {
  float x = v;
  return x / 100;
}

void MainWindow::launch_url(QString url) {
  QDesktopServices::openUrl(QUrl(url));
}

QString MainWindow::current_version_string() {
  int current_version = VERSION_CODE;
  // start string construction
  QString version_string = "v"; // first line
  version_string.append(QString::number(convert_version(current_version),'f',2));
#ifdef BETA_VERSION
  version_string.append(" BETA");
#endif
  return version_string;
}

void MainWindow::check_version(int new_version) {
  int current_version = VERSION_CODE;
  int latest_version = new_version;
  // start string construction
  QString version_string = current_version_string();
  if(new_version == 0) {
    ui->versionDisplay->setText(version_string);
    return;
  }
  version_string.append("<br>"); // second line
  if(latest_version > current_version) {
    int ret = QMessageBox::warning(this, "New EEHack Version","A new version of EEHack has been released!\nAll releases contain important new features and bug fixes.\nDo you want to be taken to the download page now?",
                                   QMessageBox::No | QMessageBox::Yes, QMessageBox::Yes);
    if(ret == QMessageBox::Yes) {
      launch_url("http://fbodytech.com/eehack-2/download-eehack/");
    }
    version_string.append("<a href=\"http://fbodytech.com/eehack-2/download-eehack/\">");
    version_string.append("v"); // second line
    version_string.append(QString::number(convert_version(latest_version),'f',2));
    version_string.append(" IS OUT!</a>"); // second line
  } else if(latest_version < current_version) {
    version_string.append("(Pre-release)");
  } else {
    version_string.append("(Up to date)");
  }
  ui->versionDisplay->setText(version_string);
}

void MainWindow::activate_module(QWidget *w) {
  if(loading_complete == false) return;
  w->raise();
  w->activateWindow();
  w->showNormal();
}

void MainWindow::on_btn_mode4_clicked() {
  activate_module(mode4controller);
}

void MainWindow::on_btn_shownotes_clicked() {
 activate_module(notes);
}

void MainWindow::on_btn_debuglog_clicked() {
  activate_module(debugwin);
}

void MainWindow::on_btn_config_clicked() {
  activate_module(settings_window);
}

void MainWindow::on_btn_about_clicked() {
  activate_module(about_win);
}

void MainWindow::on_btn_raw_clicked() {
  activate_module(raw_command_window);
}

void MainWindow::on_btn_tests_clicked() {
  activate_module(test_window);
}

void MainWindow::on_btn_analyze_clicked() {
  activate_module(analyzer_win);
}

void MainWindow::on_btn_graph_clicked() {
  graphing *gr = new graphing(0,def,datalog_win->get_active_log_ptr());

  connect(datalog_win,SIGNAL(active_log_changed(datalog*)),gr,SLOT(new_display_log(datalog*)));
  connect(datalog_win,SIGNAL(log_position_changed(double)),gr,SLOT(move_cursor(double)));
  connect(this,SIGNAL(close_graphs()),gr,SLOT(close()));

  //---- SIGNALS FROM GRAPH WINDOW
  connect(gr,SIGNAL(request_log_position_change(double)),datalog_win,SLOT(change_log_position(double)));
  activate_module(gr);
}

void MainWindow::on_btn_memio_clicked() {
  activate_module(memio_win);
}

void MainWindow::load_datastream_defs() {
  debuglog_controller debug;
  eehack_settings config;
  QString filename;

  // this goto routine causes an infinite definition-selector loop which is only broken by
  // clicking 'exit' (which exits program).  first time execution skips selection.
  goto NO_SELECT_FIRST_TIME;

  SELECT_NEW_DEF:
  if(settings_window->select_deffile() == false) exit(1);

  NO_SELECT_FIRST_TIME:
  filename = config.get_deffile();
  if(filename.isEmpty() == true || filename.isNull() == true) {
    error_dialog("Could not open definition file, name empty");
    goto SELECT_NEW_DEF;
  }
  QFile f(filename);
  if(f.exists() == false) {
    error_dialog("Could not open definition file, file doesn't exist..." + filename);
    goto SELECT_NEW_DEF;
  }
  f.open( QIODevice::ReadOnly );
  if(f.isReadable() == false) {
    error_dialog("Could not open definition file " + filename);
    goto SELECT_NEW_DEF;
  }

  //--------
  QTextStream stream_in(&f);
  qint64 line_no = 0;
  int def_no = -1; // will start with zero
  int n_elements = 0;
  forever {
    QString string_in = stream_in.readLine();
    line_no++;
    if(string_in.isNull()) {
      break; // EOF
    }
    set_loading_status("Loading Def: " + QString::number(def_no));
    if(string_in.startsWith("NEW,",Qt::CaseInsensitive) == true) {
      def_no++;
      if(def_no >= MAX_MODE1_MESSAGES) {
        recieve_fatal_error_msg("Maximum definition limit exceeded!");
        exit(1);
      }
      if(def[def_no].setup(string_in) == false) {
        recieve_fatal_error_msg("ERROR at line number " +
                                QString::number(line_no) + "\n" +
                                def[def_no].parseerr);
        exit(1);
      }
      debug.print("SYSTEM: Loading Definition DEVICE=0x"  + QString::number(def[def_no].get_device(),16).toUpper()
                  + " MSG=0x" + QString::number(def[def_no].get_msg_number(),16).toUpper());
    } else {
      if(def[def_no].load_definition_line(string_in) == false) {
        recieve_fatal_error_msg("SYSTEM: DEFINITION ERROR at line number " +
                                QString::number(line_no) + "\n" +
                                def[def_no].parseerr);
        exit(1);
      } else {
        n_elements++;
      }
    }
  }
  if(def_no == -1) {
    recieve_fatal_error_msg("No valid definitions loaded!\n"
                            "EEHack requires a proprietery definition file to operate.\n"
                            "It should have been provided with this distrubition package.");
    exit(1);
  }
  debug.print("SYSTEM: Loaded "  + QString::number(n_elements) +
              " elements into " + QString::number(def_no + 1) + " definitions.");
  // configure linked list (def_no now represents last element idx)
  if(def_no == 0) { // single definition
    def[0].next_def = &def[0];
  } else {
    for(int x=0;x<def_no;x++) { // all but last element
      def[x].next_def = &def[x+1]; // link to next element
    }
    def[def_no].next_def = &def[0]; // loop from last to first rec
  }
}

void MainWindow::on_toolButton_toggled(bool checked) {
  if(checked) {
    QFont f = font();
    f.setPointSize(15);
    f.setBold(true);
    setFont(f);
  } else {
    QFont f = font();
    f.setPointSize(8);
    f.setBold(false);
    setFont(f);
  }
}

void MainWindow::on_btn_flash_clicked() {
  QMessageBox m;
  m.setWindowTitle("Flash tool");
  m.setText("EEHack's built in flash tool has been replaced by flashhack.\n"
            "It's free and much safer.\n"
            "Please visit fbodytech.com and download it!");
  m.exec();
}
