#include <QDebug>
#include <float.h>
#include "datalog_element.h"
#include "common.h"

datalog_element::datalog_element(int offset_in, datalog_definition *parent_in) {
  if(offset_in < 0 || offset_in > MAX_M1_DATA_SIZE) {
    qDebug() << "ELEMENT MEMORY ADDRESS OUT OF RANGE!! (BAIL)";
    exit(1);
  }
  // DEFAULTS
  offset = offset_in;
  size = 1;
  min_patch = 0x00;
  max_patch = 0xFF;
  eq_add = 0.00;
  eq_mult = 0.00;
  level = VIEW_HACKER;
  def = parent_in;
  warn_low = FLT_MIN;
  warn_high = FLT_MAX;
}

QString datalog_element::describe() {
  QString out;
  out.append("OFFSET=" + QString::number(offset));
  out.append(",SIZE=" + QString::number(size));
  out.append(",SHORTNAME=" + QString(short_name));
  out.append(",LONGNAME=" + long_name);
  if(min_patch != FLT_MIN) out.append(",MIN_PATCH=" + QString::number(min_patch));
  if(max_patch != FLT_MAX) out.append(",MAX_PATCH=" + QString::number(max_patch));
  return out;
}

bool datalog_element::is_associated(datalog_packet *p) {
  if(p == nullptr) return false;
  if(p->def == def) return true;
  return false;
}

QString datalog_element::full_title() {
  QString x;
  x += "[" + QString(short_name) + "] " + long_name;
  return x;
}

int datalog_element::get_raw(datalog_packet *p) {
  if(is_associated(p) == false) {
    qDebug() << "Wrong packet association for " << long_name << " def " << def->msg_desc << "returning zero!";
    return 0;
  }
  if(size == 32) {
    unsigned int x = ((p->get_raw_byte(offset)<<24)|
                      (p->get_raw_byte(offset+1)<<16)|
                      (p->get_raw_byte(offset+2)<<8)|
                      p->get_raw_byte(offset + 3));
    return convert_signed(x);
  } else if(size == 16) {
    unsigned int x = ((p->get_raw_byte(offset)<<8)|
                      p->get_raw_byte(offset + 1));
    return convert_signed(x);
  } else {
    int x = p->get_raw_byte(offset);
    if(special == DEF_SPECIAL_AVERAGE) {
      x += p->get_raw_byte(offset+1);
      x = x / 2;
    }
    return convert_signed(x);
  }
}

int datalog_element::convert_signed(int x) {
  if(special == DEF_SPECIAL_SIGNED) {
    // FIXME handle 32
    if(size == 16) return x - 32768;
    if(size == 8) return x - 128;
  }
  return x;
}

int datalog_element::get_int(datalog_packet *p) {
  if(is_associated(p) == false) {
    return 0;
  }
  if(size == 1 && precision == 0) return get_bool(p); // is bool, just return bool
  if(special == DEF_SPECIAL_SHIFTPOS) {  // special shift selection integer
    byte raw_in = get_raw(p);
    bool a = getbit(2,raw_in);
    bool b = getbit(3,raw_in);
    if(a == true && b == true) return 1;
    if(b == true && a == false) return 2;
    if(b == false && a == true) return 4;
    if(b == false && a == false) return 3;
  }
  if(eq_mult == 1 && eq_add == 0 && special == DEF_SPECIAL_NONE) return get_raw(p); // no conversion, straight int
  float x = get_float(p); // use floating point math
  int y;
  if(x >= 0) // round
    y = (int)(x + 0.5);
  else
    y = (int)(x - 0.5);
  return y;
}

float datalog_element::get_float(datalog_packet *p) {
  if(is_associated(p) == false) {
    return 0;
  }
  if(size == 1) return get_bool(p); // is bool
  return get_float_from_value(get_raw(p));
}

float datalog_element::get_float_from_value(int value) {
  if(special == DEF_SPECIAL_RPMMINOR) { //NL rpmperiod
    float raw_rpmperiod = value;
    if(raw_rpmperiod == 0) return 0.00; // avoid divide by zero;
    return 2949120/raw_rpmperiod;
  }
  return ( eq_mult * (float)value ) + eq_add; // normal conversion
}

bool datalog_element::uom_is(QString x) {
  if(QString::compare(uom,x,Qt::CaseInsensitive) == 0) return true;
  return false;
}

float datalog_element::get_range_min() {
  return range_min;
}

float datalog_element::get_range_max() {
  return range_max;
}

void datalog_element::calc_high_range() {
  range_max = 65535;
  if(size == 1) range_max = 1;
  if(size == 8) range_max = get_float_from_value(convert_signed(0xFF));
  if(size == 16) range_max = get_float_from_value(convert_signed(0xFFFF));
  if(size == 32) range_max = get_float_from_value(convert_signed(0xFFFFFFFF));
}

void datalog_element::calc_low_range() {
  range_min = get_float_from_value(convert_signed(0x00));
}

bool datalog_element::is_low(datalog_packet *p) {
  if(p == nullptr) return false;
  if(is_bool() == true) return false;
  if(warn_low == FLT_MIN) return false; // disabled
  if(get_float(p) < warn_low) return true;
  return false;
}

bool datalog_element::is_high(datalog_packet *p) {
  if(p == nullptr) return false;
  if(is_bool() == true) return false;
  if(warn_low == FLT_MAX) return false; // disabled
  if(get_float(p) > warn_high) return true;
  return false;
}

bool datalog_element::get_bool(datalog_packet *p) {
  if(is_associated(p) == false) {
    qDebug() << "Wrong bool packet association for " << long_name << " def " << def->msg_desc << " returning zero!";
    qDebug() << "pkt msg" << p->def->msg_desc << "pkt def" << p->def << "stored def" << def;
    return false;
  }
  if(size != 1) return 0;
  return getbit(precision,p->get_raw_byte(offset));
}

QString datalog_element::get_ascii(datalog_packet *p) {
  // FIXME this is a bit inefficient.....
  if(special != DEF_SPECIAL_ASCII) return QString();
  int str_length = size / 8; // get length (bits to bytes)
  char *str_raw = new char[str_length]; // allocate space to copy string
  for(int x=0;x<str_length;x++) str_raw[x] = p->get_raw_byte(x + offset); // deep copy string
  QByteArray bytearray_raw(str_raw,str_length); // convert to bytearray
  QString str_out = QString::fromUtf8(bytearray_raw);
  delete str_raw;
  return str_out;
}

QString datalog_element::get_string(datalog_packet *p, bool sparse) {
  QString str_out;
  if(p == nullptr) {
    str_out = "No Data";
  } else if(special == DEF_SPECIAL_ASCII) { // ascii string
    return get_ascii(p);
  } else if(is_bool() == true) { // boolean
    if(get_bool(p) == true) return QStringLiteral("ON");
    return QStringLiteral("OFF");
  } else if(special == DEF_SPECIAL_RESEARCH) { // print hex/binary rep
    // this is inefficient but should only be used for research variables
    unsigned int raw_number = get_raw(p);
    if(sparse == true) {
      str_out += QString::number(raw_number,10);
    } else {
      str_out += QString::number(raw_number,10);
      str_out += " ";
      str_out += "0x";
      str_out += QString::number(raw_number,16);
      str_out += " (";
      str_out += QString::number(raw_number,2);
      str_out += ")";
    }
  } else if(special == DEF_SPECIAL_CELCIUS) { // celcius conversion
    if(def->config.get_display_celcius() == true) {
      str_out = QString::number((double)get_float(p),'f',precision);
      if(sparse == false) str_out += "c";
    } else {
      str_out = QString::number((double)celcius_to_farenheit(get_float(p)),'f',precision);
      if(sparse == false) str_out += "f";
    }
  } else if(special == DEF_SPECIAL_MPH) { // spd conversion
    if(def->config.get_display_kph() == true) { // kph
      str_out = QString::number(mph_to_kph(get_int(p)),'f',0);
      if(sparse == false) str_out += "kph";
    } else { // mph
      str_out = QString::number(get_int(p));
      if(sparse == false) str_out += "mph";
    }
  } else if(special == DEF_SPECIAL_TRIM) { // trim to percentage option
    if(def->config.get_display_blm_percent() == true) {
      str_out = percent_to_str(trim_to_percent(get_int(p)));
    } else {
      str_out = QString::number(get_int(p));
    }
  } else if(is_int() == true) { // normal int
    str_out = QString::number(get_int(p));
    if(uom.isEmpty() == false && sparse == false) str_out += uom; // add uom if one exists
  } else if(is_float() == true) { // normal float
    str_out = QString::number((double)get_float(p),'f',precision);
    if(uom.isEmpty() == false && sparse == false) str_out += uom; // add uom if one exists
  } else { // unknown type
    str_out = "ERR";
  }
  return str_out;
}

QString datalog_element::percent_to_str(double x) {
  if(def->config.get_low_precision() == true) {
    if(x < DISCARD_TRIM && x > -DISCARD_TRIM && def->config.get_low_percent_ok() == true) {
      return QStringLiteral("+0%");
    }
    if(x > 0) return QString("+%1%").arg((double)x,0,'f',0);
    if(x < 0) return QString("%1%").arg((double)x,0,'f',0); // already has minus sign ...
    return QStringLiteral("+0%");
  } else { // one decimal place
    if(x < DISCARD_TRIM && x > -DISCARD_TRIM && def->config.get_low_percent_ok() == true) {
      return QStringLiteral("+0.0");
    }
    if(x > 0) return QString("+%1").arg((double)x,0,'f',1);
    if(x < 0) return QString("%1").arg((double)x,0,'f',1); // already has minus sign ...
    return QStringLiteral("+0.0");
  }
}

float datalog_element::trim_to_percent(int trim) {
  return (((float)trim / 128.00 * 100.00)-100.00);
}

bool datalog_element::is_float() {
  if(size >= 8 && precision > 0) return true;
  return false;
}

bool datalog_element::is_int() {
  if(size >= 8 && precision == 0) return true;
  return false;
}

bool datalog_element::is_bool() {
  if(size == 1) return true;
  return false;
}

bool datalog_element::matches_patchlvl(byte patchlvl) {
  if(patchlvl < min_patch) return false;
  if(patchlvl > max_patch) return false;
  return true;
}
