#include <QApplication>
#include <QtDebug>

#include "stdlib.h"
#include "config.h"
#include "common.h"

#include "datalog_packet.h"

datalog_packet::datalog_packet(datalog_definition *definition_in, datalog *log_in, int size_in) {
  attached = false;
  next = nullptr;
  prev = nullptr;
  size = size_in;
  def = definition_in;
  log = log_in;
  // start blank
  for(int x=0;x<MAX_M1_DATA_SIZE;x++) {
    data[x] = 0x00;
  }
}

int datalog_packet::get_int(int i) {
  if(def == nullptr) return 0;
  datalog_element *e = def->element_by_index(i);
  if(e == nullptr) {
    qDebug() << "element not found!! no" << i;
    return 0;
  }
  return e->get_int(this);
}

float datalog_packet::get_float(int i) {
  if(def == nullptr) return 0;
  datalog_element *e = def->element_by_index(i);
  if(e == nullptr) {
    qDebug() << "element not found!! no" << i;
    return 0;
  }
  return e->get_float(this);
}

bool datalog_packet::get_bool(int i) {
  if(def == nullptr) return 0;
  datalog_element *e = def->element_by_index(i);
  if(e == nullptr) {
    qDebug() << "element not found!! no" << i;
    return 0;
  }
  return e->get_bool(this);
}

int datalog_packet::get_int(datalog_element *e) {
  return e->get_int(this);
}

float datalog_packet::get_float(datalog_element *e) {
  return e->get_float(this);
}

bool datalog_packet::get_bool(datalog_element *e) {
  return e->get_bool(this);
}

int datalog_packet::get_int(const char *str) {
  if(def == nullptr) return 0;
  datalog_element *e = def->element_by_name(str);
  if(e == nullptr) {
    qDebug() << "element not found!!  " << str;
    return 0;
  }
  return e->get_int(this);
}

float datalog_packet::get_float(const char *str) {
  if(def == nullptr) return 0;
  datalog_element *e = def->element_by_name(str);
  if(e == nullptr) {
    qDebug() << "element not found!!  " << str;
    return 0;
  }
  return e->get_float(this);
}

bool datalog_packet::get_bool(const char *str) {
  if(def == nullptr) return 0;
  datalog_element *e = def->element_by_name(str);
  if(e == nullptr) {
    qDebug() << "element not found!!  " << str;
    return 0;
  }
  return e->get_bool(this);
}

int datalog_packet::get_msgnumber() {
  if(def == nullptr) return -1;
  return def->get_msg_number();
}

int datalog_packet::get_device() {
  if(def == nullptr) return -1;
  return def->get_device();
}

bool datalog_packet::is_msg(int device, int msg_no) {
  if(get_msgnumber() == msg_no && get_device() == device) return true;
  return false;
}

byte datalog_packet::get_raw_byte(byte offset) {
  if(offset > MAX_M1_DATA_SIZE - 1) {
    qDebug() << "Attempt to read data out of bounds";
    return 0x00;
  }
  return data[offset];
}

quint64 datalog_packet::get_raw_checksum() {
  quint64 ck = 0;
  for(int x=0;x<MAX_M1_DATA_SIZE;x++) {
    ck += x;
  }
  return ck;
}

void datalog_packet::set_raw_byte(byte offset, byte input) {
  if(offset > MAX_M1_DATA_SIZE - 1) { // overrun
    qDebug() << "Packet byte input OFFSET OVERRUN (prevented)";
    return;
  }
  // if this packet is attached, forbid writing to it
  if(is_attached() == false) { // not attached
    data[offset] = input;
  } else {
    qDebug() << "Attempt to write attached packet!!";
  }
}

bool datalog_packet::is_attached() {
  return attached;
}

void datalog_packet::set_attached() {
  attached = true;
}

void datalog_packet::detach() {
  datalog_packet *previous_link = get_prev();
  datalog_packet *next_link = get_next();
  if(previous_link != nullptr) previous_link->set_next(next_link);
  if(next_link != nullptr) next_link->set_prev(previous_link);
  set_next(nullptr);
  set_prev(nullptr);
  attached = false;
}


void datalog_packet::set_next(datalog_packet *next_packet) {
  lock_next.lock();
  set_attached();
  next = next_packet;
  lock_next.unlock();
}

void datalog_packet::set_prev(datalog_packet *prev_packet) {
  lock_prev.lock();
  set_attached();
  prev = prev_packet;
  lock_prev.unlock();
}

datalog_packet *datalog_packet::get_next(int device, int msgnumber) {
  datalog_packet *x = this;
  lock_next.lock();
  do {
    x = x->next;
  } while(x != nullptr &&
          ( x->get_msgnumber() != msgnumber || x->get_device() != device )
          );
  lock_next.unlock();
  if(x == this) return nullptr;
  return x;
}

datalog_packet *datalog_packet::get_next() {
  datalog_packet *x;
  lock_next.lock();
  x = next;
  lock_next.unlock();
  return x;
}

datalog_packet *datalog_packet::get_latest() {
  datalog_packet *x = this;
  while(x->next != nullptr) x = x->next;
  return x;
}

datalog_packet *datalog_packet::get_prev(int device, int msgnumber) {
  datalog_packet *x = this;
  lock_prev.lock();
  do {
    x = x->prev;
  } while(x != nullptr &&
          ( x->get_msgnumber() != msgnumber || x->get_device() != device )
          );
  lock_prev.unlock();
  if(x == this) return nullptr;
  return x;
}

datalog_packet *datalog_packet::get_prev() {
  datalog_packet *x;
  lock_prev.lock();
  x = prev;
  lock_prev.unlock();
  return x;
}

bool datalog_packet::is_last_packet(int device, int msgnumber) {
  if(get_next(device,msgnumber) == nullptr) return true;
  return false;
}

bool datalog_packet::is_last_packet() {
  if(get_next() == nullptr) return true;
  return false;
}

bool datalog_packet::is_first_packet(int device, int msgnumber) {
  if(get_prev(device, msgnumber) == nullptr) return true;
  return false;
}
