#include <QMessageBox>
#include <QFileDialog>
#include <QProgressDialog>
#include <QVector>

#include <float.h>

#include "qcustomplot.h"

#include "config.h"
#include "common.h"
#include "datalog.h"
#include "graphing.h"
#include "ui_graphing.h"
#include "parameter_select.h"
#include "datalog.h"
#include "datalog_element.h"
#include "datalog_packet.h"

graphing::graphing(QWidget *parent, datalog_definition *def_in, datalog *log_in) :
  QWidget(parent),
  ui(new Ui::graphing) {
  ui->setupUi(this);
  log = log_in;
  def = def_in;
  element_topleft = nullptr;
  element_topright = nullptr;
  element_bottomleft = nullptr;
  element_bottomright = nullptr;
  element_topoverlay = nullptr;
  element_bottomoverlay = nullptr;
  configure_graphs();
}

graphing::~graphing() {
  delete ui;
}

void graphing::new_display_log(datalog *log_in) {
  log = log_in;
  replot_all_graphs();
}

void graphing::configure_graphs() {
  cursor_position = 0;
  config_graph_layout(ui->graph);
  config_graph_layout(ui->graph2);
  //colors
  ui->graph->graph(0)->setPen(QPen(Qt::darkRed));
  ui->graph->graph(1)->setPen(QPen(Qt::darkBlue));
  ui->graph->yAxis->setLabelColor(Qt::darkRed);
  ui->graph->yAxis2->setLabelColor(Qt::darkBlue);
  ui->graph2->graph(0)->setPen(QPen(Qt::darkGreen));
  ui->graph2->graph(1)->setPen(QPen(Qt::darkMagenta));
  ui->graph2->yAxis->setLabelColor(Qt::darkGreen);
  ui->graph2->yAxis2->setLabelColor(Qt::darkMagenta);
  ui->graph->graph(2)->setBrush(QBrush(QColor(0,180,0,60)));
  ui->graph2->graph(2)->setBrush(QBrush(QColor(200,0,0,60)));
  // init line style
  set_line_style(0,ui->graph);
  set_line_style(0,ui->graph2);
  // cursors
  cursor_top = new QCPItemLine(ui->graph);
  cursor_bottom = new QCPItemLine(ui->graph2);
  cursor_top->setAntialiased(false);
  cursor_bottom->setAntialiased(false);
  ui->graph->addItem(cursor_top);
  ui->graph2->addItem(cursor_bottom);
  cursor_top->start->setCoords(0, QCPRange::minRange);
  cursor_top->end->setCoords(0, QCPRange::maxRange);
  cursor_bottom->start->setCoords(0, QCPRange::minRange);
  cursor_bottom->end->setCoords(0, QCPRange::maxRange);
  // scroll/zoom links
  connect(ui->graph->xAxis, SIGNAL(rangeChanged(QCPRange)), ui->graph2->xAxis, SLOT(setRange(QCPRange)));
  connect(ui->graph2->xAxis, SIGNAL(rangeChanged(QCPRange)), ui->graph->xAxis, SLOT(setRange(QCPRange)));
  // external ..
  connect(ui->graph->xAxis, SIGNAL(rangeChanged(QCPRange)), this, SIGNAL(rangeChanged(QCPRange)));
  connect(ui->graph->xAxis, SIGNAL(setRange(QCPRange)), this, SIGNAL(setRange(QCPRange)));
  //.....
  connect(ui->graph->xAxis, SIGNAL(rangeChanged(QCPRange)), this, SLOT(xAxisChanged()));
  connect(ui->graph2->xAxis, SIGNAL(rangeChanged(QCPRange)), this, SLOT(x2AxisChanged()));
  // click links
  connect(ui->graph,SIGNAL(mouseDoubleClick(QMouseEvent*)),this,SLOT(graph_clicked_event(QMouseEvent*)));
  connect(ui->graph2,SIGNAL(mouseDoubleClick(QMouseEvent*)),this,SLOT(graph_clicked_event(QMouseEvent*)));
  // sync
  QCPMarginGroup *m_group = new QCPMarginGroup(ui->graph);
  ui->graph->axisRect()->setMarginGroup(QCP::msLeft|QCP::msRight, m_group);
  ui->graph2->axisRect()->setMarginGroup(QCP::msLeft|QCP::msRight, m_group);
}

void graphing::graph_clicked_event(QMouseEvent *event) {
  emit request_log_position_change(ui->graph->xAxis->pixelToCoord(event->pos().x()) * 1000);
}

void graphing::move_cursor(double position) {
  cursor_position = position / 1000;
  if(this->isVisible() == true) redraw_cursor();
}

void graphing::showEvent(QShowEvent*) {
  redraw_cursor();
}

void graphing::change_cursor_position(double pos) {
  cursor_position = pos;
  cursor_top->start->setCoords(pos, QCPRange::minRange);
  cursor_top->end->setCoords(pos, QCPRange::maxRange);
  cursor_bottom->start->setCoords(pos, QCPRange::minRange);
  cursor_bottom->end->setCoords(pos, QCPRange::maxRange);
}

void graphing::redraw_cursor() {
  change_cursor_position(cursor_position);
  ui->graph->replot();
  ui->graph2->replot();
}

void graphing::x2AxisChanged() {
  ui->graph2->replot();
}

void graphing::xAxisChanged() {
  ui->graph->replot();
}

void graphing::config_graph_layout(QCustomPlot *graph) {
  // interactions
  graph->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom);
  // rects
  QCPAxisRect *axis_rect = graph->axisRect(); // get def axis ref
  axis_rect->setRangeZoom(Qt::Horizontal);
  axis_rect->setRangeDrag(Qt::Horizontal);
  // left hand axis
  graph->addGraph();
  // right hand axis
  graph->addGraph(graph->xAxis,graph->yAxis2);
  graph->yAxis2->setVisible(true);
  config_graph_axis(graph->xAxis, "TIME (SEC)", 400);
  config_graph_axis(graph->yAxis); // config with no data
  config_graph_axis(graph->yAxis2); // config with no data
  // binary chart
  graph->addGraph(graph->xAxis,graph->yAxis2);
  QPen pen;
  pen.setColor(QColor(0, 0, 0, 0));
  graph->graph(2)->setPen(pen);
  // color the backgound same as the window
  graph->setBackground(ui->setting_graph_smooth->palette().color(QWidget::backgroundRole()));
  graph->replot();
}

void graphing::set_plot_style(int style, QCustomPlot *graph) {
  if(style == 0) {
    graph->graph(0)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssDisc, 3));
    graph->graph(1)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssDisc, 3));
    graph->graph(2)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssNone, 0));
  } else {
    graph->graph(0)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssNone, 0));
    graph->graph(1)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssNone, 0));
    graph->graph(2)->setScatterStyle(QCPScatterStyle(QCPScatterStyle::ssNone, 0));
  }
}

void graphing::set_line_style(int style, QCustomPlot *graph) {
  if(style == 0) {
    graph->graph(0)->setLineStyle(QCPGraph::lsStepLeft);
    graph->graph(1)->setLineStyle(QCPGraph::lsStepLeft);
    graph->graph(2)->setLineStyle(QCPGraph::lsStepLeft);
  } else {
    graph->graph(0)->setLineStyle(QCPGraph::lsLine);
    graph->graph(1)->setLineStyle(QCPGraph::lsLine);
    graph->graph(2)->setLineStyle(QCPGraph::lsStepLeft);
  }
}

void graphing::config_graph_axis(QCPAxis *axis, datalog_element *element_in) {
  if(element_in == nullptr) {
    config_graph_axis(axis);
    return;
  }
  QString label;
  label += element_in->long_name;
  if(element_in->uom.isEmpty() == false) label += "(" + element_in->uom + ")";
  axis->setLabel(label);
  axis->setRange(element_in->get_range_min(),element_in->get_range_max());
}

void graphing::config_graph_axis(QCPAxis *axis, QString label, int maxrange) {
  axis->setLabel(label);
  axis->setRange(0,maxrange);
}

void graphing::config_graph_axis(QCPAxis *axis) {
  axis->setLabel("NONE");
  axis->setRange(0,100);
}

void graphing::replot_all_graphs() {
  replot_upper_graph();
  replot_lower_graph();
}

void graphing::replot_upper_graph() {
  config_graph_axis(ui->graph->yAxis, element_topleft);
  config_graph_axis(ui->graph->yAxis2, element_topright);
  if(log->is_data() == false) {
    for(int x=0;x<=2;x++) ui->graph->graph(x)->clearData();
  } else {
    get_graphdata(0,ui->graph,element_topleft,false);
    get_graphdata(1,ui->graph,element_topright,false);
    get_graphdata(2,ui->graph,element_topoverlay,true);
  }
  ui->graph->replot();
}

void graphing::replot_lower_graph() {
  config_graph_axis(ui->graph2->yAxis, element_bottomleft);
  config_graph_axis(ui->graph2->yAxis2, element_bottomright);
  if(log->is_data() == false) { // no data
    for(int x=0;x<=2;x++) ui->graph2->graph(x)->clearData();
  } else {
    get_graphdata(0,ui->graph2,element_bottomleft,false);
    get_graphdata(1,ui->graph2,element_bottomright,false);
    get_graphdata(2,ui->graph2,element_bottomoverlay,true);
  }
  ui->graph2->replot();
}

void graphing::get_graphdata(int lr, QCustomPlot *graph, datalog_element *element, bool is_bool) {
  graph->graph(lr)->clearData();
  if(element == nullptr) return;
  unsigned int n_packets = log->n_packets(element);
  datalog_packet *p = log->oldest_packet();
  QVector<double> x(n_packets);
  QVector<double> y(n_packets);
  // get minimum and maximum values
  double max_range = graph->yAxis2->range().upper;
  unsigned int i=0;
  forever {
    if(element->is_associated(p) == true) {
      if(is_bool) {
        if(element->get_bool(p) == true) { // temp
          x[i] = max_range;
        } else {
          x[i] = 0;
        }
      } else {
        x[i] = element->get_float(p);
      }
      y[i] = (double)p->timestamp / 1000.00; // conv to seconds
      i++;
      if(i > n_packets) break;
    }
    if(p->is_last_packet() == false) {
      p = p->get_next();
    } else {
      break;
    }
  }
  graph->graph(lr)->setData(y,x);
}

void graphing::on_setting_graph_dots_toggled(bool checked) {
  if(checked == true) {
    set_plot_style(0,ui->graph);
    set_plot_style(0,ui->graph2);
  } else {
    set_plot_style(1,ui->graph);
    set_plot_style(1,ui->graph2);
  }
  replot_all_graphs();
}

void graphing::on_setting_graph_smooth_toggled(bool checked) {
  if(checked == true) {
    set_line_style(1,ui->graph);
    set_line_style(1,ui->graph2);
  } else {
    set_line_style(0,ui->graph);
    set_line_style(0,ui->graph2);
  }
  replot_all_graphs();
}

datalog_element *graphing::select_element(datalog_element *current_element, bool display_bools) {
  parameter_select::restriction_t r;
  if(display_bools == true) {
    r = parameter_select::RESTRICT_SWITCH;
  } else {
    r = parameter_select::RESTRICT_NUM;
  }
  parameter_select parameter_dialog(this,def,log,r);
  if(parameter_dialog.exec() == QDialog::Accepted) {
    return parameter_dialog.get_result();
  } else {
    return current_element;
  }
}

void graphing::set_element_title(QPushButton *btn, datalog_element *e) {
  if(e == nullptr) {
    btn->setText("Select Element");
  } else {
    btn->setText(e->long_name);
  }
}

void graphing::on_btn_select_topleft_clicked() {
  element_topleft = select_element(element_topleft,false);
  set_element_title(ui->btn_select_topleft,element_topleft);
  replot_upper_graph();
}

void graphing::on_btn_select_topright_clicked() {
  element_topright = select_element(element_topright,false);
  set_element_title(ui->btn_select_topright,element_topright);
  replot_upper_graph();
}

void graphing::on_btn_select_bottomleft_clicked() {
  element_bottomleft = select_element(element_bottomleft,false);
  set_element_title(ui->btn_select_bottomleft,element_bottomleft);
  replot_lower_graph();
}

void graphing::on_btn_select_bottomright_clicked() {
  element_bottomright = select_element(element_bottomright,false);
  set_element_title(ui->btn_select_bottomright,element_bottomright);
  replot_lower_graph();
}

void graphing::on_btn_select_topoverlay_clicked() {
  element_topoverlay = select_element(element_topoverlay,true);
  set_element_title(ui->btn_select_topoverlay,element_topoverlay);
  replot_upper_graph();
}

void graphing::on_btn_select_bottomoverlay_clicked() {
  element_bottomoverlay = select_element(element_bottomoverlay,true);
  set_element_title(ui->btn_select_bottomoverlay,element_bottomoverlay);
  replot_lower_graph();
}

void graphing::on_refresh_graph_btn_clicked() {
  replot_all_graphs();
}

void graphing::on_graph_reset_zoom_btn_clicked() {
  float n_timestamp = clamp((float)log->newest_timestamp() / 1000,(float)100,(float)6000000);
  ui->graph->xAxis->setRange(0,n_timestamp);
}
