#include <QListWidget>
#include <QListWidgetItem>
#include <QVariant>

#include "parameter_select.h"
#include "ui_parameter_select.h"

#include "datalog.h"
#include "datalog_definition.h"
#include "datalog_element.h"
#include "datalog_packet.h"

parameter_select::parameter_select(QWidget *parent, datalog_definition *def_in, datalog *log_in,
                                   restriction_t restrict_type,
                                   datalog_definition *restrict_definition) :
  QDialog(parent),
  ui(new Ui::parameter_select) {
  ui->setupUi(this);
  selected_element = nullptr;
  def = def_in;
  log = log_in;
  restriction = restrict_type;
  filter.setPatternSyntax(QRegExp::Wildcard);
  filter.setPattern("*");
  filter.setCaseSensitivity(Qt::CaseInsensitive);
  // static crap
  ui->selector_viewlevel->addItem(EXTENDED_VIEWLEVEL_STR); // 0
  ui->selector_viewlevel->addItem(BASIC_VIEWLEVEL_STR);    // 1
  ui->selector_viewlevel->addItem(HACKER_VIEWLEVEL_STR);   // 2
  if(restrict_definition == nullptr) { // multiple definition mode
    for(int x=0;x<MAX_MODE1_MESSAGES;x++) {
      datalog_definition *w_def = &def[x]; // working def
      if(w_def->is_empty() == true) continue;
      add_def_to_dropbox(w_def);
    }
  } else { // single definition
    add_def_to_dropbox(restrict_definition);
    ui->selector_message->setEnabled(false);
  }
}

void parameter_select::add_def_to_dropbox(datalog_definition *w_def) {
  QString msg_title;
  if(w_def->msg_desc.isEmpty() == false) msg_title += w_def->msg_desc + " ";
  msg_title += "(Msg" + QString::number(w_def->get_msg_number(),16).toUpper() + "/" +
      QString::number(w_def->get_device(),16).toUpper() + ")";
  QVariant def_ptr = qVariantFromValue((void *)w_def);
  ui->selector_message->addItem(msg_title,def_ptr);
}

parameter_select::~parameter_select() {
  delete ui;
}

datalog_element *parameter_select::get_result() {
  return selected_element;
}

void parameter_select::on_btn_cancel_clicked() {
  selected_element = nullptr;
  reject();
}

void parameter_select::on_btn_select_clicked() {
  accept();
}

void parameter_select::refresh() {
  // safety
  if(def == nullptr) return;
  if(log == nullptr) return;
  // get some variables
  int patch_level = get_patch_level();
  int view_level = get_view_level();
  datalog_definition *selected_def = get_selected_def();
  if(selected_def == nullptr) return;
  datalog_element *x = selected_def->get_first_element();
  if(x == nullptr) return;
  // iterate
  ui->parameter_list->clear(); // restart list
  while(x != nullptr) {
    // qualify parameter
    if(x->level >= view_level &&
       patch_level >= x->min_patch &&
       patch_level <= x->max_patch &&
       qualify_element_type(x) == true) {
       QString current_title = x->full_title();
       if(filter.isEmpty() || current_title.contains(filter)) {
         QListWidgetItem *new_item = new QListWidgetItem(x->full_title());
         QVariant element_ptr = qVariantFromValue((void *)x);
         new_item->setData(Qt::UserRole,element_ptr);
         ui->parameter_list->addItem(new_item);
       }
    }
    x = x->next;
  }
}

bool parameter_select::qualify_element_type(datalog_element *x) {
  if(x->is_bool() == true && restriction == RESTRICT_NUM) return false;
  if(x->is_bool() == false && restriction == RESTRICT_SWITCH) return false;
  return true;
}

int parameter_select::get_patch_level() {
  if(log == nullptr) return 0x00;
  return log->info.patch_version.get();
}

datalog_definition *parameter_select::get_selected_def() {
  QVariant v = ui->selector_message->currentData();
  datalog_definition *def_out = (datalog_definition*) v.value<void *>();
  return def_out;
}

datalog_element::VIEW_LEVEL parameter_select::get_view_level() {
  QString view_level_text = ui->selector_viewlevel->currentText();
  if(view_level_text == BASIC_VIEWLEVEL_STR) {
    return datalog_element::VIEW_BASIC;
  } else if(view_level_text == EXTENDED_VIEWLEVEL_STR) {
    return datalog_element::VIEW_EXTENDED;
  } else if(view_level_text == HACKER_VIEWLEVEL_STR) {
    return datalog_element::VIEW_HACKER;
  }
  // fallback
  return datalog_element::VIEW_BASIC;
}


void parameter_select::on_selector_message_currentIndexChanged(int) {
  refresh();
}

void parameter_select::on_selector_viewlevel_currentIndexChanged(int) {
  refresh();
}

void parameter_select::on_parameter_list_currentItemChanged(QListWidgetItem *current, QListWidgetItem *item_index) {
  (void)item_index;
  if(current == nullptr) {
    selected_element = nullptr;
  } else {
    QVariant v = current->data(Qt::UserRole);
    selected_element = (datalog_element*) v.value<void *>();
  }
}

void parameter_select::on_parameter_list_doubleClicked(const QModelIndex) {
  if(selected_element == nullptr) return;
  accept();
}

void parameter_select::on_parameter_filter_textChanged(const QString &arg1) {
  if(arg1.isEmpty()) {
    filter.setPattern("*");
  } else {
    filter.setPattern(arg1);
  }
  refresh();
}
