#include <QElapsedTimer>
#include "datalog_window.h"
#include "config.h"
#include "knock_warning.h"
#include "common.h"

void datastream::post_thread_create_init() {
  port = new QSerialPort();
  if(port == nullptr) {
    critical_error_message("Serial port structure can't be created.");
    return;
  }
}

void datastream::disconnected_event_loop() {
  change_connection_state(STATE_DISCONNECTED);
  forever {
    if(control->quit.get() == true) { // quit context.  we're dying.
      display_event(QStringLiteral("Quitting"));
      aldl_close_connection(); // fails if port closed anyway
      control->quit.set(false); // re-used as an 'im dead' flag @ ui
      QThread::exit(0); // ends event loop and kills thread
      return; // ...
    } else if(control->connection.get() == true) { // switch to connected context
      log->invalidate_log_timers(); // we've been disconnected a while now ...
      connected_event_loop();
      // post event loop cleanup
      calc_avg_pktrate(0,0);
      aldl_close_connection();
      close_serial_port();
      restart_garbage_collection();
      change_connection_state(STATE_DISCONNECTED);
      control->connection.set(false);
      control->reset();
    } else if(control->custom_a.send.get() == true) {
      if(acq_openserialport() == false) continue;
      display_event(QStringLiteral("Send custom command A"));
      control->custom_a.get(buf_out); // cp to output buffer
      emit custom_a_copied();
      emit custom_log_append(custom_command_as_hexstring(control->custom_a.postdelay.get()));
      if(control->custom_a.loop.get() == false) control->custom_a.send.set(false);
      close_serial_port();
    } else if(control->custom_b.send.get() == true) {
      if(acq_openserialport() == false) continue;
      display_event(QStringLiteral("Send custom command B"));
      control->custom_b.get(buf_out); // cp to output buffer
      emit custom_b_copied();
      emit custom_log_append(custom_command_as_hexstring(control->custom_b.postdelay.get()));
      if(control->custom_b.loop.get() == false) control->custom_b.send.set(false);
      close_serial_port();
    } else if(control->custom_idle.get() == true) {
      if(acq_openserialport() == false) continue;
      scan_idle();
      close_serial_port();
      control->custom_idle.set(false);
    } else {
      msleep(THROTTLE_MS * 20); // nothing happening, don't peg the CPU!
      display_event(QStringLiteral("Idle"));
    }
  }
}

void datastream::scan_idle() {
  QElapsedTimer t;
  t.start();
  emit custom_log_append("START IDLE SCAN LOG");
  qint64 start_time = 0;
  qint64 end_time = 0;
  while(t.hasExpired(15000) == false) {
    start_time = t.elapsed();
    int in_bytes = serial_read_any();
    if(in_bytes > 0) {
      emit custom_log_append("::: GAP" + QString::number(start_time - end_time) + "ms");
      end_time = t.elapsed();
      QString x = QString::number(start_time) +
          "ms to " +
          QString::number(end_time) +
          "ms (" +
          QString::number(end_time - start_time) + "ms) :: " +
          buf_as_colored_hexstring(buf_in,in_bytes);
      emit custom_log_append(x);
    }
  }
  emit custom_log_append("FINISH IDLE SCAN LOG");
}

void datastream::display_event(QString str) {
  emit acq_event_display(str);
}

void datastream::change_connection_state(int c) {
  control->connection_state.set(c);
  emit connection_state_changed(c);
}

void datastream::critical_error_message(QString msg) {
  emit(display_critical_error(msg));
}

bool datastream::aldl_init_connection() {
  change_connection_state(STATE_CONNECTING);
  emit(update_pktrate(0)); // set initial packet rate to zero
  restart_garbage_collection();
  serial_discard_all(100);
  int shutup_result = false;
  while(shutup_result == false) {
    if(control->connection.get() == false) return false;
    if(control->quit.get() == true) return false;
    silence_modules();
    while(serial_read_any() == true); // discard garbage
    msleep(25);            // short delay
    shutup_result = m8_silence_request(); // try a real mode8 request now to see if the ecm is listening
  }
  while(serial_read_any() == true); // discard garbage
  return true;
}

void datastream::aldl_close_connection() {
  control->connection.set(false); // just in case
  change_connection_state(STATE_DISCONNECTING);
  if(port->isOpen() == false) return;
  if(config.get_graceful_disconnect() == false) return;
  request_resume();
}

bool datastream::acq_openserialport() {
  display_event(QStringLiteral("Opening Serial Port"));
  if(open_serial_port() == false) { // fail
    critical_error_message(QStringLiteral("The selected serial port doesn't seem to work.\n...Try another one?\nThe debug log has more info about this failure.\nYou will now be taken to the config page to choose a new port."));
    emit invalid_serial_port();
    change_connection_state(STATE_DISCONNECTED);
    return false; // break back to idle
  }
  return true;
}

void datastream::connected_event_loop() {
  setPriority(HighestPriority);

  // -- variables
  int m4_interval = 0; // interval at which to re-send advance
  QElapsedTimer pktrate_timer;

  // -- open serial port
  if(acq_openserialport() == false) return;

  emit reset_error_count();

  // -- silence ECM
  LShutup:
  display_event("Connecting...");
  if(aldl_init_connection() == false) return;

  // -- from this point on consider us 'connected'
  change_connection_state(STATE_CONNECTED);
  control->time_connected.restart();

  // reset some variables
  pkt_fail_counter = 0;
  pktrate_timer.start();

  // datastream current acq record
  datalog_definition *current_def = control->def;

  // -- MAIN EVENT LOOP
  forever {
    // check for quit/cancel
    if(control->connection.get() == false || control->quit.get() == true) {
      return;
    }

    // check for closed port
    if(is_serial_port_open() == false) {
      emit(display_critical_error(QStringLiteral("The serial port dissappeared.  Try plugging it in again...")));
      return;
    }

    // this is for gaps between commands.  if a successful command has not been processed in a while,
    // the ecm will likely resume idle traffic, we want to shut it up before that happens.
    if(keepalive_timer.elapsed() > 1500) {
      // ... or if we wait too long we'll have to return to a 'reconnecting state'.
      if(keepalive_timer.elapsed() > 2500) {
        goto LShutup;
      } else {
        display_event(QStringLiteral("Keep-alive"));
        m8_silence_request();
      }
    }

    // check for repeated fails
    if(pkt_fail_counter > MAX_FAIL_PKTS) {
      display_event(QStringLiteral("Resynchronize"));
      msleep(250);
      goto LShutup;
    }

    //--------- ECM INFO
    if(control->ecminfo.getclear() == true) {
      display_event(QStringLiteral("Get ECM Info"));
      if(get_ecminfo() == false) control->ecminfo.set(true);
      emit updated_ecm_info(); // transmit update signal regardless
    }

    //--------- CYLDROPTEST
    if(control->cyldroptest.getclear() == true) {
      display_event(QStringLiteral("Cyl. Drop Test"));
      cyl_drop_test();
      // must retry manually on failure for this op
    }

    //--------- MODE1 RETRIEVAL (always loops....)
    datalog_definition *next_def = current_def->next_acq_def();
    if(next_def != nullptr) {
      current_def = next_def;
      display_event(QStringLiteral("Acquire Data"));
      datalog_packet *incoming_pkt = acquire_mode1_packet(current_def);
      if(incoming_pkt != nullptr) {
        calc_avg_pktrate(pktrate_timer.elapsed(),incoming_pkt->size);
        pktrate_timer.start();
      }
      if(current_def->loop.get() == false) current_def->acq.disable(); // one shot
    } else if(def->any_acq_enabled() == false) {
      msleep(1); // ensure the loop doesn't freeze up if no work is being done
      calc_avg_pktrate(0,0);
    }

    // ---------M4 KEEPALIVE COUNTER
    if(control->m4_keepalive.get() == true) {
      if(m4_interval >= M4_RESEND_INTERVAL) {
        control->m4_updated.request();
        m4_interval = 0;
      } else {
        m4_interval++;
      }
    }

    // ---------MODE4 SEND
    if(control->m4_updated.get() == true) display_event(QStringLiteral("Send M4 Parameters"));
    if(m4_send_if_requested() == false) { // send OK or no request made
      control->m4_updated.set(true); // resend if fail
    }

    // ---------CUSTOM COMMANDS
    if(control->custom_a.send.getclear() == true) {
      display_event(QStringLiteral("Send custom command A"));
      control->custom_a.get(buf_out); // cp to output buffer
      emit custom_a_copied();
      emit custom_log_append(custom_command_as_hexstring(control->custom_a.postdelay.get()));
      if(control->custom_a.loop.get() == true) control->custom_a.send.set(true);
    }
    if(control->custom_b.send.getclear() == true) {
      display_event(QStringLiteral("Send custom command B"));
      control->custom_b.get(buf_out); // cp to output buffer
      emit custom_b_copied();
      emit custom_log_append(custom_command_as_hexstring(control->custom_b.postdelay.get()));
      if(control->custom_b.loop.get() == true) control->custom_b.send.set(true);
    }

    // ---------CLEAR DTC SEND
    if(control->clear_dtc.getclear() == true) {
      display_event(QStringLiteral("Clear DTC"));
      if(clear_dtc() == false) control->clear_dtc.set(true); // resend if fail
    }

    //---------DUMP BLM MEMORY
    if(control->blmcell_dump.getclear() == true) {
      display_event(QStringLiteral("Dump BLM Memory"));
      if(get_blm_cell_memory() == true) {
        emit update_blm_cells(process_blm_cell_memory()); // signal main thread with byte array
      } else {
        control->blmcell_dump.set(true); // resend if fail
      }
    }

    //-------SET VIN/CALID
    if(control->set_vin.getclear() == true) {
      display_event(QStringLiteral("Program VIN"));
      QByteArray vin(log->info.vin_number.get().toUtf8());
      if(vin.length() != 17) {
        critical_error_message(QStringLiteral("Error: Wrong Vin Length!!"));
        control->set_vin.set(false);
      } else {
        if(set_vin(vin) == false) control->set_vin.set(true); // resend if fail
      }
    }
    if(control->set_calid.getclear() == true) {
      display_event(QStringLiteral("Program CALID"));
      if(set_calid(log->info.calibration_id.get()) == false) control->set_calid.set(true);
    }

  } //-----END MAIN LOOP
}

bool datastream::get_main_data_packet() {
  return acquire_mode1_packet(&def[0]);
}

void datastream::calc_avg_pktrate(qint64 pktrate_in, int size) {
  static float pktrate_avg = 0;
  static int avg_counter = 0;
  if(pktrate_in == 0) { // avoid divide by zero
    pktrate_avg += 0;
  } else {
    pktrate_avg += ((float)size / (float)pktrate_in) * 1000.00 * 8.00;
  }
  avg_counter++;
  if(avg_counter > 8) {
    emit(update_pktrate(pktrate_avg / avg_counter));
    pktrate_avg = 0;
    avg_counter = 0;
  }
}


