#ifndef ANALYZER_H
#define ANALYZER_H

#include <QApplication>
#include <QGraphicsScene>
#include <QWidget>
#include "config.h"
#include "settings.h"
#include "datalog.h"

namespace Ui {
class analyzer;
}

class fcell { // floating point analysis
public:
  fcell();
  void clear();
  //--------
  float low;
  float high;
  int count;
  float avg(); // calculate average
  //--------
  void add_data(float value);
  void add_data(float value, float min, float max); //reject data outside threshold
private:
  float average_accumulator;
};

struct knockevent_t {
  int rpm;
  int map;
  int count;
  bool wot;
  int spark;
  int kr;
  int interval;
  timestamp_t timestamp;
};

class analysis : public QObject {
  Q_OBJECT
public:
  analysis();
  void reset_analyzer();
  void analyze_segment(datalog_packet *p);
  // input parameters
  int min_cooltmp;
  int min_timestamp;
  int min_afrcounts;
  bool use_integrator;
  bool anl_filter_idleanddecel;
  bool wideband;
  bool wbcl;
  bool wbpercent;
  bool worst_case_blm;
  bool enhanced_accuracy;
  // oxygen sensors
  fcell lo2, ro2; // o2 sensor average voltage
  int lcrosscount, rcrosscount; // cross counts
  int lstuck, rstuck; // stuck o2 counts
  // knock chart
  int total_knock_events;
  unsigned long int total_knock_counts;
  QVector <knockevent_t>knock_list;
  // VE fuel
  fcell ve_trim[25][17]; // 0-7000rpm/0-100 map cells
  fcell maf_trim[70];
  fcell idle_trim[2];
  fcell idle_map;
  fcell idle_maf;
  // PE fuel
  fcell pe_fuel[28][3]; // 0=<70 1=70-90 2=>90
  // blm avg
  fcell blm[19][4]; // 0=rpm 1=map 2=l_trim 3=r_trim
  // table configs...
  QVector <int>ve_map_list;
  QVector <int>ve_rpm_list;
  QVector <int>maf_afgs_list;
private:
  datalog_packet *parse_record; // current working record
  eehack_settings config;
  // internal analyzers
  void analyze_o2();
  void analyze_knock();
  void analyze_nb_afr();
  void analyze_wb_afr();
  void analyze_pe();
  void analyze_idle();
  void analyze_blm();
  // internal states
  int lstate, rstate; // states
  int select_cell(int range, int cells, float data);
  int select_cell(QVector<int>list, float data);
  bool packet_valid(datalog_packet *p);


};

class analyzer_ui : public QWidget {
  Q_OBJECT

public:
  explicit analyzer_ui(datalog *log_in, QWidget *parent = 0);
  ~analyzer_ui();
  void on_interface_loaded();
  datalog *log;
public slots:
  void wideband_config_updated();
  void def_parse_err_in(QString str);
private:
  Ui::analyzer *ui;
  analysis *anl;
  eehack_settings config;
  //--------------------
  void draw_analysis_results();
  void draw_results_o2();
  void draw_results_nb_afr();
  void draw_results_wb_afr();
  void draw_results_pe();
  void draw_results_blm();
  void config_afr_ve_table();
  void draw_results_idle();
  void label_anl_graphs();
  QString percent_to_str(double x);
  float trim_to_percent(int trim);
  QGraphicsScene *knock_scene;
  void draw_knock_grid(int width, int height);
  void add_knock_point(int width, int height, knockevent_t e);
  void resizeEvent(QResizeEvent *event);
signals:
  void info_dialog(QString);
  void error_dialog(QString);
private slots:
  //--------------------
  void on_analyzeDataBtn_clicked();
  void on_anl_mincooltmp_slider_valueChanged(int value);
  void on_anl_mintime_slider_valueChanged(int value);
  void on_anl_afrcounts_slider_valueChanged(int value);
  void on_setting_afrblmavg_toggled(bool checked);
  void on_setting_anl_extracells_toggled(bool checked);
  void on_setting_anl_int_toggled(bool checked);
  void on_setting_wb_cl_toggled(bool checked);
  void on_setting_enh_accuracy_toggled(bool checked);
  void on_afr_input_narrow_toggled(bool checked);
  void on_afr_input_wide_toggled(bool checked);
  void on_setting_display_wb_percent_toggled(bool checked);
  void draw_results_knock();
};




#endif // ANALYZER_H
