#include "controller.h"
#include "ui_controller.h"
#include "common.h"
#include <QCloseEvent>

controller::controller(datastream_control *control_in, datalog *log_in, QWidget *parent = 0) :
  QWidget(parent),
  ui(new Ui::controller) {
  ui->setupUi(this);
  control = control_in;
  //--------
  aspk_ui = new autospark;
  connect(aspk_ui,SIGNAL(send_new_advance(int)),this,SLOT(autospark_new_advance(int)));
  connect(this,SIGNAL(send_autospark_parameters(datalog_packet*,datastream_control*)),aspk_ui,SLOT(recv_new_sample(datalog_packet*,datastream_control*)));
  //--------
  log = log_in;
  commanded_idle = 0;
}

void controller::on_interface_loaded() {
  update_m4_raw_display();
  // toggle timing limit
  ui->setting_limit_timing->setChecked(false);
  ui->setting_limit_timing->setChecked(true);
  // initalize slider to display stuff (loop existing value back into 'change' signal)
  // ... this ensures a new default value placed on a ui slider will also upd. the display
  on_slider_linepressure_valueChanged(ui->slider_linepressure->value());
  on_slider_afr_valueChanged(ui->slider_afr->value());
  on_slider_idle_sliderMoved(ui->slider_idle->value());
  on_slider_sparkmod_valueChanged(ui->slider_sparkmod->value());
  on_slider_egr_valueChanged(ui->slider_egr->value());
  on_btn_dangerous_toggled(false);
}

controller::~controller() {
  delete aspk_ui;
  delete ui;
}

void controller::reset_controls_default() {
  ui->ac_swauto->setChecked(true);
  ui->air_swauto->setChecked(true);
  ui->autospark_box->setChecked(false);
  ui->blm_swauto->setChecked(true);
  ui->ccp_swauto->setChecked(true);
  ui->check_forceafr->setChecked(false);
  ui->cylcut_0->setChecked(true);
  ui->fan1_swauto->setChecked(true);
  ui->fan2_swauto->setChecked(true);
  ui->idle_override->setChecked(false);
  ui->loop_swauto->setChecked(true);
  ui->modify_spark_box->setChecked(false);
  ui->shift_auto->setChecked(true);
  ui->tcc_auto->setChecked(true);
  ui->check_linepressure->setChecked(false);
}

void controller::closeEvent(QCloseEvent *x) {
  reset_controls_default();
  x->accept();
}

void controller::recieve_new_packet(datalog_packet *p) {
  if(ui->autospark_box->isChecked() == true) {
    emit send_autospark_parameters(p,control);
  }
}

void controller::on_fan1_swauto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_fan(1,false,false);
  update_m4_raw_display();
}

void controller::on_fan1_swon_toggled(bool checked) {
  if(checked == false) return;
  control->m4_fan(1,true,true);
  update_m4_raw_display();
}

void controller::on_fan1_swoff_toggled(bool checked) {
  if(checked == false) return;
  control->m4_fan(1,false,true);
  update_m4_raw_display();
}

void controller::on_fan2_swauto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_fan(2,false,false);
  update_m4_raw_display();
}

void controller::on_fan2_swon_toggled(bool checked) {
  if(checked == false) return;
  control->m4_fan(2,true,true);
  update_m4_raw_display();
}

void controller::on_fan2_swoff_toggled(bool checked) {
  if(checked == false) return;
  control->m4_fan(2,false,true);
  update_m4_raw_display();
}

void controller::on_loop_swauto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_cl(false,false);
  update_m4_raw_display();
}

void controller::on_loop_swopen_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_cl(false,true);
  update_m4_raw_display();
}

void controller::on_loop_swclosed_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_cl(true,true);
  update_m4_raw_display();
}

void controller::on_blm_swauto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_blm(false,false);
  update_m4_raw_display();
}

void controller::on_blm_swon_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_blm(true,true);
  update_m4_raw_display();
}

void controller::on_blm_swoff_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_blm(false,true);
  update_m4_raw_display();
}

void controller::on_ccp_swoff_toggled(bool checked) {
  if(checked == false) return;
  control->m4_ccp(false,true);
  update_m4_raw_display();
}

void controller::on_ccp_swauto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_ccp(false,false);
  update_m4_raw_display();
}

void controller::on_ccp_swon_toggled(bool checked) {
  if(checked == false) return;
  control->m4_ccp(true,true);
  update_m4_raw_display();
}

void controller::on_ac_swauto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_ac(false,false);
  update_m4_raw_display();
}

void controller::on_ac_swon_toggled(bool checked) {
  if(checked == false) return;
  control->m4_ac(true,true);
  update_m4_raw_display();
}

void controller::on_ac_swoff_toggled(bool checked) {
  if(checked == false) return;
  control->m4_ac(false,true);
  update_m4_raw_display();
}

void controller::on_air_swauto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_air(false,false);
  update_m4_raw_display();
}

void controller::on_air_swon_toggled(bool checked) {
  if(checked == false) return;
  control->m4_air(true,true);
  update_m4_raw_display();
}

void controller::on_air_swoff_toggled(bool checked) {
  if(checked == false) return;
  control->m4_air(false,true);
  update_m4_raw_display();
}

void controller::update_idle_mode4() {
  if(ui->idle_override->isChecked() == true) { // overridden
    if(ui->check_usesteps->isChecked() == true) { // step mode
       control->m4_comm_idle(commanded_idle,0);
    } else { //rpm mode
       control->m4_comm_idle(commanded_idle,1);
    }
  }
  update_m4_raw_display();
}

void controller::on_slider_idle_sliderMoved(int position) {
  // update linked text box
  if(ui->check_usesteps->isChecked() == true) { // step mode
    commanded_idle = position;
    ui->entry_idle->setText(QString("%1").arg(commanded_idle));
  } else {
    commanded_idle = position * 12.5;
    ui->entry_idle->setText(QString("%1").arg(commanded_idle));
  }
  update_idle_mode4();
}

void controller::on_check_usesteps_toggled(bool) {
  if(ui->idle_override->isChecked() == true) {
    update_idle_mode4();
  }
}

void controller::on_idle_override_toggled(bool checked) {
  if(checked == true) {
    update_idle_mode4();
  } else {
    control->m4_comm_idle(0,0);
    update_idle_mode4();
  }
}

void controller::on_slider_sparkmod_valueChanged(int value) {
  if(ui->modify_spark_box->isChecked() == true) {
    control->m4_comm_spark(value,0);
    update_m4_raw_display();
  }
  ui->display_sparkmod->setText(QString("%1").arg(value));
}

void controller::on_slider_afr_valueChanged(int value) {
  ui->display_afr->setText(QString("%1:1").arg((float)(value) / 10));
  if(ui->check_forceafr->isChecked() == true) {
    control->m4_comm_afr(value);
    update_m4_raw_display();
  }
}

void controller::on_check_forceafr_toggled(bool checked) {
  static int blm_setting_memory = 0;
  static int loop_setting_memory = 0;
  if(checked == true) {
    // cylcut force 0 and disable
    ui->cylcut_0->setEnabled(true);
    ui->grp_cylcut->setEnabled(false);
    //--------
    blm_setting_memory = ui->btngroup_blm->checkedId(); // remember blm
    loop_setting_memory = ui->btngroup_loop->checkedId(); // remember loop
    ui->loop_swopen->setChecked(true);
    ui->blm_swoff->setChecked(true);
    if(ui->btn_dangerous->isChecked() == false) enable_clblm_controls(false);
    //-------
    ui->clear_blm_btn->click();
    //-------
    control->m4_comm_afr(ui->slider_afr->value());
    update_m4_raw_display();
  } else { // disable
    // recover old settings
    ui->btngroup_blm->button(blm_setting_memory)->setChecked(true);
    ui->btngroup_loop->button(loop_setting_memory)->setChecked(true);
    enable_clblm_controls(true);
    //-------
    control->m4_comm_afr(0);
    // cylcut re-enable
    ui->grp_cylcut->setEnabled(true);
    update_m4_raw_display();
  };
}

void controller::enable_clblm_controls(bool value) {
  ui->loop_swauto->setEnabled(value);
  ui->loop_swclosed->setEnabled(value);
  ui->loop_swopen->setEnabled(value);
  ui->blm_swauto->setEnabled(value);
  ui->blm_swon->setEnabled(value);
  ui->blm_swoff->setEnabled(value);
}

void controller::on_autospark_box_toggled(bool checked) {
  if(checked == true) {
    //nothing
  } else {
    //nothing
  }
}

void controller::on_btn_autospark_edit_clicked() {
  if(aspk_ui->isVisible()) {
    aspk_ui->hide();
  } else {
    aspk_ui->show();
  }
}

void controller::on_setting_limit_timing_toggled(bool checked) {
   if(checked == true) {
     ui->label_min_spk->setText("-5");
     ui->label_max_spk->setText("+5");
     ui->slider_sparkmod->setValue(0);
     ui->slider_sparkmod->setRange(-5,5);
   } else {
     ui->label_min_spk->setText("-15");
     ui->label_max_spk->setText("+15");
     ui->slider_sparkmod->setValue(0);
     ui->slider_sparkmod->setRange(-15,15);
   }
}

void controller::on_clear_blm_btn_clicked() {
  control->m4_reset_blm(true);
  update_m4_raw_display();
}

void controller::on_modify_spark_box_toggled(bool checked) {
  if(checked == false) { //disable
    control->m4_comm_spark(0,0);
    update_m4_raw_display();
  } else {
    control->m4_comm_spark(ui->slider_sparkmod->value(),0);
    update_m4_raw_display();
  }
}

void controller::autospark_new_advance(int advance) {
  if(ui->autospark_box->isChecked() == false) return;
  ui->slider_sparkmod->setValue(advance);
}

void controller::update_m4_raw_display() {
  ui->m4_string->setText("RAW " + control->m4_getpkt());
}

void controller::on_shift_auto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_gear(0);
  update_m4_raw_display();
}

void controller::on_shift_1_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_gear(1);
  update_m4_raw_display();
}

void controller::on_shift_2_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_gear(2);
  update_m4_raw_display();
}

void controller::on_shift_3_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_gear(3);
  update_m4_raw_display();
}

void controller::on_shift_4_toggled(bool checked) {
  if(checked == false) return;
  control->m4_force_gear(4);
  update_m4_raw_display();
}

void controller::on_tcc_auto_toggled(bool checked) {
  if(checked == false) return;
  control->m4_tcc(false,false);
  update_m4_raw_display();
}

void controller::on_tcc_off_toggled(bool checked) {
  if(checked == false) return;
  control->m4_tcc(false,true);
  update_m4_raw_display();
}

void controller::on_tcc_on_toggled(bool checked) {
  if(checked == false) return;
  control->m4_tcc(true,true);
  update_m4_raw_display();
}

void controller::drop_cyl(int cylid) {
  if(cylid == 0) {
    ui->check_forceafr->setEnabled(true);
  } else {
    ui->check_forceafr->setEnabled(false);
  }
  control->m4_drop_cyl(cylid);
  update_m4_raw_display();
}

void controller::on_cel_btn_toggled(bool checked) {
  control->m4_mil(checked);
  update_m4_raw_display();
}

void controller::on_cylcut_0_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(0);
}

void controller::on_cylcut_1_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(1);
}

void controller::on_cylcut_2_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(2);
}

void controller::on_cylcut_3_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(3);
}

void controller::on_cylcut_4_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(4);
}

void controller::on_cylcut_5_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(5);
}

void controller::on_cylcut_6_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(6);
}

void controller::on_cylcut_7_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(7);
}

void controller::on_cylcut_8_toggled(bool checked) {
  if(checked == false) return;
  drop_cyl(8);
}

//-----------------------

void controller::append_memo(QString str) {
  emit send_new_note(str);
}

void controller::on_btn_snapshop_idle_clicked() {
  QString str; //idle_override entry_idle check_usesteps
  if(log->is_data() == false) return; // no data
  if(ui->idle_override->isChecked() == false) return; // not enabled
  datalog_packet *p = log->latest_packet(0x00);
  str += "------------------\n";
  str += "IDLE Memo :: ";
  if(ui->check_usesteps->isChecked() == true) { // step mode
    str += QString(" IDLE=%1 STEPS @ ").arg(ui->entry_idle->text());
  } else {
    str += QString(" IDLE=%1RPM @ ").arg(ui->entry_idle->text());
  }
  str += QString(" COOLTMP=%1").arg(p->get_float("COOLTMP"));
  str += QString(" MAP=%1").arg(p->get_float("MAP"));
  if(ui->modify_spark_box->isChecked() == true) {
    str += QString(" SPKTRIM=%1").arg(ui->slider_sparkmod->value());
  }
  append_memo(str);
}

void controller::on_btn_snapshop_spark_clicked() {
  QString str;
  if(log->is_data() == false) return; // no data
  if(ui->modify_spark_box->isChecked() == false) return; // not enabled
  datalog_packet *p = log->latest_packet(0x00);
  str += "------------------\n";
  str += "SPARKTRIM Memo :: ";
  str += QString(" SPKTRIM=%1 @ ").arg(ui->slider_sparkmod->value());
  str += QString(" RPM=%1").arg(p->get_float("RPM"));
  str += QString(" MAP=%1").arg(p->get_float("MAP"));
  str += QString(" SPARKOUPUT=%1").arg(p->get_int("SPKADV"));
  append_memo(str);
}

void controller::on_btn_snapshop_afr_clicked() {
  QString str;
  if(log->is_data() == false) return; // no data
  if(ui->check_forceafr->isChecked() == false) return; // not enabled
  datalog_packet *p = log->latest_packet(0x00);
  str += "------------------\n";
  str += "AFR Memo :: ";
  str += QString(" AFR=%1 @ ").arg(ui->display_afr->text());
  str += QString(" COOLTMP=%1").arg(p->get_float("COOLTMP"));
  str += QString(" RPM=%1").arg(p->get_float("RPM"));
  str += QString(" MAP=%1").arg(p->get_float("MAP"));
  append_memo(str);
}

void controller::on_check_linepressure_toggled(bool checked) {
  control->m4_comm_linepressure(checked,ui->slider_linepressure->value());
  update_m4_raw_display();
  ui->group_egr->setEnabled(!checked); // disable egr group to avoid conflict
}

void controller::on_slider_linepressure_valueChanged(int value) {
  // upd. text display always
  double percentage = (double)value / 96 * 100.00;
  ui->display_linepressure->setText(QString::number(percentage,'f',1) + "%");
  // upd. mode4
  if(ui->check_linepressure->isChecked() == true) {
    control->m4_comm_linepressure(true,ui->slider_linepressure->value());
    update_m4_raw_display();
  }
}

void controller::on_check_egr_toggled(bool checked) {
  control->m4_comm_egr(checked,ui->slider_egr->value());
  update_m4_raw_display();
  ui->display_linepressure->setEnabled(!checked);
  ui->slider_linepressure->setEnabled(!checked);
  ui->check_linepressure->setEnabled(!checked);
}

void controller::on_slider_egr_valueChanged(int value) {
  // upd. text display always
  double percentage = (double)value / 255.00 * 100.00;
  ui->display_egr->setText(QString::number(percentage,'f',1) + "%");
  // upd. mode4
  if(ui->check_egr->isChecked() == true) {
    control->m4_comm_egr(true,ui->slider_egr->value());
    update_m4_raw_display();
  }
}

void controller::on_btn_dangerous_toggled(bool checked) {
  if(checked == false) {
    // disable linepressure
    ui->check_linepressure->setChecked(false);
    ui->check_linepressure->hide();
    ui->display_linepressure->hide();
    ui->slider_linepressure->hide();
    // timing limit
    ui->setting_limit_timing->setChecked(true);
    ui->setting_limit_timing->hide();
    // raw display
    ui->m4_string->hide();
    // afr slider
    ui->slider_afr->setMinimum(120);
    ui->slider_afr->setMaximum(160);
  } else {
    // enable linepressure
    ui->check_linepressure->show();
    ui->display_linepressure->show();
    ui->slider_linepressure->show();
    // timing limit
    ui->setting_limit_timing->show();
    // raw display
    ui->m4_string->show();
    // afr slider
    ui->slider_afr->setMinimum(90);
    ui->slider_afr->setMaximum(200);
  }
}
