#include <QMessageBox>
#include <QFileDialog>
#include <QProgressDialog>
#include "datalog_window.h"
#include "ui_datalog_window.h"
#include "config.h"
#include "autospark.h"
#include "ui_autospark.h"
#include "datalog.h"

QString datalog_window::get_save_filename(QString extension, QString type_name) {
  QFileDialog dialog;
  dialog.setFileMode(QFileDialog::AnyFile);
  dialog.setDefaultSuffix(extension);
  dialog.setNameFilter(type_name + " (*." + extension + ")");
  dialog.setAcceptMode(QFileDialog::AcceptSave);
  dialog.setDirectory(config.get_savepath());
  QStringList file_names;
  if(dialog.exec()) {
    file_names = dialog.selectedFiles();
    return file_names.takeFirst();
  } else {
    return QString();
  }
}

bool datalog_window::save_raw_file(datalog *log_out) {
  // check if ok to save
  if(log_out->is_data() == false) {
    error_dialog("No data to save!");
    return false;
  }
  // get filename
  QString filename;
  filename = get_save_filename("eedata","EEData Log File"); // get filename
  if(filename.isEmpty() == true || filename.isNull() == true) return false; // cancelled
  // open file
  QFile f(filename);
  f.open( QIODevice::WriteOnly );
  if(f.isWritable() == false) {
    error_dialog("Could not open file for writing.");
    return false;
  }
  QDataStream data_out(&f); // stream output
  last_save_filename = filename;
  // progress
  QProgressDialog progress("Exporting data...","ABORT",0,log_out->n_packets());
  progress.setWindowModality(Qt::WindowModal);
  // header
  data_out << (quint32)LOG_MAGIC_NUMBER; // identifier, 32 bit unsigned
  data_out << (quint8)log_out->info.patch_version.get(); // patch version, 8 bit unsigned
  data_out << (quint8)MAX_M1_DATA_SIZE; // m1 data fixed size
  data_out << (quint32)log_out->n_packets();
  data_out << (QString)log_out->info.vin_number.get();
  data_out << (quint32)log_out->info.calibration_id.get();
  data_out << (bool)log_out->info.auto_trans.get();
  data_out << (bool)log_out->info.eside_comms_basic.get();
  data_out << (bool)log_out->info.maf_enable.get();
  // data
  datalog_packet *pkt = log_out->oldest_packet();
  while(pkt != nullptr) {
    data_out << (quint8)pkt->def->get_msg_number(); // first byte, msg number
    data_out << (quint8)pkt->def->get_device(); // second byte, msg number
    data_out << (timestamp_t)pkt->timestamp; // timestamp
    for(int x=0;x<MAX_M1_DATA_SIZE;x++) { // raw data
      data_out << (quint8)pkt->get_raw_byte(x);
    }
    progress.setValue(pkt->sequence);
    pkt = pkt->get_next();
    if(progress.wasCanceled()) {
      f.remove();
      f.close();
      progress.setValue(log_out->n_packets());
      return false;
    }
  }
  data_out << LOG_MAGIC_NUMBER;
  progress.setValue(log_out->n_packets());
  unsaved_log = false;
  return true;
}

QStringList datalog_window::get_log_select_dialog() {
  QStringList filename = QFileDialog::getOpenFileNames(
        this,"Select raw data file(s) to load",
              config.get_savepath(),"EEData Log File (*.eedata)");
  return filename;
}

bool datalog_window::load_raw_file(QString filename, datalog *new_log) {
  if(filename.isEmpty() == true || filename.isNull() == true) return false;
  // check suffix
  QFileInfo file_info(filename);
  if(QString::compare(file_info.suffix(),QStringLiteral("eedata")) != 0) return false;
  QFile f(filename);
  if(f.exists() == false) return false;
  f.open( QIODevice::ReadOnly );
  if(f.isReadable() == false) {
    info_dialog("Could not open file.");
    return false;
  }
  QDataStream data_in(&f);
  // get header
  quint32 header_id;
  data_in >> header_id;
  if(header_id != LOG_MAGIC_NUMBER) {
    //error_dialog("This log file isn't compatible with this version of EEHack...");
    error_dialog("This log MAY be an old obselete format.  We're going to try to load it anyway.\nThis might go away in future versions.  If it crashes now, it's probably no good.");
    f.close();
    return load_old_file(filename,new_log);;
  }
  // patch version
  quint8 patch_version;
  data_in >> patch_version;
  new_log->info.patch_version.set(patch_version);
  // message size
  quint8 data_size;
  data_in >> data_size;
  if(data_size > MAX_M1_DATA_SIZE) {
    error_dialog("This log file isn't compatible with this build of EEHack, due to a difference in MAX_M1_DATA_SIZE...");
    return false;
  }
  // n packets
  quint32 n_packets;
  data_in >> n_packets;
  // info struct
  QString vin_in;
  data_in >> vin_in;
  new_log->info.vin_number.set(vin_in);
  quint32 calid_in;
  data_in >> calid_in;
  new_log->info.calibration_id.set(calid_in);
  bool autotrans_in;
  data_in >> autotrans_in;
  new_log->info.auto_trans.set(autotrans_in);
  bool eside_in;
  data_in >> eside_in;
  new_log->info.eside_comms_basic.set(eside_in);
  bool maf_in;
  data_in >> maf_in;
  new_log->info.maf_enable.set(maf_in);
  new_log->info.valid.enable();
  QProgressDialog progress("Importing data...","ABORT",0,n_packets);
  progress.setWindowModality(Qt::WindowModal);
  for(unsigned int x=0;x<n_packets;x++) {
    unblock_ui();
    if(data_in.atEnd() == true) { // reached end prematurely
      error_dialog("Corrupt log - premature end of file");
      return false;
    }
    progress.setValue(x);
    quint8 msg_n;
    quint8 dev_n;
    data_in >> msg_n;
    data_in >> dev_n;
    datalog_definition *defn = def->get_msg_number((int)dev_n,(int)msg_n);
    if(defn == nullptr) {
      error_dialog("Log contains undefined mesasges or devices.  Can't load!");
      return false;
    }
    datalog_packet *pkt = new datalog_packet(defn,new_log,MAX_M1_DATA_SIZE); // create packet
    timestamp_t tstamp;
    data_in >> tstamp;
    pkt->timestamp = tstamp;
    for(int y=0;y<data_size;y++) { // raw data
      quint8 raw_byte_in;
      data_in >> raw_byte_in;
      pkt->set_raw_byte(y,raw_byte_in);
    }
    for(int y=data_size;y<MAX_M1_DATA_SIZE;y++) { // zero-fill tail if necessary
      pkt->set_raw_byte(y,0x00);
    }
    new_log->add_packet(pkt);
    if(progress.wasCanceled() == true) {
      return false;
    }
  }
  progress.setValue(n_packets);
  return true;
}

void datalog_window::load_log_from_file(QString filename) {
  if(filename.isEmpty() == true) return;
  if(filename.isNull() == true) return;
  QFileInfo f(filename);
  QString short_name(f.fileName());
  //create new log
  datalog *new_log = new datalog(short_name,filename,nullptr);
  // load data
  if(load_raw_file(filename,new_log) == false) { // load fail
    delete new_log;
  } else { // load success
    add_new_log(new_log);
  }
}

bool datalog_window::load_old_file(QString filename, datalog *new_log) {
  if(filename.isEmpty() == true || filename.isNull() == true) return false;
  // check suffix
  QFileInfo file_info(filename);
  if(QString::compare(file_info.suffix(),QStringLiteral("eedata")) != 0) return false;
  QFile f(filename);
  if(f.exists() == false) return false;
  f.open( QIODevice::ReadOnly );
  if(f.isReadable() == false) {
    info_dialog("Could not open file.");
    return false;
  }
  bool success = true;
  QTextStream data_in(&f);
  QProgressDialog progress("Importing data...","ABORT",0,10000);
  progress.setWindowModality(Qt::WindowModal);
  timestamp_t last_timestamp = 0;
  forever {
    unblock_ui();
    QString line_in = data_in.readLine(0);
    if(line_in.isNull() == true) break; // end of file
    QStringList fields = line_in.split('+');
    if(fields.size() != 2) {
      info_dialog("The file appears to be corrupt.");
      success = false;
      goto RAW_READ_FAIL;
    }
    datalog_packet *pkt = new datalog_packet(&def[0],new_log,MAX_M1_DATA_SIZE);

    // timestamping... a mess but it works!
    pkt->timestamp = fields.at(0).toLongLong() - last_timestamp; // offset from last pkt
    if(pkt->timestamp < 0 || pkt->timestamp > LOG_TIMESTAMP_LIMIT) {
      error_dialog("We've encountered bad timestamps in the log, which means it's probably corrupt.\nWill stop loading now, but some records may have already been added.");
      delete pkt;
      success = false;
      goto RAW_READ_FAIL;
    }
    last_timestamp = pkt->timestamp + last_timestamp;
    pkt->timestamp += new_log->newest_timestamp(); // add this timestamp

    QString hex_data = fields.at(1);
    int h_idx = 0;
    for(int x=0;x<=60;x++) {
      bool ok;
      QStringRef hex_chars(&hex_data, h_idx, 2);
      h_idx += 2;
      pkt->set_raw_byte(x,hex_chars.toUInt(&ok,16));
      if(ok == false) {
        info_dialog("While loading the log, we've encountered corrupt data.\nWill stop loading now, but some records may have already been added.");
        delete pkt;
        success = false;
        goto RAW_READ_FAIL;
      }
    }

    // link packet to list
    new_log->add_packet(pkt);

    if(progress.wasCanceled() == true) {
      success = false;
      goto RAW_READ_FAIL;
    }
    if(pkt->sequence < 10000) progress.setValue(pkt->sequence);
  }
  RAW_READ_FAIL:
  progress.setValue(10000);
  f.close();
  return success;
}

