#include <QMessageBox>
#include <QFileDialog>
#include "datalog_window.h"
#include "config.h"
#include "datalog.h"
#include "datastream.h"
#include "common.h"

void datastream::cyl_drop_test() {
  if(control->is_connected() == false) return;

  emit new_cylbalance_progress(" ",0);

  datalog_packet *p = log->latest_packet(0x00);
  if(p->get_float("TPS") > 0.25) {
    emit display_critical_error("Can't run the cylinder drop test with the throttle open.");
    return;
  }
//  if(p->get_rpm() < 200) {
//    emit display_critical_error("Can't run the cylinder drop test without the engine running.");
//    return;
//  }

  int idle_steps = p->get_float("IAC");
  int spkadv = p->get_int("SPKADV");

  // remember old mode4 packet
  byte old_m4_command[18];
  control->m4_get_raw(old_m4_command);

  // lock idle and blms
  emit new_cylbalance_progress("Locking IAC/SPK/BLM...",1);
  control->m4_comm_init();
  control->m4_force_blm(true,false); // lock blms to off
  control->m4_reset_blm(true);
  control->m4_comm_spark(clamp(spkadv,18,32),1);
  control->m4_comm_idle(clamp(idle_steps + 20,10,160),2); // lock idle steps
  if(m4_send_if_requested() == false) { // send request
    emit display_critical_error("Due to a communication problem, the cylinder drop test could not continue.");
    emit new_cylbalance_progress(" ",0);
    control->m4_set_raw(old_m4_command);
    m4_send_if_requested();
    return;
  }
  control->m4_reset_blm(false);

  // engine settling
  emit new_cylbalance_progress("Waiting for engine to settle...",2);
  for(int x=0;x<5;x++) {
    m4_send_if_requested(); // keep-alive
    msleep(500);
  };

  emit new_cylbalance_progress("Testing for reference...",3);
  float ref = get_rpm_map_diff(CYL_DROP_TESTLENGTH);
  if(control->quit.is_set() == true) return;
  if(ref == 9999) {
    emit display_critical_error("Due to a communication problem, the cylinder drop test could not continue.");
    emit new_cylbalance_progress(" ",0);
    control->m4_set_raw(old_m4_command);
    m4_send_if_requested();
    return;
  }

  // drop each cyl and get map average
  int max_str = 0;
  float strength_arr[8];
  for(int x=1;x<=8;x++) {
    emit new_cylbalance_progress(QString("Testing Cyl#%1").arg(x),3 + x);
    control->m4_drop_cyl(x); // drop cyl
    if(m4_send_if_requested() == false) { // send request
      emit display_critical_error("Due to a communication problem, the cylinder drop test could not continue.");
      emit new_cylbalance_progress(" ",0);
      control->m4_set_raw(old_m4_command);
      m4_send_if_requested();
      return;
    }
    float test_result = get_rpm_map_diff(CYL_DROP_TESTLENGTH) - ref;
    if(control->quit.is_set() == true) return;
    if(test_result == 9999) {
      emit display_critical_error("Due to a communication problem, the cylinder drop test could not continue.");
      emit new_cylbalance_progress(" ",0);
      control->m4_set_raw(old_m4_command);
      m4_send_if_requested();
      return;
    }
    test_result -= ref;
    strength_arr[x - 1] = test_result;
    if(test_result > max_str) max_str = test_result; // update max
  }

  // clamp max_str to 1
  if(max_str < 1) max_str = 1;

  // reset to initial mode4
  emit new_cylbalance_progress("Resetting Parameters...",12);
  control->m4_set_raw(old_m4_command);
  if(m4_send_if_requested() == false) {
    emit display_critical_error("Due to a communication problem, your original settings could not be restored!  Reccommend program restart.");
    return;
  }

  // emit results
  for(int x=1;x<=8;x++) {
    emit new_cylbalance_result(x,max_str,strength_arr[x - 1]);
  }

  emit new_cylbalance_progress("Test Complete!",13);
}

float datastream::get_rpm_map_diff(int quality) {
  float avg = 0;
  int x;
  msleep(800);
  int hangcheck = 0;
  for(x=1;x<=quality;x++) { // get 'quality' n packets
    if(control->quit.is_set() == true) return 9999;
    while(get_main_data_packet() == false) {
      hangcheck++;
      if(hangcheck > 10) return 9999;
    }
    datalog_packet *p = log->latest_packet(0x00);
    avg += ( (float)p->get_float("RPM") / 100.00) - p->get_float("MAP");
  }
  avg = avg / quality;
  return avg;
}


