#include <QNetworkAccessManager>
#include <QNetworkConfigurationManager>
#include <QUrl>
#include <QNetworkRequest>
#include <QNetworkReply>
#include "update_notify.h"
#include "debuglog_window.h"
#include "config.h"

update_notify::update_notify(QObject *parent) : QObject(parent) {
  version_retr = new QNetworkAccessManager(this);
  connect(version_retr, SIGNAL(finished(QNetworkReply*)),
          this, SLOT(http_version_request_finished(QNetworkReply*)));
}

update_notify::~update_notify() {
  delete version_retr;
  return;
}

void update_notify::recheck_version() {
  fetch_update_version();
}

void update_notify::set_update_status(int stage) {
  switch(stage) {
  case 0:
    emit update_network_status(" ");
    break;
  case 1:
    emit update_network_status("Requesting version...");
    break;
  case 2:
    emit update_network_status("Network error!  (See Log)");
    break;
  case 3:
    emit update_network_status("Version check succeeded.");
  }
}

void update_notify::fetch_update_version() {
  set_update_status(0);
  debuglog_controller debug;
  set_update_status(1);
  debug.print("VERSION: Re-checking, network went online!");
  QString update_url = EEHACK_UPDATE_URL;
  QNetworkRequest net_req(update_url);
  version_retr->get(QNetworkRequest(QUrl(update_url)));
}

void update_notify::http_version_request_finished(QNetworkReply *reply) {
  debuglog_controller debug;
  if(reply->error()) { // err
    debug.print("VERSION ERR: " + reply->errorString());
    set_update_status(2);
  } else { // success
    process_http_version(reply->readAll());
  }
  reply->deleteLater();
}

void update_notify::process_http_version(QByteArray raw_reply) {
  debuglog_controller debug;
  //verify
  if(raw_reply.length() < 12) {
    debug.print("VERSION: Wrong length: " + QString::number(raw_reply.length()));
    set_update_status(2);
    return;
  }

  QByteArray v_header("VERSION");
  for(int x=0;x<v_header.length();x++) {
    if(raw_reply.at(x) != v_header.at(x)) {
      debug.print("VERSION: Header is incorrect at byte " + QString::number(x));
      set_update_status(2);
      return;
    }
  }
  // convert
  char c[5]; // 0-4
  for(int x=0;x<4;x++) c[x] = raw_reply.at(x + v_header.length());
  c[4] = 0; // terminate
  int version = atoi(c);
  // set
  emit new_version_recieved(version);
  debug.print("VERSION: Fetched version from network.");
  set_update_status(3);
}
