#include <QApplication>
#include <QMainWindow>
#include <QtDebug>
#include <QList>

#include "datastream.h"
#include "datalog.h"
#include "config.h"
#include "bin_file.h"
#include "common.h"

// patch details
#define PATCH_VERSION_BYTE 0x3CB3

// table defs
#define VE_TABLE_LOW_OFFSET 0x2BEE
#define VE_TABLE_HIGH_OFFSET 0x2C8A

bin_file::bin_file() {
  // zero out the bin
  for(int x=0x00;x<=0xFFFF;x++) {
    eside[x] = 0x00;
    tside[x] = 0x00;
  }
}

void bin_file::zero_ram() {
  for(int x=0;x<0x0E00;x++) { // before aux eeprom
    tside[x] = 0x00;
    eside[x] = 0x00;
  }
  for(int x=0x1001;x<0x2000;x++) { // between eeprom and main eeprom
    tside[x] = 0x00;
    eside[x] = 0x00;
  }
}

bool bin_file::test_checksum(byte *data) {
  quint64 temp_sum = 0;
  for(int x=0x2018;x<=0xffff;x++) temp_sum += data[x];
  quint16 t_sum = (quint16)temp_sum;
  byte low_byte = t_sum & 0xFF;
  byte high_byte = t_sum >> 8;
  if(data[0x2015] == high_byte &&
     data[0x2016] == low_byte) return true;
  return false;
}

bool bin_file::test_checksum() {
  if(test_checksum(tside) == false) return false;
  if(test_checksum(eside) == false) return false;
  return true;
}

void bin_file::reset_checksums() {
  set_checksum(tside);
  set_checksum(eside);
}

bool bin_file::set_checksum(byte *data) {
  quint64 temp_sum = 0;
   for(int x=0x2018;x<=0xffff;x++) temp_sum += data[x];
   quint16 t_sum = (quint16)temp_sum;
   byte low_byte = t_sum & 0xFF;
   byte high_byte = t_sum >> 8;
   data[0x2015] = high_byte;
   data[0x2016] = low_byte;
   return true;
}

bool bin_file::load_file(QString path) {
  QFile f(path);
  if(f.exists() == false) return false;
  f.open( QIODevice::ReadOnly );
  if(f.isReadable() == false) { // can't open
    return false;
  }
  QByteArray unsplit_bin = f.readAll();
  if(f.size() != 0x20000) return false;
  for(int x=0;x<=0xFFFF;x++) {
    tside[x] = unsplit_bin.at(x);
    eside[x] = unsplit_bin.at(x + 0x10000);
  }
  if(test_checksum() == false) return false;
  return true;
}

void bin_file::set_byte8(_bin_side side, int offset, byte value) {
  if(offset < 0x0000 || offset >= 0xFFFF) return;
  switch(side) {
  case TSIDE:
    tside[offset] = value;
    break;
  case ESIDE:
    eside[offset] = value;
    break;
  }
}

void bin_file::set_byte16(_bin_side side, int offset, quint16 value) {
  if(offset < 0x0000 || offset + 1 >= 0xFFFF) return;
  byte low_byte = value & 0xFF;
  byte high_byte = value >> 8;
  switch(side) {
  case TSIDE:
    tside[offset] = high_byte;
    tside[offset + 1] = low_byte;
    break;
  case ESIDE:
    eside[offset] = high_byte;
    eside[offset + 1] = low_byte;
    break;
  }
}

byte bin_file::get_byte8(_bin_side side, int offset) {
  if(offset < 0x0000 || offset >= 0xFFFF) return 0x00;
  switch(side) {
  case TSIDE:
    return tside[offset];
    break;
  case ESIDE:
    return eside[offset];
    break;
  }
}

quint16 bin_file::get_byte16(_bin_side side, int offset) {
  if(offset < 0x0000 || offset + 1 >= 0xFFFF) return 0x00;
  switch(side) {
  case TSIDE:
    return convert_twobytes(offset, tside);
    break;
  case ESIDE:
    return convert_twobytes(offset, eside);
    break;
  }
}


bool bin_file::save_file(QString path) {
  QFile f(path);
  f.open( QIODevice::WriteOnly );
  if(f.isWritable() == false) {
    return false;
  }
  QDataStream bin_out(&f);
  bin_out.writeRawData((char*)&tside,0x10000);
  bin_out.writeRawData((char*)&eside,0x10000);
  f.close();
  return true;
}

bool bin_file::compare_tside(bin_file comp) {
  for(int x=0x2000;x<=0xFFFF;x++) {
    if(tside[x] != comp.tside[x]) return false;
  }
  return true;
}

bool bin_file::compare_eside(bin_file comp) {
  for(int x=0x2000;x<=0xFFFF;x++) {
    if(eside[x] != comp.eside[x]) return false;
  }
  return true;
}

bool bin_file::copy_to(bin_file *cp) {
  for(int x=0x00;x<=0xFFFF;x++) {
    cp->eside[x] = eside[x];
    cp->tside[x] = tside[x];
  }
  return true; // can't fail ...
}

void bin_file::zero_bin() {
  for(int x=0x0000;x<=0xFFFF;x++) {
    tside[x] = 0x00;
    eside[x] = 0x00;
  }
}

bool bin_file::apply_patches() {
  int success = true;
  if(tside[PATCH_VERSION_BYTE] != 0x00) return false;
  if(patch_enhanced_logging() == false) success = false;
  if(patch_afr_msg0() == false) success = false;
  if(patch_eside_comms() == false) success = false;
  set_patch_version();
  return success;
}

bool bin_file::is_automatic_transmission() {
  return getbit(0x2026,6,tside);
}

bool bin_file::is_maf_enabled() {
  return !getbit(0x2028,1,eside);
}

bool bin_file::is_data_in_blank_eside_region_a() {
  // large area in e-side from 0x8329 to 0xff8f
  for(int x=0x8330; x<=0x8fff; x++) {
    if(eside[x] != 0x00 && eside[x] != 0xFF) {
      return true;
    }
  }
  return false;
}

bool bin_file::is_data_in_blank_eside_region_b() {
  for(int x=0x9401; x<=0xff8f; x++) {
    if(eside[x] != 0x00 && eside[x] != 0xFF) {
      return true;
    }
  }
  return false;
}

bool bin_file::is_tunercat_notes_blank() {
  // large area in e-side from 0x8329 to 0xff8f
  for(int x=0x9000; x<=0x9400; x++) {
    if(eside[x] != 0x00 && eside[x] != 0xFF && eside[x] != 0x20) {
      return false;
    }
  }
  return true;
}

QString bin_file::ecm_info_string() {
  QString info_string;
  info_string += "MAF: " + bool_to_enabled(is_maf_enabled()) + "\n";
  info_string += "Automatic Transmission: " + bool_to_enabled(is_automatic_transmission()) + "\n";
  info_string += "Data in unused E-Side Region A: " + bool_to_present(is_data_in_blank_eside_region_a()) + "\n";
  info_string += "Data in unused E-Side Region B: " + bool_to_present(is_data_in_blank_eside_region_b()) + "\n";
  info_string += "TunerCat Notes Field: " + bool_to_present(!is_tunercat_notes_blank());
  return info_string;
}

QString bin_file::bool_to_enabled(bool x) {
  if(x == true) return QString("Enabled");
  return QString("Disabled");
}

QString bin_file::bool_to_present(bool x) {
  if(x == true) return QString("Present");
  return QString("Not Present");
}

int bin_file::ff_unused_regions() {
  int total_filled = 0;

  // fill ff regions
  if(is_automatic_transmission() == false) { // manual transmission, kill all auto tables
    //28A0-38E1 appears to be all automatic transmission related junk
    total_filled += fill_0xff(tside,0x28A0,4150);
  }

  if(is_maf_enabled() == false) { // maf disabled
    total_filled += fill_0xff(eside,0x241a,154); // maf table
  }

  // eside blank regions:
  if(is_data_in_blank_eside_region_a() == false) {
    total_filled += fill_0xff(eside,0x8330,0x8fff-0x8330);
  }

  if(is_tunercat_notes_blank() == true) {
    total_filled += fill_0xff(eside,0x9000,0x9400-0x9000);
  }

  if(is_data_in_blank_eside_region_b() == false) {
    total_filled += fill_0xff(eside,0x9401,0xff8f-0x9401);
  }

  return total_filled;
}

// enhanced logging, patch id A0

bool bin_file::patch_enhanced_logging() {
  byte patch_data[] = {
    0x00,0x00,  //-----
    0xF4,0xF4,  //-----
    0x80,0x23,  //-----
    0x19,0x92,  //-----
    0x19,0x92,  //-----
    0x00,0xE6,  //-----
    0x00,0xE7,  //-----
    0xFF,0xFF,  //-----
    0x01,0xC3,
    0x00,0xE5,
    0x01,0x34,
    0x01,0xA7,
    0x02,0x59,
    0x01,0x02,
    0x01,0x08,
    0x01,0x76,
    0x01,0x17,
    0x18,0x28,
    0x01,0x24,
    0x01,0x26,
    0x01,0x65,
    0x01,0x63,
    0x01,0x64,
    0x01,0x62,
    0x01,0x61,
    0xFF,0xFF,
    0x02,0xB3,
    0xFF,0xFF,
    0x02,0xBD,
    0xFF,0xFF,
    0x02,0xC5,
    0x18,0x25,
    0x02,0x41,
    0x02,0x3A,
    0xFF,0xFF,
    0x02,0x38,
    0x01,0x03,
    0x01,0x9D,
    0x02,0x3F
  };
  int patch_offset = 0xF5F3;
  // install patch
  install_patch(tside,patch_offset,patch_data,sizeof(patch_data));
  return true;
}

bool bin_file::patch_afr_msg0() {
  // set data byte 24 to AFR byte
  int patch_offset = 0xF3DB;
  //byte compare_data[] = {0x01,0xBD};
  byte patch_data[] = {0x02,0x3F};
  //if(compare_patch(tside,patch_offset,compare_data,sizeof(compare_data)) == false) return false;
  install_patch(tside,patch_offset,patch_data,sizeof(patch_data));
  return true;
}

bool bin_file::patch_eside_comms() {
  // these are kur4o's patches.  thanks kur4o!  great work.
  byte patch_0x82DD[] = {0x83,0x42};
  byte patch_0x824C[] = {0x41};
  byte patch_0x8256[] = {0x11};
  byte patch_0x7237[] = {0x83,0x42};
  byte patch_0x6F78[] = {0x83,0x05};
  byte patch_0x82F7[] = {0x83,0x68,0x83,0xEA,0x84,0x50,0x84,0xC4,0x85,0x54,0x85,0xB8,0x86,0x84,
                         0x01,0x2A,0x01,0x2B,0x01,0x2E,0x01,0x2C,0x01,0x2D,0x00,0xCC,0x00,0xCD,
                         0x01,0x2F,0x01,0x29,0x01,0x28,0x00,0x0C,0x20,0x00,0x20,0x01,0x20,0x02,
                         0x20,0x03,0x00,0xCA,0x00,0xCB,0x01,0x88,0x01,0xC9,0x01,0xCA,0x01,0xCB,
                         0x01,0xCC,0x01,0xCD,0x01,0x8D,0x01,0x8E,0x00,0x00,0x00,0x00,0x00,0x00,
                         0x00,0x00,0x00,0x00,0x00,0x83,0x5E,0x83,0xE0,0x84,0x46,0x84,0xBA,0x85,
                         0x4A,0x85,0xAE,0x86,0x7A,0x86,0xD0,0x87,0x52,0x87,0xB8,0x88,0x2C,0x88,
                         0xBC,0x89,0x20,0x89,0xEC,0x00,0x00,0xE4,0xE4,0x80,0x3D,0x18,0x89,0x18,
                         0x89,0x00,0x01,0x00,0x02,0x00,0x03,0x00,0x04,0x00,0x05,0x00,0x06,0x00,
                         0x07,0x00,0x08,0x00,0x09,0x00,0x0A,0x00,0x0B,0x00,0x0C,0x00,0x0D,0x00,
                         0x0E,0x00,0x0F,0x00,0x10,0x00,0x11,0x00,0x12,0x00,0x13,0x00,0x14,0x00,
                         0x15,0x00,0x16,0x00,0x17,0x00,0x18,0x00,0x19,0x00,0x1A,0x00,0x1B,0x00,
                         0x1C,0x00,0x1D,0x00,0x1E,0x00,0x1F,0x00,0x20,0x00,0x21,0x00,0x22,0x00,
                         0x23,0x00,0x24,0x00,0x25,0x00,0x26,0x00,0x27,0x00,0x28,0x00,0x29,0x00,
                         0x2A,0x00,0x2B,0x00,0x2C,0x00,0x2D,0x00,0x2E,0x00,0x2F,0x00,0x30,0x00,
                         0x31,0x00,0x32,0x00,0x33,0x00,0x34,0x00,0x35,0x00,0x36,0x00,0x37,0x00,
                         0x38,0x00,0x39,0x00,0x3A,0x00,0x3B,0x00,0x3C,0x00,0x00,0xE4,0xE4,0x80,
                         0x2F,0x18,0x89,0x18,0x89,0x20,0x00,0x20,0x01,0x20,0x02,0x20,0x03,0x20,
                         0x04,0x20,0x05,0x20};
  byte patch_0x83F7[] = {0x06,0x20,0x07,0x20,0x08,0x20,0x09,0x20,0x0A,0x20,0x0B,0x20,0x0C,0x20,
                         0x0D,0x20,0x0E,0x20,0x0F,0x20,0x10,0x20,0x11,0x20,0x12,0x20,0x12,0x20,
                         0x13,0x20,0x14,0x20,0x15,0x20,0x16,0x20,0x17,0x20,0x18,0x20,0x19,0x20,
                         0x1A,0x20,0x1B,0x20,0x1C,0x18,0x00,0x18,0x01,0x18,0x02,0x18,0x03,0x18,
                         0x04,0x18,0x05,0x18,0x06,0x18,0x07,0x18,0x08,0x18,0x09,0x18,0x0A,0x18,
                         0x0B,0x18,0x0C,0x18,0x0D,0x18,0x0E,0x18,0x0F,0x00,0x00,0xE4,0xE4,0x80,
                         0x36,0x18,0x89,0x18,0x89,0x18,0x7F,0x18,0x80,0x18,0x81,0x18,0x82,0x18,
                         0x83,0x18,0x84,0x18,0x85,0x18,0x86,0x18,0x87,0x18,0x88,0x18,0x89,0x18,
                         0x8A,0x18,0x8B,0x18,0x8C,0x18,0x8D,0x01,0x49,0x01,0x4A,0x01,0x4B,0x01,
                         0x4C,0x01,0x4D,0x01,0x4E,0x01,0x4F,0x01,0x50,0x01,0x51,0x01,0x52,0x01,
                         0x53,0x01,0x54,0x01,0x55,0x01,0x56,0x01,0x57,0x01,0x58,0x01,0x59,0x01,
                         0x5A,0x01,0x5B,0x01,0x5C,0x01,0x5D,0x01,0x5E,0x01,0x5F,0x01,0x60,0x01,
                         0x61,0x01,0x62,0x01,0x63,0x01,0x64,0x01,0x65,0x01,0x66,0x01,0x67,0x01,
                         0x68,0x01,0x69,0x01,0x6A,0x01,0x6B,0x01,0x6C,0x01,0x6D,0x01,0x6E,0x00,
                         0x00,0xE4,0xE4,0x80,0x44,0x18,0x89,0x18,0x89,0x01,0x02,0x01,0x03,0x01,
                         0x04,0x01,0x05,0x01,0x06,0x01,0x07,0x01,0x08,0x01,0x09,0x01,0x0A,0x01,
                         0x0B,0x01,0x0C,0x01,0x0D,0x01,0x0E,0x01,0x0F,0x18,0x28,0x01,0x11,0x01,
                         0x12,0x01,0x13,0x01,0x14,0x01,0x15,0x01,0x16,0x01,0x17,0x01,0x18,0x01,
                         0x19,0x01,0x1A,0x01};
  byte patch_0x84F7[] = {0x1B,0x01,0x1C,0x01,0x1D,0x01,0x1E,0x01,0x24,0x01,0x25,0x01,0x26,0x01,
                         0x27,0x00,0x7D,0x00,0x7E,0x00,0x7F,0x10,0x00,0x10,0x02,0x02,0xB7,0x02,
                         0xB6,0x02,0xAE,0x02,0xAF,0x02,0xB1,0x02,0xB2,0x02,0xB0,0x02,0xB5,0x02,
                         0xC0,0x02,0xC1,0x02,0xC2,0x02,0xC3,0x02,0xC4,0x82,0xF7,0x82,0xF8,0x82,
                         0xF9,0x82,0xFA,0x82,0xFB,0x82,0xFC,0x82,0xFD,0x82,0xFE,0x82,0xFF,0x83,
                         0x00,0x83,0x01,0x83,0x02,0x83,0x03,0x83,0x04,0xF7,0xD7,0xF7,0xD8,0x00,
                         0x00,0xE4,0xE4,0x80,0x2E,0x18,0x89,0x18,0x89,0x0E,0x24,0x0E,0x25,0x0E,
                         0x26,0x0E,0x27,0x0E,0x28,0x0E,0x29,0x0E,0x2A,0x0E,0x2B,0x0E,0x2C,0x0E,
                         0x2D,0x0E,0x2E,0x0E,0x2F,0x0E,0x30,0x0E,0x31,0x0E,0x32,0x0E,0x33,0x0E,
                         0x34,0x0E,0x20,0x0E,0x21,0x0E,0x22,0x0E,0x23,0x0E,0x04,0x0E,0x05,0x0E,
                         0x06,0x0E,0x07,0x0E,0x08,0x0E,0x09,0x0E,0x0A,0x0E,0x0B,0x0E,0x0C,0x0E,
                         0x0D,0x0E,0x0E,0x0E,0x0F,0x0E,0x10,0x0E,0x11,0x0E,0x12,0x0E,0x13,0x20,
                         0x00,0x20,0x01,0x20,0x02,0x20,0x03,0x02,0xB9,0x02,0xBA,0x02,0xBB,0x02,
                         0xBC,0x00,0x00,0xE4,0xE4,0x80,0x62,0x18,0x89,0x18,0x89,0x00,0x5C,0x00,
                         0x5D,0x00,0x5E,0x00,0x5F,0x00,0x60,0x00,0x61,0x00,0x44,0x00,0x45,0x00,
                         0x46,0x00,0x47,0x00,0x48,0x00,0x49,0x00,0x56,0x00,0x57,0x00,0x58,0x00,
                         0x59,0x00,0x5A,0x00,0x5B,0x00,0x4A,0x00,0x4B,0x00,0x4C,0x00,0x4D,0x00,
                         0x4E,0x00,0x4F,0x00,0x50,0x00,0x51,0x00,0x52,0x00,0x53,0x00,0x54,0x00,
                         0x55,0x00,0x69,0x00};
  byte patch_0x85F7[] = {0x63,0x00,0x6A,0x00,0x64,0x00,0x6B,0x00,0x65,0x00,0x6C,0x00,0x66,0x00,
                         0x6D,0x00,0x67,0x00,0x6E,0x00,0x68,0x00,0x6F,0x00,0x70,0x00,0x71,0x00,
                         0x72,0x00,0x73,0x00,0x74,0x00,0x75,0x00,0x76,0x00,0x77,0x00,0x78,0x00,
                         0x79,0x00,0x7A,0xF7,0xD3,0xF7,0xD4,0xF7,0xD5,0xF7,0xD6,0x19,0x2B,0x19,
                         0x2C,0x19,0x2D,0x19,0x2E,0x19,0x2F,0x19,0x30,0x19,0x31,0x19,0x32,0x19,
                         0x33,0x19,0x34,0x19,0x35,0x19,0x36,0x19,0x37,0x19,0x38,0x19,0x39,0x19,
                         0x3A,0x19,0x3D,0x19,0x3E,0x19,0x41,0x19,0x42,0x19,0x43,0x19,0x44,0x19,
                         0x45,0x19,0x46,0x19,0x47,0x19,0x48,0x19,0x49,0x19,0x4A,0x19,0x4B,0x19,
                         0x4C,0x19,0x4D,0x19,0x4E,0x19,0x4F,0x19,0x50,0x19,0x26,0x19,0x27,0x19,
                         0x28,0x19,0x29,0x19,0x2A,0x00,0x00,0xE4,0xE4,0x80,0x27,0x18,0x89,0x18,
                         0x89,0x01,0x4F,0x01,0x50,0x01,0x53,0x01,0x5C,0x00,0x35,0x01,0xEB,0x01,
                         0x59,0x01,0xE1,0x01,0x58,0xFF,0xFF,0x01,0xD4,0xFF,0xFF,0x01,0xDF,0xFF,
                         0xFF,0x02,0x20,0x01,0x52,0x01,0x4A,0x01,0x4C,0x01,0x49,0x01,0x4B,0x00,
                         0x34,0x00,0x01,0x00,0x25,0x00,0x24,0xFF,0xFF,0x01,0x8B,0xFF,0xFF,0x01,
                         0x8D,0x01,0x90,0x01,0x60,0x01,0x2A,0x01,0x93,0x01,0x51,0x01,0x97,0x01,
                         0x96,0x01,0x9B,0x00,0x08,0x00,0xF8,0x00,0x00,0xE4,0xE4,0x80,0x3D,0x18,
                         0x89,0x18,0x89,0x00,0x0C,0x00,0x04,0x00,0x05,0x00,0x06,0x00,0x07,0x00,
                         0x08,0x00,0x09,0x00,0x0A,0x00,0xE2,0x00,0xE3,0x00,0xE4,0x00,0xE5,0x00,
                         0xE6,0x00,0xE7,0x00};
  byte patch_0x86F7[] = {0x0D,0x00,0xD7,0x02,0x43,0xFF,0xFF,0x01,0xC3,0x02,0x42,0x01,0x34,0x01,
                         0x56,0x01,0xA7,0x01,0xBD,0x02,0x59,0x02,0x50,0x01,0x02,0x01,0x08,0x01,
                         0x07,0x01,0x76,0x01,0x17,0x18,0x28,0x01,0x24,0x01,0x26,0x01,0x13,0x01,
                         0x1B,0x01,0x65,0x01,0x63,0x01,0x64,0x01,0x62,0x01,0x61,0xFF,0xFF,0x02,
                         0xB3,0xFF,0xFF,0x02,0xBD,0xFF,0xFF,0x02,0xC5,0x02,0x2A,0x18,0x25,0x18,
                         0x26,0x02,0x41,0x02,0x3A,0xFF,0xFF,0x02,0x38,0x01,0x03,0x02,0x37,0x00,
                         0x0E,0x01,0x9D,0xFF,0xFF,0x00,0x42,0x00,0x00,0xE4,0xE4,0x80,0x2F,0x18,
                         0x89,0x18,0x89,0x00,0x09,0x00,0x0A,0x00,0x0B,0x00,0x0C,0x00,0x0D,0x01,
                         0x07,0x1A,0xD3,0xFF,0xFF,0x1A,0xD8,0x1B,0x4A,0x1B,0x57,0x1A,0x99,0x1A,
                         0x9A,0x1A,0x98,0x00,0xD7,0x01,0x17,0x1B,0x4E,0x1A,0x94,0x1A,0x95,0x1A,
                         0x85,0x1A,0x86,0xFF,0xFF,0x1B,0x5B,0x1A,0x82,0x1A,0x83,0xFF,0xFF,0x28,
                         0x73,0x1A,0x7F,0x01,0x9D,0xFF,0xFF,0x1B,0x3F,0x01,0xA2,0x1B,0x15,0xFF,
                         0xFF,0x1A,0x77,0x00,0xE2,0x00,0xE3,0x00,0xE4,0x00,0xE6,0x00,0xE7,0x00,
                         0xE8,0x00,0xE9,0x00,0xD3,0x00,0xEA,0x00,0xEB,0x00,0xEC,0x00,0x00,0xE4,
                         0xE4,0x80,0x36,0x18,0x89,0x18,0x89,0x02,0x03,0x02,0x04,0x02,0x05,0x02,
                         0x06,0x02,0x07,0x02,0x08,0x02,0x09,0x02,0x0A,0x02,0x0B,0x02,0x0C,0x02,
                         0x0D,0x02,0x0E,0x00,0x03,0x00,0x04,0x00,0x05,0x00,0x06,0x00,0x07,0x00,
                         0x08,0x00,0x09,0x00,0x0A,0x00,0x0B,0x00,0x0C,0x00,0x0D,0x00,0x0E,0x02,
                         0x6D,0xFF,0xFF,0x01};
  byte patch_0x87F7[] = {0xC3,0x00,0xE3,0x00,0xD7,0x00,0xDA,0x00,0xD9,0x00,0xDC,0x00,0xDB,0x00,
                         0xDE,0x00,0xDD,0x00,0x44,0x00,0x45,0x00,0x46,0x00,0x47,0x00,0x48,0x00,
                         0x49,0x00,0x4A,0x00,0x4B,0x00,0x4C,0x00,0x4D,0x00,0x4E,0x00,0x4F,0x00,
                         0x0F,0x00,0x10,0x00,0x11,0x00,0x12,0x00,0x13,0x00,0x14,0x00,0x00,0xE4,
                         0xE4,0x80,0x44,0x18,0x89,0x18,0x89,0x01,0x02,0x01,0x03,0x01,0x04,0x01,
                         0x05,0x01,0x06,0x01,0x07,0x01,0x08,0x01,0x09,0x01,0x0A,0x01,0x0B,0x01,
                         0x0C,0x01,0x0D,0x01,0x0E,0x01,0x0F,0x18,0x28,0x01,0x11,0x01,0x12,0x01,
                         0x13,0x01,0x14,0x01,0x15,0x01,0x16,0x01,0x17,0x01,0x18,0x01,0x19,0x01,
                         0x1A,0x01,0x1B,0x01,0x1C,0x01,0x1D,0x01,0x1E,0x01,0x24,0x01,0x25,0x01,
                         0x26,0x01,0x27,0x00,0x7D,0x00,0x7E,0x00,0x7F,0x10,0x00,0x10,0x02,0x02,
                         0xB7,0x02,0xB6,0x02,0xAE,0x02,0xAF,0x02,0xB1,0x02,0xB2,0x02,0xB0,0x02,
                         0xB5,0x02,0xC0,0x02,0xC1,0x02,0xC2,0x02,0xC3,0x02,0xC4,0x82,0xF7,0x82,
                         0xF8,0x82,0xF9,0x82,0xFA,0x82,0xFB,0x82,0xFC,0x82,0xFD,0x82,0xFE,0x82,
                         0xFF,0x83,0x00,0x83,0x01,0x83,0x02,0x83,0x03,0x83,0x04,0xF7,0xD7,0xF7,
                         0xD8,0x00,0x00,0xE4,0xE4,0x80,0x2E,0x18,0x89,0x18,0x89,0x0E,0x24,0x0E,
                         0x25,0x0E,0x26,0x0E,0x27,0x0E,0x28,0x0E,0x29,0x0E,0x2A,0x0E,0x2B,0x0E,
                         0x2C,0x0E,0x2D,0x0E,0x2E,0x0E,0x2F,0x0E,0x30,0x0E,0x31,0x0E,0x32,0x0E,
                         0x33,0x0E,0x34,0x0E,0x20,0x0E,0x21,0x0E,0x22,0x0E,0x23,0x0E,0x04,0x0E,
                         0x05,0x0E,0x06,0x0E};
  byte patch_0x88F7[] = {0x07,0x0E,0x08,0x0E,0x09,0x0E,0x0A,0x0E,0x0B,0x0E,0x0C,0x0E,0x0D,0x0E,
                         0x0E,0x0E,0x0F,0x0E,0x10,0x0E,0x11,0x0E,0x12,0x0E,0x13,0x20,0x00,0x20,
                         0x01,0x20,0x02,0x20,0x03,0x02,0xB9,0x02,0xBA,0x02,0xBB,0x02,0xBC,0x00,
                         0x00,0xE4,0xE4,0x80,0x62,0x18,0x89,0x18,0x89,0x00,0x5C,0x00,0x5D,0x00,
                         0x5E,0x00,0x5F,0x00,0x60,0x00,0x61,0x00,0x44,0x00,0x45,0x00,0x46,0x00,
                         0x47,0x00,0x48,0x00,0x49,0x00,0x56,0x00,0x57,0x00,0x58,0x00,0x59,0x00,
                         0x5A,0x00,0x5B,0x00,0x4A,0x00,0x4B,0x00,0x4C,0x00,0x4D,0x00,0x4E,0x00,
                         0x4F,0x00,0x50,0x00,0x51,0x00,0x52,0x00,0x53,0x00,0x54,0x00,0x55,0x00,
                         0x69,0x00,0x63,0x00,0x6A,0x00,0x64,0x00,0x6B,0x00,0x65,0x00,0x6C,0x00,
                         0x66,0x00,0x6D,0x00,0x67,0x00,0x6E,0x00,0x68,0x00,0x6F,0x00,0x70,0x00,
                         0x71,0x00,0x72,0x00,0x73,0x00,0x74,0x00,0x75,0x00,0x76,0x00,0x77,0x00,
                         0x78,0x00,0x79,0x00,0x7A,0xF7,0xD3,0xF7,0xD4,0xF7,0xD5,0xF7,0xD6,0x19,
                         0x2B,0x19,0x2C,0x19,0x2D,0x19,0x2E,0x19,0x2F,0x19,0x30,0x19,0x31,0x19,
                         0x32,0x19,0x33,0x19,0x34,0x19,0x35,0x19,0x36,0x19,0x37,0x19,0x38,0x19,
                         0x39,0x19,0x3A,0x19,0x3D,0x19,0x3E,0x19,0x41,0x19,0x42,0x19,0x43,0x19,
                         0x44,0x19,0x45,0x19,0x46,0x19,0x47,0x19,0x48,0x19,0x49,0x19,0x4A,0x19,
                         0x4B,0x19,0x4C,0x19,0x4D,0x19,0x4E,0x19,0x4F,0x19,0x50,0x19,0x26,0x19,
                         0x27,0x19,0x28,0x19,0x29,0x19,0x2A,0x00,0x00,0xE4,0xE4,0x80,0x27,0x18,
                         0x89,0x18,0x89,0x18};
  byte patch_0x89F7[] = {0x30,0x18,0x31,0x18,0x32,0x18,0x33,0x18,0x34,0x18,0x35,0x18,0x36,0x18,
                         0x37,0x18,0x38,0x18,0x39,0x18,0x3A,0x18,0x3B,0x18,0x3C,0x18,0x3D,0x18,
                         0x3E,0x18,0x3F,0x18,0x40,0x18,0x41,0x18,0x42,0x18,0x43,0x18,0x44,0x18,
                         0x45,0x18,0x46,0x18,0x47,0x18,0x48,0x18,0x49,0x18,0x4A,0x18,0x4B,0x18,
                         0x4C,0x18,0x4D,0x18,0x4E,0x18,0x4F,0x18,0x50,0x18,0x51,0x00,0xE9,0x1B,
                         0x56,0x1A,0x94,0x1A,0x95,0x00,0x00,0x00,0x00,0x00};
  install_patch(eside,0x82DD,patch_0x82DD,sizeof(patch_0x82DD));
  install_patch(eside,0x824C,patch_0x824C,sizeof(patch_0x824C));
  install_patch(eside,0x8256,patch_0x8256,sizeof(patch_0x8256));
  install_patch(eside,0x7237,patch_0x7237,sizeof(patch_0x7237));
  install_patch(eside,0x6F78,patch_0x6F78,sizeof(patch_0x6F78));
  install_patch(eside,0x82F7,patch_0x82F7,sizeof(patch_0x82F7));
  install_patch(eside,0x83F7,patch_0x83F7,sizeof(patch_0x83F7));
  install_patch(eside,0x84F7,patch_0x84F7,sizeof(patch_0x84F7));
  install_patch(eside,0x85F7,patch_0x85F7,sizeof(patch_0x85F7));
  install_patch(eside,0x86F7,patch_0x86F7,sizeof(patch_0x86F7));
  install_patch(eside,0x87F7,patch_0x87F7,sizeof(patch_0x87F7));
  install_patch(eside,0x88F7,patch_0x88F7,sizeof(patch_0x88F7));
  install_patch(eside,0x89F7,patch_0x89F7,sizeof(patch_0x89F7));
  // msg6---basics
  byte patch_emsg0[] {
    0x00,0x00, //------
    0xE4,0xE4, //------
    0x80,0x11, // length
    0x18,0x89, //------
    0x18,0x89, //------
    0x01,0x4c, // 0x00: pe spark adder
    0x01,0x85, // 0x01: raw maf frequency
    0x00,0xf8,  // 0x02: ve selected value
    0x01,0x60, // 0x03: rpm25
    0x01,0xE2, // 0x04: dfco delay counter
    0x00,0x35, // status word 1
    0x01,0x4f, // main/ext/idle spk advance
    0x01,0x53, // egr spark advance
    0x01,0x50, // coolant temp spark advance
    0x00,0x34, // status word 2
    0x00,0x01, // status word 3
    0x00,0x25, // status word 4
    0x01,0x9b, // pe AFR
  };
  install_patch(eside,0x835E,patch_emsg0,sizeof(patch_emsg0));
  return true;
}

bool bin_file::is_null() {
  for(int x=0x0000;x<=0xFFFF;x++) {
    if(tside[x] != 0x00) return false;
    if(eside[x] != 0x00) return false;
  }
  return true;
}

void bin_file::set_patch_version() {
  tside[PATCH_VERSION_BYTE] = CURRENT_PATCH_VERSION;
}

//---utility functions

int bin_file::fill_0xff(byte *data, int offset, int size) {
  for(int x=0;x<size;x++) {
    data[x + offset] = 0xFF;
  }
  return size;
}

void bin_file::install_patch(byte *bin, int offset, byte *patch, int size) {
  for(int x=0;x<size;x++) {
    bin[x + offset] = patch[x];
  }
}

bool bin_file::compare_patch(byte *bin, int offset, byte *patch, int size) {
  for(int x=0;x<size;x++) {
    if(bin[x + offset] != patch[x]) return false;
  }
  return true;
}
