#include <QFileDialog>
#include <QMessageBox>
#include <QSerialPort>
#include "settings_editor.h"
#include "ui_settings_editor.h"
#include "datalog.h"

settings_editor::settings_editor(datastream_control *control_in, datalog_definition *def_in, QWidget *parent) :
  QWidget(parent),
  ui(new Ui::settings_editor) {
  control = control_in;
  ui->setupUi(this);
  def = def_in;
}

void settings_editor::on_interface_loaded() {
  get_all_settings();
  process_port_list();
}

settings_editor::~settings_editor() {
  delete ui;
}

void settings_editor::get_all_settings() {
  get_wideband_config();
  get_config_misc();
  // save path
  if(config.get_savepath() != nullptr) {
    ui->display_save_dir->setText(config.get_savepath());
  } else {
    ui->display_save_dir->setText("None");
  }
  if(config.get_deffile() != nullptr) {
    ui->display_deffile->setText(config.get_deffile());
  }
  // serial port
  if(config.get_serialport() != nullptr) {
    select_port_from_list(config.get_serialport());
  }
}

// new misc group

void settings_editor::get_config_misc() {

  // fixed
  ui->setting_warn_log->setChecked(config.get_warn_log());
  ui->setting_warn_notes->setChecked(config.get_warn_notes());
  ui->setting_disable_patch->setChecked(config.get_disable_patch());
  ui->setting_check_update->setChecked(config.get_check_update());
  ui->setting_graceful_disconnect->setChecked(config.get_graceful_disconnect());
  ui->setting_display_low_precision->setChecked(config.get_low_precision());
  ui->setting_display_percent->setChecked(config.get_display_blm_percent());
  ui->setting_display_kph->setChecked(config.get_display_kph());
  ui->setting_display_celcius->setChecked(config.get_display_celcius());
  ui->setting_display_low_percent_ok->setChecked(config.get_low_percent_ok());
  ui->setting_async_display->setChecked(config.get_async_display());
  ui->setting_m4keepalive->setChecked(config.get_m4_keepalive());
  ui->setting_extra_mods->setChecked(config.get_silence_extra());
  ui->setting_alwaysdraw->setChecked(config.get_always_draw());
}

void settings_editor::on_setting_warn_log_toggled(bool checked) {
  config.set("WARN_LOG",checked);
}

void settings_editor::on_setting_warn_notes_toggled(bool checked) {
  config.set("WARN_NOTES",checked);
}

void settings_editor::on_setting_disable_patch_toggled(bool checked) {
  config.set("DISABLE_PATCH",checked);
}

void settings_editor::on_setting_graceful_disconnect_toggled(bool checked) {
  config.set("GRACEFUL_DISCONNECT",checked);
}

void settings_editor::on_setting_display_percent_toggled(bool checked) {
  config.set("DISPLAY_BLM_PERCENT",checked);
  emit display_config_changed();
}

void settings_editor::on_setting_display_celcius_toggled(bool checked) {
  config.set("DISPLAY_CELCIUS",checked);
  emit display_config_changed();
}

void settings_editor::on_setting_display_kph_toggled(bool checked) {
  config.set("DISPLAY_KPH",checked);
  emit display_config_changed();
}

void settings_editor::on_setting_display_low_percent_ok_toggled(bool checked) {
  config.set("LOW_PERCENT_OK",checked);
  emit display_config_changed();
}

void settings_editor::on_setting_display_low_precision_toggled(bool checked) {
  config.set("LOW_PRECISION",checked);
  emit display_config_changed();
}

void settings_editor::on_serialPortSelect_currentIndexChanged(const QString &arg1) {
  control->port_name.set(arg1);
  config.set_serialport(arg1);
}

void settings_editor::on_wb_use_ac_toggled(bool checked) {
  if(checked == false) return;
  config.set_wideband_src(eehack_settings::WB_SRC_AC);
  set_wideband_parameters();
}

void settings_editor::on_wb_use_d27_toggled(bool checked) {
  if(checked == false) return;
  config.set_wideband_src(eehack_settings::WB_SRC_D27);
  set_wideband_parameters();
}

void settings_editor::get_wideband_config() {
  ui->wb_high_afr->setValue(config.get_float("WB_HIGH_AFR",20.00));
  ui->wb_low_afr->setValue(config.get_float("WB_LOW_AFR",10.00));
  ui->wb_high_voltage->setValue(config.get_float("WB_HIGH_VOLTAGE",5.00));
  ui->wb_trim->setValue(config.get_float("WB_TRIM",0.00));
  ui->setting_smooth_wb->setChecked(config.get_wideband_smooth());
  set_wb_src(config.get_wideband_src());
  ui->setting_enable_wideband->setChecked(config.get_wideband_enable());
  set_wideband_parameters();
  emit datastream_config_changed();
}

void settings_editor::on_setting_enable_wideband_toggled(bool checked) {
  config.set_wideband_enable(checked);
  set_wideband_parameters();
}

void settings_editor::on_setting_smooth_wb_toggled(bool checked) {
   config.set_wideband_smooth(checked);
}

void settings_editor::on_btn_save_dir_clicked() {
  QString dir = QFileDialog::getExistingDirectory(this,
                                                  tr("Select Directory"),"",QFileDialog::ShowDirsOnly
                                                  | QFileDialog::DontResolveSymlinks);
  if(dir != nullptr) {
    config.set_savepath(dir);
    ui->display_save_dir->setText(dir);
  }
}

void settings_editor::on_wb_preset_fast_gas_clicked() {
  ui->wb_high_afr->setValue(21.0);
  ui->wb_high_voltage->setValue(4.0960);
  ui->wb_low_afr->setValue(9.6);
  ui->wb_trim->setValue(0);
}

void settings_editor::on_wb_preset_fast_e85_clicked() {
  ui->wb_high_afr->setValue(14.3);
  ui->wb_high_voltage->setValue(4.0960);
  ui->wb_low_afr->setValue(6.4);
  ui->wb_trim->setValue(0);
}

void settings_editor::on_wb_preset_fast_gas_2_clicked() {
  //LM1/LC1=0V = 7.35 AFR and 5V = 22.39 AFR
  ui->wb_high_afr->setValue(22.390);
  ui->wb_high_voltage->setValue(5.000);
  ui->wb_low_afr->setValue(7.35);
  ui->wb_trim->setValue(0);
}

void settings_editor::on_wb_preset_fast_gas_3_clicked() {

}

void settings_editor::on_wb_low_afr_valueChanged(double arg1) {
  config.set_float("WB_LOW_AFR",arg1);
  set_wideband_parameters();
}

void settings_editor::on_wb_high_voltage_valueChanged(double arg1) {
  config.set_float("WB_HIGH_VOLTAGE",arg1);
  set_wideband_parameters();
}

void settings_editor::on_wb_high_afr_valueChanged(double arg1) {
  config.set_float("WB_HIGH_AFR",arg1);
  set_wideband_parameters();
}

void settings_editor::on_wb_trim_valueChanged(double arg1) {
  config.set_float("WB_TRIM",arg1);
  set_wideband_parameters();
}

void settings_editor::set_wideband_parameters() {
  // decimalized input
  float afr_trim = ui->wb_trim->value(); // trim to two decimal places
  float afr_lowv = ui->wb_low_afr->value(); // zero volt value to one decimal place
  float afr_highv = ui->wb_high_afr->value(); // five volt value to one decimal place
  float volts_highv = ui->wb_high_voltage->value();
  // linear equation
  float wb_mult = ( ( afr_highv - afr_lowv ) / volts_highv) ; // voltage to afr multiplier
  float wb_add = afr_lowv + afr_trim; // voltage to afr adder plus trim
  // fill equation box
  ui->wb_eq->setText(QString("%1N+%2").arg(wb_mult).arg(wb_add));
  // input byte
  for(int x=0;x<MAX_MODE1_MESSAGES;x++) {
    def[x].config_wideband(wb_add,wb_mult,config.get_wideband_enable());
  }
  emit display_config_changed();
}

void settings_editor::set_wb_src(int value) {
  if(value == eehack_settings::WB_SRC_AC) {
    ui->wb_use_d27->setChecked(false);
    ui->wb_use_ac->setChecked(true);
  } else if(value == eehack_settings::WB_SRC_D27) {
    ui->wb_use_ac->setChecked(false);
    ui->wb_use_d27->setChecked(true);
  } else { // this shouldn't happen, default to AC
    set_wb_src(eehack_settings::WB_SRC_AC);
    ui->wb_use_d27->setChecked(false);
    ui->wb_use_ac->setChecked(true);
  }
  set_wideband_parameters();
}

void settings_editor::on_refreshPortsBtn_clicked() {
  process_port_list();
}

void settings_editor::enable_port_selection(bool value) {
  ui->serialPortSelect->setEnabled(value);
}

void settings_editor::get_update_status(QString str) {
  ui->version_check_status->setText(str);
}

void settings_editor::process_port_list() {
  // get list
  QList<QSerialPortInfo> serList = QSerialPortInfo::availablePorts();
  ui->serialPortSelect->clear(); // start new list
  int list_idx = 0;
  if(serList.count() < 1) {
    ui->serialPortSelect->addItem("NONE");
  } else {
    foreach (const QSerialPortInfo &serInfo, serList) {
      ui->serialPortSelect->addItem(serInfo.portName());
      list_idx++;
    }
  }
}

bool settings_editor::select_port_from_list(QString port_name) {
  int i = ui->serialPortSelect->findText(port_name);
  if(i > 0) { // port exists
    ui->serialPortSelect->setCurrentIndex(i);
    return true;
  } else {
    return false;
  }
}

void settings_editor::on_reset_settings_btn_clicked() {
  reset_all_settings();
}

void settings_editor::reset_all_settings() {
  int ret = QMessageBox::warning(this, "EEHack",
                                 "Are you sure you want to discard ALL of your settings?",
                                 QMessageBox::Save | QMessageBox::Discard,
                                 QMessageBox::Save);
  if(ret == QMessageBox::Save) return;
  config.clear();
  get_all_settings();
}

void settings_editor::on_update_force_check_clicked() {
  emit update_recheck_requested();
}


void settings_editor::on_setting_check_update_toggled(bool checked) {
  config.set_check_update(checked);
}

void settings_editor::on_setting_m4keepalive_toggled(bool checked) {
  config.set_m4_keepalive(checked);
  control->m4_keepalive.set(checked);
}

void settings_editor::on_btn_set_deffile_clicked() {
  if(select_deffile() == true) { // cancel wasn't clicked
    QMessageBox::warning(this,"EEHack","You must restart EEHack to load the new definition.",QMessageBox::Ok);
  }
}

bool settings_editor::select_deffile() {
  QString filename = QFileDialog::getOpenFileName(
        this,"Select definition file",
              config.get_savepath(),"CSV File (*.csv)");
  if(filename != nullptr) {
    config.set_deffile(filename);
    ui->display_deffile->setText(filename);
    return true;
  }
  return false;
}

void settings_editor::on_setting_extra_mods_toggled(bool checked) {
  config.set_silence_extra(checked);
}

void settings_editor::on_setting_alwaysdraw_toggled(bool checked) {
  config.set_always_draw(checked);
  emit display_config_changed();
}

void settings_editor::on_setting_async_display_toggled(bool checked) {
  config.set_async_display(checked);
  emit display_config_changed();
}
