#include <QMessageBox>
#include <QFileDialog>
#include <QProgressDialog>
#include <QShortcut>

#include "datalog_window.h"
#include "ui_datalog_window.h"
#include "config.h"
#include "ui_autospark.h"
#include "debuglog_window.h"
#include "bin_file.h"
#include "knock_warning.h"
#include "datalog.h"
#include "common.h"
#include "parameter_select.h"
#include "datalog_window.h"

datalog_window::datalog_window(datastream_control *control_in, datalog *log_in, datalog_definition *def_in, QWidget *parent) :
  QWidget(parent),
  ui(new Ui::datalog_window) {
  ui->setupUi(this);
  // -------------
  def = def_in;
  // -------------
  control = control_in;
  // -------------
  control->ecminfo.request(); // default request on first connect
  // -------------
  dashboard = nullptr;
  dash_wideband = nullptr;
  dash_user_a = nullptr;
  dash_user_b = nullptr;
  dash_user_c = nullptr;
  dash_user_d = nullptr;
  // -------------
  playback_pkt = nullptr;
  dashboard_pkt = nullptr;
  playback_status = false;
  slider_canary = false;
  unsaved_log = false;
  wb_reset_cache();
  draw_in_progress = false;
  // initial persistent log for recording ...
  log = log_in;
  add_log_to_logselector(log->get_first());
  // configure playback
  playback_timer = new QTimer(this);
  connect(playback_timer,SIGNAL(timeout()),this,SLOT(advance_playback_pkt()));
}

void datalog_window::change_log_position(double position) {
  if(log == nullptr) return;
  move_to_record(log->get_packet_by_timestamp(position));
}

void datalog_window::add_log_to_logselector(datalog *d) {
  ui->active_log_selector->addItem(d->name.get());
  int new_item_index = ui->active_log_selector->findText(d->name.get());
  if(new_item_index == -1) {
    return;
  }
  if(new_item_index == 0) {
    QIcon icon(":/icons_new/cassette_icon&16.png");
    ui->active_log_selector->setItemIcon(new_item_index,icon);
  } else {
    QIcon icon(":/icons_new/doc_empty_icon&16.png");
    ui->active_log_selector->setItemIcon(new_item_index,icon);
  }
  ui->active_log_selector->setCurrentIndex(new_item_index);
}

void datalog_window::display_config_changed() {
  if(dashboard != nullptr) {
    if(config.get_wideband_enable() == true) {
      datalog_element *e = def[0].element_by_name(config.get_wb_str());
      dash_wideband->configure_element(e); // upd wb ptr
      ui->display_wideband->show();
      ui->label_wb->show();
    } else {
      dash_wideband->configure_element(nullptr);
      ui->display_wideband->hide();
      ui->label_wb->hide();
    }
    display_dashboard();
  }
  emit display_config_changed_();
}

void datalog_window::recieve_connection_state(int c) {
  switch(c) {
  case STATE_CONNECTED:
    set_interface_connstate(true);
    break;
  case STATE_CONNECTING:
    set_interface_connstate(false);
    break;
  case STATE_DISCONNECTING:
    set_interface_connstate(false);
    break;
  case STATE_DISCONNECTED:
    set_interface_connstate(false);
    break;
  }
}

void datalog_window::wb_reset_cache() {
  wb_cache_population = 0;
  for(int x=0;x<WB_SMOOTH_DEPTH;x++) wb_cache[x] = 0.00;
}

float datalog_window::wb_get_cache(float wbafr) {
  // shift all values back
  for(int x=1;x<WB_SMOOTH_DEPTH;x++) {
    wb_cache[x-1] = wb_cache[x];
  }
  // add current afr to top of stack
  wb_cache[WB_SMOOTH_DEPTH - 1] = wbafr;
  // config cache population
  if(wb_cache_population < WB_SMOOTH_DEPTH) wb_cache_population++;
  // shortcut if only one cache value
  if(wb_cache_population <= 1) return wbafr;
  // step backwards and get avg
  float avg = 0;
  for(int x=WB_SMOOTH_DEPTH-1;x>=WB_SMOOTH_DEPTH-wb_cache_population;x--) {
    avg += wb_cache[x];
  }
  return avg / wb_cache_population;
}

void datalog_window::recieve_new_ecm_info() {
   // only update if recording log active to save time
  if(log == log->get_first()) update_ecm_info();
  // global stuff
  int patch_version = log->get_first()->info.patch_version.get();
  // pass on signal
  emit patch_level_changed_(patch_version);
}

void datalog_window::update_ecm_info() {
  if(log->info.valid.get() == false) { // invalid ecminfo
    ui->calid_display->clear();
    ui->vinnumber_display->clear();
    ui->display_afrtarget->hide();
    ui->label_afr->hide();
    parameter_indicator_display(ui->info_maf,false);
    parameter_indicator_display(ui->info_auto_trans,false);
    parameter_indicator_display(ui->info_eside,false);
  } else { // valid
    ui->calid_display->setText(QString::number((qint32)log->info.calibration_id.get(),10));
    ui->vinnumber_display->setText(log->info.vin_number.get());
    parameter_indicator_display(ui->info_auto_trans,log->info.auto_trans.get());
    set_patch_version_display();
    if(log->info.patch_version.get() < 0x01) {
      ui->display_afrtarget->hide();
      ui->label_afr->hide();
    } else {
      ui->display_afrtarget->show();
      ui->label_afr->show();
    }
    if(log->info.patch_version.get() < 0x02 || log->info.eside_comms_basic.get() == false) { // maf info unavailable
      parameter_indicator_display(ui->info_maf,false);
      parameter_indicator_display(ui->info_eside,false);
    } else { // eside ok
      parameter_indicator_display(ui->info_maf,log->info.maf_enable.get());
      parameter_indicator_display(ui->info_eside,true);
    }
  }
}

void datalog_window::set_patch_version_display() {
  int patch_version = log->info.patch_version.get();
  if(patch_version == 0x00) {
      ui->patch_status->setText("NONE");
  } else if(patch_version > CURRENT_PATCH_VERSION) {
    ui->patch_status->setText(QStringLiteral("Update EEHACK!!"));
  } else {
    ui->patch_status->setText("v" + QString::number(patch_version) + " (Latest v" + QString::number(CURRENT_PATCH_VERSION) + ")");
  }
}

void datalog_window::on_interface_loaded() {
  config_dashboard();
  create_extended_tabs();
  //--------
  set_interface_connstate(false); // this disables stuff that only works once connected
  set_patch_version_display(); // disable enhancements
  // -------
  ui->mainTabSwitcher->setCurrentIndex(0); // start with dashboard
  // load file if specificed
  int n_files = QApplication::arguments().size() - 1;
  for(int x=0;x<n_files;x++) { // load files
    load_log_from_file(QApplication::arguments().at(x + 1));
  }
  display_config_changed();
}

void datalog_window::create_extended_tabs() {
  for(int x=0;x<MAX_MODE1_MESSAGES;x++) {
    if(def[x].is_empty() == true) continue; // definition empty
    QString tab_title;
    if(def[x].msg_desc.isEmpty() == true) {
      tab_title = "Msg" + QString::number(def[x].get_msg_number(),16).toUpper() + "/" +
          QString::number(def[x].get_device(),16).toUpper();
    } else {
      tab_title = def[x].msg_desc;
    }

    ext[x] = new datalog_extended();
    ui->mainTabSwitcher->addTab(ext[x],tab_title);
    connect(this,SIGNAL(display_new_packet(datalog_packet*)),ext[x],SLOT(new_packet_recieved(datalog_packet*)));
    connect(this,SIGNAL(display_config_changed_()),ext[x],SLOT(display_config_changed()));
    connect(this,SIGNAL(patch_level_changed_(int)),ext[x],SLOT(new_patch_level(int)));
    ext[x]->set_definition(&def[x]);
  }
}

void datalog_window::config_dashboard() {
  dash_add_field(ui->display_lblm,"LBLM");
  dash_add_field(ui->display_rblm,"RBLM");
  dash_add_field(ui->display_rint,"RINT");
  dash_add_field(ui->display_lint,"LINT");
  dash_add_field(ui->display_lblm,"LBLM");
  dash_add_field(ui->display_rblm,"RBLM");
  dash_add_field(ui->display_rint,"RINT");
  dash_add_field(ui->display_lint,"LINT");
  dash_add_field(ui->display_lpw,"LBPW");
  dash_add_field(ui->display_rpw,"RBPW");
  dash_add_field(ui->display_lo2,"LO2");
  dash_add_field(ui->display_ro2,"RO2");
  dash_add_field(ui->display_blmcell,"BLMCELL");
  dash_add_indicator(ui->display_pe,"PE");
  dash_add_indicator(ui->display_cl,"CL");
  dash_add_indicator(ui->display_blm,"BLM");
  dash_add_indicator(ui->flag_fan1,"FAN1");
  dash_add_indicator(ui->flag_fan2,"FAN2");
  dash_add_indicator(ui->flag_ccp,"CCP");
  dash_add_indicator(ui->flag_air,"AIR");
  dash_add_indicator(ui->flag_ac,"AC");
  dash_add_field(ui->display_iac,"IAC");
  dash_add_field(ui->display_idle,"IDLETARGET");
  dash_add_field(ui->display_vss,"VSS");
  dash_add_field(ui->display_rpm,"RPM");
  dash_add_field(ui->display_maf,"MAF");
  dash_add_field(ui->display_map,"MAP");
  dash_add_field(ui->display_tps,"TPS");
  dash_add_field(ui->display_baro,"BARO");
  dash_add_field(ui->display_coolant,"COOLTMP");
  dash_add_field(ui->display_intake,"IAT");
  dash_add_field(ui->display_retard,"KR");
  dash_add_field(ui->display_knock,"KCOUNT");
  dash_add_field(ui->display_advance,"SPKADV");
  dash_add_field(ui->display_afrtarget,"AFR");
  dash_add_field(ui->display_gear,"GEAR");
  dash_add_indicator(ui->display_parkneutral,"PNSWITCH");
  dash_add_indicator(ui->display_tcc,"TCCENABLE");
  dash_wideband = dash_add_field(ui->display_wideband,config.get_wb_str());
  dash_user_a = dash_add_field(ui->display_dash_user_a,(datalog_element *)nullptr);
  dash_user_b = dash_add_field(ui->display_dash_user_b,(datalog_element *)nullptr);
  dash_user_c = dash_add_field(ui->display_dash_user_c,(datalog_element *)nullptr);
  dash_user_d = dash_add_field(ui->display_dash_user_d,(datalog_element *)nullptr);

}

// for text indicators
dashboard_field *datalog_window::dash_add_field(QLineEdit *displ, const char *parameter_name) {
  datalog_element *e = def[0].element_by_name(parameter_name);
  if(e == nullptr) return nullptr; // element not found
  dashboard_field *new_field = new dashboard_field(displ,e);
  new_field->next = nullptr;
  if(dashboard == nullptr) {
    dashboard = new_field;
  } else {
    dashboard->get_last()->next = new_field;
  }
  return new_field;
}

dashboard_field *datalog_window::dash_add_field(QLineEdit *displ, datalog_element *element_in) {
  dashboard_field *new_field = new dashboard_field(displ,element_in);
  new_field->next = nullptr;
  if(dashboard == nullptr) {
    dashboard = new_field;
  } else {
    dashboard->get_last()->next = new_field;
  }
  return new_field;
}

// for boolean indicators
dashboard_field *datalog_window::dash_add_indicator(QLineEdit *displ, const char *parameter_name) {
  datalog_element *e = def[0].element_by_name(parameter_name);
  if(e == nullptr) {
    error_dialog("Definition is missing a critical dashboard parameter: " + QString(parameter_name));
    return nullptr;
  }
  dashboard_field *new_field = new dashboard_field(displ,e,true);
  new_field->next = nullptr;
  if(dashboard == nullptr) {
    dashboard = new_field;
  } else {
    dashboard->get_last()->next = new_field;
  }
  return new_field;
}

void datalog_window::set_interface_connstate(bool connected) {
  ui->clear_dtc_btn->setEnabled(connected);
  ui->getecminfo_btn->setEnabled(connected);
  ui->setvin_btn->setEnabled(connected);
  ui->setcal_btn->setEnabled(connected);
}

datalog_window::~datalog_window() {
  delete ui;
}

bool datalog_window::handle_unsaved_data() {
  if(unsaved_log == false) return true;
  if(log->get_first()->is_data() == false) return true;
  int ret = QMessageBox::warning(this, tr("EEHack"),
                                 tr("There is unsaved log data!\n"
                                    "Do you want to save the log?"),
                                 QMessageBox::Save | QMessageBox::Discard
                                 | QMessageBox::Cancel,
                                 QMessageBox::Save);
  if(ret == QMessageBox::Save) {
    save_raw_file(log->get_first());
    return true;
  }
  if(ret == QMessageBox::Discard) return true;
  if(ret == QMessageBox::Cancel) return false;
  return true; // never reached
}

datalog *datalog_window::get_active_log_ptr() {
  return log;
}

//--------------------------------------------

void datalog_window::update_pktcount() {
  // update packet count
  parameter_text_display(ui->display_pktcount,log->get_pkt_count());
  parameter_text_display(ui->display_logtime,(float)log->newest_timestamp() / 1000.00 / 60.00,1,"Min");
  if(log->is_data() == true) {
    ui->time_slider->setMaximum(log->n_packets());
  } else {
    ui->time_slider->setMaximum(1);
  }
}

void datalog_window::add_new_log(datalog *new_log) {
  // avoid duplicate names
  int x = 1;
  QString filter_name = new_log->name.get();
  while(log->get_by_name(filter_name) != nullptr) { // name exists
    filter_name = QString("%1 (%2)").arg(new_log->name.get()).arg(x);
    x++;
  }
  new_log->name.set(filter_name);
  // add to list after last log
  datalog *last_log = log->get_last();
  last_log->set_next(new_log); // link
  new_log->set_prev(last_log);
  add_log_to_logselector(new_log); // add to list
}

void datalog_window::display_dashboard() {
  if(dashboard == nullptr) return;
  if(config.get_always_draw() == false && this->isVisible() == false) return;
  if(config.get_async_display() == false) { // sync
    display_dashboard_worker();
  } else if(draw_in_progress == false) { // async
    QTimer::singleShot(0,this,SLOT(display_dashboard_worker()));
  }
}

void datalog_window::display_dashboard_worker() {
  draw_in_progress = true; // working (ignored in sync mode)
  datalog_packet *p = dashboard_pkt;
  // draw main controls
  dashboard->draw_all(p);
  // draw errors
  if(p == nullptr || p->get_msgnumber() == 0x05) {
    ui->error_text->setPlainText("DTC Errors Unavailable");
  } else {
    ui->error_text->setPlainText(def[0].err_str(p,p->log->info.patch_version.get()));
  }
  draw_in_progress = false; // finished
}

void datalog_window::parameter_text_display(QWidget *widget, int i, QString uom) {
  QLineEdit *w = (QLineEdit *)widget;
  w->setText(QString("%1 %2").arg(i).arg(uom));
}

void datalog_window::parameter_text_display(QWidget *widget, int i) {
  QLineEdit *w = (QLineEdit *)widget;
  w->setText(QString::number(i));
}

void datalog_window::parameter_text_display(QWidget *widget, float i, int precision) {
  QLineEdit *w = (QLineEdit *)widget;
  w->setText(QString::number((double)i,'f',precision));
}

void datalog_window::parameter_text_display(QWidget *widget, float i, int precision, QString uom) {
  QLineEdit *w = (QLineEdit *)widget;
  w->setText(QString("%1 %2").arg((double)i,0,'f',precision).arg(uom));
}

void datalog_window::parameter_text_display(QWidget *widget, QString str) {
  QLineEdit *w = (QLineEdit *)widget;
  w->setText(str);
}

void datalog_window::parameter_indicator_display(QWidget *widget, bool i) {
  QLineEdit *w = (QLineEdit *)widget;
  QPalette palette;
  if(i == 1) {
    palette.setColor(QPalette::Base,QColor(INDICATOR_GREEN,255));
  } else {
    //palette.setColor(QPalette::Base,QColor(INDICATOR_NEUTRAL,255));
  }
  w->setPalette(palette);
}

void datalog_window::parameter_err_display(QWidget *widget, bool i) {
  if(i == 1) {
    qtextedit_bgcolor(widget,INDICATOR_RED);
  } else {
    qtextedit_bgcolor(widget,INDICATOR_NEUTRAL);
  }
}

void datalog_window::lineedit_indicator_display(QLineEdit *edit, bool i) {
  QPalette palette;
  if(i == true) {
    palette.setColor(QPalette::Base,QColor(INDICATOR_GREEN,255));
  }
  edit->setPalette(palette);
}

void datalog_window::lineedit_error_display(QLineEdit *edit, bool i) {
  QPalette palette;
  if(i == true) {
    palette.setColor(QPalette::Base,QColor(INDICATOR_RED,255));
  }
  edit->setPalette(palette);
}

void datalog_window::qtextedit_bgcolor(QWidget *widget, int r, int g, int b) {
  QTextEdit *w = (QTextEdit *)widget;
  w->setStyleSheet(QString("background-color: rgb(%1,%2,%3)").arg(r).arg(g).arg(b));
}

void datalog_window::unblock_ui() {
  QCoreApplication::processEvents();
}

void datalog_window::info_dialog(QString q) {
  emit display_info_dialog(q);
}

void datalog_window::error_dialog(QString q) {
  emit display_error_dialog(q);
}

void datalog_window::append_memo(QString str) {
  emit add_new_note(str);
}

void datalog_window::move_to_record(datalog_packet *p) {
  if(p == nullptr) {
    playback_pkt = nullptr;
    playback_status = false;
    ui->display_currentrecord->setText("NO DATA");
    if(dashboard != nullptr) dashboard->draw_all(nullptr);
    emit display_new_packet(nullptr);
    return;
  }
  playback_pkt = p;
  emit log_position_changed(p->timestamp);
  if(config.get_always_draw() == false && this->isVisible() == false) return;
  ui->display_currentrecord->setText(
        QString("#%1 @ ").arg(playback_pkt->sequence) +
        QString().number((float)playback_pkt->timestamp / 1000,'f',1));
  slider_canary = true;
  ui->time_slider->setValue(playback_pkt->sequence); // avoid loop
  slider_canary = false;
  move_dashboard(p);
  emit display_new_packet(p);
}

void datalog_window::move_dashboard(datalog_packet *p) {
  if(p->is_msg(0xF4,0x00) ||
     (p->is_msg(0xF4,0x05) &&
      p->log->info.patch_version.get() >= 0x01)
      ) { // belongs to us
    dashboard_pkt = p;
    if(ui->mainTabSwitcher->currentIndex() == 0) display_dashboard();
  } else {
    datalog_packet *prev_pkt = p->get_prev(0xF4,0x00);
    if(prev_pkt == dashboard_pkt) return; // we're already there
    dashboard_pkt = prev_pkt;
    if(ui->mainTabSwitcher->currentIndex() == 0) display_dashboard();
  }
}

bool datalog_window::calid_validate() {
  if(ui->calid_display->text().length() > 1 && ui->calid_display->text().toUInt() > 1 &&
     ui->calid_display->text().toUInt() < 4294967294) return true;
  return false;
}

bool datalog_window::vin_validate() {
  if(ui->vinnumber_display->text().length() == 17) return true;
  return false;
}

//-------------------------------------

void datalog_window::recieve_new_packet(datalog_packet *display_pkt) {
  if(display_pkt == nullptr) return;
  if(log == log->get_first()) { // record log active
    move_to_record(display_pkt);
    update_pktcount();
  }
  unsaved_log = true;
}

void datalog_window::recieve_new_pktrate(float rate) {
  parameter_text_display(ui->display_pktrate,rate,0,"bps");
}

void datalog_window::on_vinnumber_display_textChanged(const QString) {
  if(vin_validate() == false) {
    lineedit_error_display(ui->vinnumber_display,true);
  } else {
    lineedit_indicator_display(ui->vinnumber_display,false);
  }
}

void datalog_window::on_calid_display_textChanged(const QString) {
  if(calid_validate() == false) {
    lineedit_error_display(ui->calid_display,true);
  } else {
    lineedit_indicator_display(ui->calid_display,false);
  }
}

void datalog_window::on_mainTabSwitcher_currentChanged(int index) {
  if(index == 0) display_dashboard();
}

void datalog_window::on_restart_log_btn_clicked() {
  if(ui->active_log_selector->currentIndex() == 0) {
    QMessageBox msgBox;
    msgBox.setText("This log is only stored in memory!!");
    msgBox.setInformativeText("Clearing this log will destroy the data forever.  Are you sure you want to discard the data?");
    msgBox.setIcon(QMessageBox::Warning);
    msgBox.setStandardButtons(QMessageBox::Discard | QMessageBox::Cancel);
    msgBox.setDefaultButton(QMessageBox::Discard);
    int ret = msgBox.exec();
    if(ret == QMessageBox::Discard) clear_running_log();
  } else {
    datalog *del_log = log;
    int index_to_remove = ui->active_log_selector->currentIndex();
    ui->active_log_selector->setCurrentIndex(index_to_remove - 1);
    ui->active_log_selector->removeItem(index_to_remove);
    delete del_log;
  }
}

void datalog_window::on_export_data_btn_clicked() {
  if(log != log->get_first()) return; // not the first log
  if(save_raw_file(log->get_first()) == true) {
    load_log_from_file(last_save_filename);
    clear_running_log();
  }
}

void datalog_window::clear_running_log() {
  log->get_first()->restart_log();
  update_pktcount();
  dashboard_pkt = nullptr;
  recieve_new_packet(nullptr);
  dashboard->draw_all(nullptr);
  unsaved_log = false;
}

void datalog_window::on_clear_dtc_btn_clicked() {
  control->clear_dtc.request();
}

void datalog_window::on_getecminfo_btn_clicked() {
  control->ecminfo.request();
}

void datalog_window::on_btn_beginning_clicked() {
  move_to_record(log->oldest_packet());
}

void datalog_window::on_btn_stepback_clicked() {
  if(playback_pkt == nullptr) return;
  if(playback_pkt->get_prev() == nullptr) return;
  move_to_record(playback_pkt->get_prev());
}

void datalog_window::on_btn_play_clicked() {

  if(playback_status == true) {
     stop_playback();
  }  else {
     start_playback();
  }
}

void datalog_window::start_playback() {
  wb_reset_cache();
  playback_status = true;
  // if at end or whatever ...
  if(playback_pkt == nullptr || playback_pkt->is_last_packet()) {
    playback_pkt = log->oldest_packet(); // try to rewind...
    if(playback_pkt == nullptr) return; // failing that go null.
  }
  ui->btn_play->setIcon(QIcon(":/icons_new/playback_pause_icon&16.png"));
  playback_timer->setInterval(0); // first one should be instant .
  playback_timer->start();
}

void datalog_window::stop_playback() {
  playback_timer->stop();
  playback_status = false;
  ui->btn_play->setIcon(QIcon(":/icons_new/playback_play_icon&16.png"));
}

void datalog_window::advance_playback_pkt() {
  if(playback_pkt == nullptr) return;

  // this could happen if we use the 'skip to end' button while playing.
  if(playback_pkt->is_last_packet() == true) {
    stop_playback();
    return;
  }

  // draw the thing and figure out how long it took
  QElapsedTimer draw_speed;
  draw_speed.start();
  move_to_record(playback_pkt->get_next());
  int draw_interval = draw_speed.elapsed();

  // handle end of datalog reached
  if(playback_pkt == nullptr || playback_pkt->is_last_packet() == true) {
    stop_playback();
    return;
  }

  // calculate delay to next packet, factoring in the time we just spent drawing this one
  float current_timestamp = playback_pkt->timestamp;
  float interval = ( ( playback_pkt->get_next()->timestamp - current_timestamp ) / ui->playback_speed_box->value() )  - draw_interval;

  // while less than minimum delay, just skip more packets ...
  while(playback_pkt->is_last_packet() == false && interval < MIN_DATALOG_PLAYBACK_GAP) {
    playback_pkt = playback_pkt->get_next();
    interval = ( ( playback_pkt->get_next()->timestamp - current_timestamp ) / ui->playback_speed_box->value() )  - draw_interval;
  }

  // hard limit at maximum
  if(interval > MAX_DATALOG_PLAYBACK_GAP) interval = MAX_DATALOG_PLAYBACK_GAP;

  // reset interval.
  playback_timer->setInterval(interval);
}

void datalog_window::on_btn_stepforward_clicked() {
  if(playback_pkt == nullptr) return;
  if(playback_pkt->get_next() == nullptr) return;
  move_to_record(playback_pkt->get_next());
}

void datalog_window::on_btn_end_clicked() {
  move_to_record(log->latest_packet());
}

void datalog_window::on_time_slider_valueChanged(int value) {
  if(slider_canary == true) return;
  datalog_packet *new_pkt = log->get_packet_by_sequence(value);
  if(new_pkt != nullptr) move_to_record(new_pkt);
}

void datalog_window::on_setcal_btn_clicked() {
  if(calid_validate() == false) {
    info_dialog("ERROR: Your calibration ID is invalid, and cannot be programmed.");
  } else {
    log->info.calibration_id.set(ui->calid_display->text().toUInt());
    control->set_calid.request();
  }
}

void datalog_window::on_setvin_btn_clicked() {
  if(vin_validate() == false) {
    info_dialog("ERROR: Your VIN# is invalid, and cannot be programmed.\nPerhaps it's too short.");
  } else {
    log->info.vin_number.set(ui->vinnumber_display->text());
    control->set_vin.request();
  };
}

void datalog_window::on_import_data_btn_clicked() {
  QStringList filename = get_log_select_dialog();
  if(filename.isEmpty() == true) return;
  for(int x=0;x<filename.size();x++) {
    if(filename.at(x).isNull() || filename.at(x).isEmpty()) continue;
    load_log_from_file(filename.at(x));
  }
}

void datalog_window::update_datalog_context() {
  int current_index = ui->active_log_selector->currentIndex();
  if(current_index == 0) { // recorded data
    ui->export_data_btn->setEnabled(true); // only raw export first log
  } else {
    ui->export_data_btn->setEnabled(false);
  }
}

void datalog_window::on_active_log_selector_currentIndexChanged(const QString &arg1) {
  datalog *new_log = log->get_by_name(arg1);
  update_datalog_context();
  if(new_log == nullptr) {
    error_dialog("The log has been invalidated.  It'll be removed now.");
    // FIXME remove log here
    return;
  }
  log = new_log;
  update_pktcount();
  emit active_log_changed(new_log);
  emit patch_level_changed_(new_log->info.patch_version.get());
  move_to_record(log->latest_packet());
  update_ecm_info();
  // enable some controls if we're in an actual logging situation
  bool enable_controls = false;
  if(arg1 == RECORDED_DATA_LOGNAME) enable_controls = true;
  ui->grp_ecminfo->setEnabled(true);
  ui->setvin_btn->setEnabled(enable_controls);
  ui->setcal_btn->setEnabled(enable_controls);
  ui->calid_display->setEnabled(enable_controls);
  ui->patch_status->setEnabled(enable_controls);
  ui->vinnumber_display->setEnabled(enable_controls);
  ui->clear_dtc_btn->setEnabled(enable_controls);
}

dashboard_field::dashboard_field(QLineEdit *field_in, datalog_element *element_in, bool is_indicator) {
  configure_element(element_in);
  indicator = is_indicator;
  widget = field_in;
  //---------
  cache_bool = true; // this will go false when we first draw the dashboard and force an intial draw .
  draw(nullptr);
}

void dashboard_field::configure_element(datalog_element *element_in) {
  main_element = element_in; // set main element
  if(main_element == nullptr) {
    speedlog_element = nullptr;
    return;
  }
  // find alternate msg5 element
  datalog_definition *alt_def = main_element->def->get_msg_number(0xF4,0x05);
  if(alt_def == nullptr) {
    speedlog_element = nullptr;
    return;
  }
  speedlog_element = alt_def->element_by_name(main_element->short_name);
}

void dashboard_field::draw(datalog_packet *pkt) { 
  // handle no pkt ..
  if(pkt == nullptr) {
    if(indicator == true) {
      display_bool(false);
    } else {
      display_str(QString(),INDICATOR_STYLE_NEUTRAL);
    }
    return;
  }

  // here we're going to default to the normal element unless this packet came from speedlog
  // message 5 with a patch.  if not we just bail.
  datalog_element *displ_element = nullptr;
  if(pkt->get_msgnumber() == 0x00) {
    displ_element = main_element;
  } else if(pkt->get_msgnumber() == 0x05 && pkt->log->info.patch_version.get() > 0x00) {
    displ_element = speedlog_element;
  } else {
    return;
  }

  if(displ_element == nullptr) {
    if(indicator == true) {
      display_bool(false);
    } else {
      display_str(QString());
    }
    return;
  }

  if(indicator == true) {
    display_bool(displ_element->get_bool(pkt));
  } else {
#ifdef ENABLE_DATALOG_INDICATORS
    if(displ_element->is_low(pkt) == true) {
      display_str(displ_element->get_string(pkt),INDICATOR_STYLE_RED);
    } else if(displ_element->is_high(pkt) == true) {
      display_str(displ_element->get_string(pkt),INDICATOR_STYLE_GREEN);
    } else {
      display_str(displ_element->get_string(pkt),QString());
    }
#else
    display_str(displ_element->get_string(pkt),QString());
#endif
  }
}

void dashboard_field::display_str(QString str, QString style) {
  if(indicator == true) return;
  if(str == cache_str) return;
  cache_str = str;
  widget->setText(str);
#ifdef ENABLE_DATALOG_INDICATORS
  if(widget->styleSheet() != style) widget->setStyleSheet(style);
#endif
}

void dashboard_field::display_bool(bool value) {
  if(indicator == false) return;
  if(cache_bool == value) return; // cache hit
  cache_bool = value;
  if(value == true) {
    widget->setStyleSheet(QString(INDICATOR_STYLE_GREEN));
  } else {
    widget->setStyleSheet(QString());
  }
}

void dashboard_field::draw_all(datalog_packet *pkt) {
  draw(pkt);
  if(next != nullptr) next->draw_all(pkt); // recurse
}

dashboard_field *dashboard_field::get_last() {
  dashboard_field *x = this;
  while(x->next != nullptr) x = x->next;
  return x;
}

void datalog_window::on_btn_dash_user_a_clicked() {
  set_user_parameter(ui->label_dash_user_a,dash_user_a);
}

void datalog_window::on_btn_dash_user_b_clicked() {
  set_user_parameter(ui->label_dash_user_b,dash_user_b);
}

void datalog_window::on_btn_dash_user_c_clicked() {
  set_user_parameter(ui->label_dash_user_c,dash_user_c);
}

void datalog_window::on_btn_dash_user_d_clicked() {
  set_user_parameter(ui->label_dash_user_d,dash_user_d);
}

void datalog_window::set_user_parameter(QLabel *label, dashboard_field *dashfield) {
  parameter_select parameter_dialog(this,def,log); // restrict to this msg
  datalog_element *e;
  if(parameter_dialog.exec() == QDialog::Accepted) {
    e = parameter_dialog.get_result();
  } else {
    return;
  }
  if(e == nullptr) return; // if no parameter selected ...
  dashfield->configure_element(e);
  label->setText(e->long_name);
  display_config_changed();
}
