#include <QApplication>
#include <QMainWindow>
#include <QThread>
#include <chrono>
#include <thread>

#include "datastream.h"
#include "datalog.h"
#include "config.h"
#include "common.h"

datastream::datastream(datastream_control *control_in, datalog *log_in, datalog_definition *def_in) {
  // this happens before thread creation; beware!!!
  control = control_in; // set control structure
  log = log_in;
  post_delay_amount = 0;
  pkt_fail_counter = 0;
  def = def_in;
  restart_garbage_collection();
}

bool datastream::serial_discard_new(int gap, int timeout) {
  QElapsedTimer t;
  t.start();

  QByteArray discarded;

  while(port->waitForReadyRead(gap) == true) {
    discarded.append(port->readAll());
    if(t.elapsed() > timeout) {
      return true;
    }
  }

  if(discarded.isEmpty() == false) {
    return true;
  } else {
    return false;
  }
}

bool datastream::silence_module_new(unsigned char device) {
  // the basic concept is we want to hit our heartbeat frame in the middle, because of unknown interface lag.
  // each iteration we add an additional 1ms of pre-delay but reset that pre-delay at 10 in case we've missed.
  // this should cover most serial timing possibilities and error rates.

  // store request as bytes just because we might be using it quite a bit
  set_length(4);
  set_mode(0x08);
  set_device(device);
  checksum_generate();

  // we only need half the heartbeat due to preemption.
  QByteArray heartbeat;
  heartbeat.append(0xF0);
  heartbeat.append(0x56);

  // for main timeout
  QElapsedTimer total_elapsed;
  total_elapsed.start();

  QSettings s;

  // configuration -------------------------------------------------
  // default to a reliable configuration
  int timeout = 20000;             // ms until fail
  int min_silence_length = 300;    // ms of silence until win
  int max_heartbeat_window = 12;   // heartbeat frame must be in last window bytes
  int buf_err_siz = 1024;          // when buffer is this large, probably timing is lunched
  int alt_heartbeat_window = 500;  // ... so use a massive window instead
  int predelay = 0;                // pre-emption delay (seems like 2ms would be good)
  int max_predelay = 10;           // when predelay exceeds this ms value reset to 0
  int postdelay = 12;              // ms after silence request is sent
  //----------------------------------------------------------------

  // secondary buffer
  QByteArray buf_in_alt;

  // run main loop until timeout expires
  while(total_elapsed.elapsed() < timeout) {

    // get data...
    buf_in_alt.append(port->readAll());

    // wait for just the first two bytes of the heartbeat to appear in the secondary buffer within the window.
    if(buf_in_alt.right(max_heartbeat_window).contains(heartbeat) == true) {

      // restart the buffer to clear the current heartbeat out of the window
      buf_in_alt.clear();

      // incremental but resetting predelay
      sleep_ms(predelay);
      predelay++;
      if(predelay > max_predelay) predelay = 0;

      // send the request
      serial_write_raw(4);

      // post delay and clear.  this effectively removes some of the search time for min_silence_length,
      // but doesn't seem to hurt.  there's no way any action will be required in this time.
      sleep_ms(postdelay);
      port->clear();
    }

    // see if the buffer is filling heavily, which means we're looping hard and overshooting our window
    // probably due to a really laggy serial interface.
    if(buf_in_alt.size() > buf_err_siz) {
      max_heartbeat_window = alt_heartbeat_window; // switch to large window size for remaider
    }

    // wait for more data or (hopefully) silence
    if(port->waitForReadyRead(min_silence_length) == false) {
      port->readAll(); // just in case ...
      return true; // bus silent
    }
  }
  return false;
}

bool datastream::silence_modules() {
  debug.print("DATASTREAM: Attempting to become bus master...");

  int n_retries = 3;

  for(int x=0;x<n_retries;x++) {

    // if there is lets check for a heartbeat, if there is, silence the bus master device.
    master_id = wait_for_heartbeat(500);
    if(master_id > 0) {
      bool silence_success = silence_module_new(master_id);
      if(silence_success == true) {
        return true;
      }
    } else {
      // now see if the bus is actually silent.
      port->waitForReadyRead(500);
      QByteArray noise = port->readAll();
      if(noise.isEmpty()) {
        return true;
      }
    }
  }

  return false;
}

unsigned char datastream::wait_for_heartbeat(int msTimeout) {
  // ** thanks to spfautsch for this routine **
  // listen to the bus for broadcast heartbeat
  // returns bus master device byte or zero if timeout reached

  // this routine has been modified for bus smashing as interface latency is a huge variable.

  QElapsedTimer t;
  t.start();

  port->clear();
  serial_discard_new(10,2000);

  QByteArray buf_in;

  while(t.hasExpired(msTimeout) == false) {
    if(port->waitForReadyRead(20) == false) continue;

    buf_in.append(port->readAll());

    // minimum 4 bytes.
    if(buf_in.size() < 4) continue;

    // the location we are testing for vailidity as a heartbeat.  start 4 bytes back.
    int cursor = buf_in.size() - 1 - 4;

    while(cursor > 0) {
      cursor--;
      if((unsigned char)buf_in.at(cursor) == 0xF0 &&
         (unsigned char)buf_in.at(cursor + 1) == 0x56 &&
         (unsigned char)buf_in.at(cursor + 2) > 0x52) {
        if(checksum_test((unsigned char*)buf_in.mid(cursor,4).data(),4) == true) {
          return (unsigned char)buf_in.at(cursor + 2); // valid.  return id of bus master.
        } else {
          continue;
        }
      }
    }
  }

  return 0x00;
}

void datastream::force_raw_resume_request(byte device) {
  debug.print_verbose("DATASTREAM: Resuming device " + QString::number(device,16));
  set_length(4);
  set_mode(0x09);
  set_device(device);
  checksum_generate();
  serial_write_raw(4);
  post_delay_amount = 0;
  restart_garbage_collection();
  serial_discard_all(SHUTUP_CLEARING_TIME);
}

bool datastream::send_keepalive() {
  debug.print_verbose("DATASTREAM: Send keepalive",buf_out,4);
  set_device(0x00);
  set_mode(0x00);
  set_length(4);
  checksum_generate();
  return serial_write(4);
}

void datastream::request_resume() {
  if(master_id == 0) return;
  control->enable_errors.disable(); // don't generate error counts
  msleep(25);
  force_raw_resume_request(master_id);
  control->enable_errors.enable(); // end error ignorance
  post_delay_amount = 0;
  restart_garbage_collection();
}

bool datastream::m8_silence_request() {
  serial_discard_all();
  set_device(PRIMARY_ECM);
  set_mode(0x08);
  if(aldl_request_reply_length(4,4) == false) {
    return false;
  }
  return true;
}

datalog_packet *datastream::acquire_mode1_packet(datalog_definition *def) {
  if(def == nullptr || def->is_empty() == true) return nullptr;
  set_device(def->get_device());
  set_mode(0x01); // mode
  set_msg(def->get_msg_number()); // message
  int reply_size = aldl_request_reply_undefined(5,500);
  if(reply_size != 0) {
    // process data and emit signal
    datalog_packet *new_pkt = log->add_mode1_packet(buf_in,def,reply_size);
    emit(new_packet_ready(new_pkt));
    return new_pkt;
  } else {
    return nullptr;
  }
}

bool datastream::get_ecminfo() {
  // msg4 packet vinnumber/calid
  bool success = true;
  datalog_definition *msg4def = def->get_msg_number(0xF4,0x04); // search for msg 4
  datalog_packet *infopkt = acquire_mode1_packet(msg4def);
  if(infopkt == nullptr) {
    log->info.calibration_id.set(0);
    log->info.vin_number.set("ERROR");
    success = false;
  } else {
    log->info.calibration_id.set(infopkt->get_int("CALID"));
    log->info.vin_number.set(msg4def->element_by_name("VINNUMBER")->get_ascii(infopkt));
    emit(new_packet_ready(infopkt)); // might as well use it
  }

  // transmission type
  int trans_byte_raw = m3_get(0x20,0x26);
  if(trans_byte_raw == -9999) {
    log->info.auto_trans.set(true); // default auto?
    success = false;
  } else {
    log->info.auto_trans.set(getbit(6,trans_byte_raw));
  }

  // patch version
  if(config.get_disable_patch() == true) {
    log->info.patch_version.set(0x00);
  } else {
    int raw_patch_version = m3_get(0x3C,0xB3);
    if(raw_patch_version == -9999) {
      log->info.patch_version.set(0x00);
      success = false;
    } else {
      log->info.patch_version.set(raw_patch_version);
    }
  }

  if(log->info.patch_version.get() >= 0x02) { // eside comms OK
    int raw_stat_a = m3_get(0x20,0x28,0xE4);
    if(raw_stat_a != -9999) { // eside comms definitely OK
      log->info.eside_comms_basic.set(true); // confirm that eside comms are ok
      log->info.maf_enable.set(!getbit(1,(byte)raw_stat_a));
    } else {
      log->info.eside_comms_basic.set(false);
    }
  } else {
    log->info.eside_comms_basic.set(false);
  }

  log->info.valid.set(success);
  return success;
}

QString datastream::custom_command_as_hexstring(int postdelay) {
  QString str_out;
  int len = buf_out[1] - 0x52; // get length from command
  // command
  str_out.append("TX+");
  str_out.append(aldl_as_colored_hexstring(buf_out));
  int in_len = aldl_request_reply_undefined(len,200,postdelay);
  str_out.append("<br>RX+");
  if(in_len == 0) {
    str_out.append("NO REPLY");
  } else {
    str_out.append(aldl_as_colored_hexstring(buf_in));
  }
  return str_out;
}

byte datastream::get_raw_data(byte offset) {
  // this comparison is technically useless if MAX_READ_BUF is greater than the sizeof byte?
  // ... which it is unless someone screws with it.
  //if(offset >= MAX_READ_BUF) return 0x00; // out of range
  return buf_in[offset];
}

QString datastream::aldl_as_colored_hexstring(byte *buf) {
  int len = buf[1] - 0x52; // get length from command
  return buf_as_colored_hexstring(buf,len);
}

QString datastream::buf_as_colored_hexstring(byte *buf, int len) {
  QString str_out;
  QByteArray out;
  out.setRawData((char *)buf,1); // ecm and length
  str_out.append(out.toHex().toUpper());
  out.setRawData((char *)buf + 1,1); // ecm and length
  str_out.append(out.toHex().toUpper());
  out.setRawData((char *)buf + 2,1); // mode
  str_out.append(out.toHex().toUpper());
  if(len > 3) {
    out.setRawData((char *)buf + 3,len - 4); // data
    str_out.append(out.toHex().toUpper());
  }
  if(len > 2) {
    out.setRawData((char *)buf + len - 1,1); // cksum
    str_out.append(out.toHex().toUpper());
  }
  return str_out;
}

byte datastream::checksum_generate(byte *buf, int len) {
  int x = 0;
  unsigned int sum = 0;
  for(x=0;x<len;x++) sum += buf[x];
  return ( 256 - ( sum % 256 ) );
}

byte datastream::checksum_generate(int len) {
  buf_out[len] = checksum_generate((byte*)&buf_out,len);
  return buf_out[len];
}

byte datastream::checksum_generate() {
  return checksum_generate(buf_out[1] - 0x52 - 1);
}

bool datastream::checksum_test(byte *buf, int len) {
  int x = 0;
  unsigned int sum = 0;
  for(x=0;x<len;x++) sum += buf[x];
  if((sum & 0xFF) == 0) {
    return true;
  } else {
    return false;
  }
}

bool datastream::checksum_test() {
  int len = buf_in[1] - 0x52;
  return checksum_test(buf_in,len);
}

void datastream::set_device(byte d) {
  buf_out[BDEVICE] = d;
}

void datastream::set_mode(byte m) {
  buf_out[BMODE] = m;
}

void datastream::set_msg(byte m) {
  buf_out[BMSG] = m;
}

void datastream::set_length(int len) {
  buf_out[1] = calc_msglength(len);
}

bool datastream::is_serial_port_open() {
  return port->isOpen();
}

bool datastream::clear_dtc() {
  set_device(PRIMARY_ECM);
  set_mode(0x0A);
  return aldl_request_reply_length(4,4,100);
}

//-----------------

void datastream::restart_garbage_collection() {
  garbage_collection_amount = GARBAGE_COLLECTION_MAXTIME;
}

void datastream::increase_garbage_collection() {
  garbage_collection_amount += 5;
  if(garbage_collection_amount > GARBAGE_COLLECTION_MAXTIME) garbage_collection_amount = GARBAGE_COLLECTION_MAXTIME;
}

void datastream::decrease_garbage_collection() {
  garbage_collection_amount -= 3;
  if(garbage_collection_amount < GARBAGE_COLLECTION_MINTIME) garbage_collection_amount = GARBAGE_COLLECTION_MINTIME;
}

//-----------------

error_result::error_result() {
  reset();
}

void error_result::reset() {
  errstring = QString("Undefined Error");
}

void error_result::set_errstring(QString err) {
  errstring = err;
}

void error_result::set_errstring(QString err, byte b) {
  errstring = err;
  err.append(QString::number(b,16));
}

QString error_result::get_errstring() {
  return errstring;
}

//-----------------

int datastream::aldl_read_reply(int timeout) {
  QElapsedTimer time_spent;
  time_spent.start();
  // get header
  if(serial_read_bytestream((byte*)&buf_in, 2, timeout) == false) return 0;
  // determine length
  int pkt_length(clamp(buf_in[1] - 0x52,0x01,0xFF));
  // mid delay
  sleep_ms(THROTTLE_MS);
  // get data @ offset, also determine new timeout of serial timing not request timing...
  if(serial_read_bytestream((byte*)&buf_in[2], pkt_length - 2, timeout - time_spent.elapsed()) == false) {
    serial_error_occurred();
    pkt_fail_counter++;
    return 0;
  }
  // look for bad checksum
  if(checksum_test() == false) {
    serial_error_occurred();
    pkt_fail_counter++;
    return 0;
  }
  // good to go!
  decrease_garbage_collection();
  pkt_fail_counter--;
  if(pkt_fail_counter < 0) pkt_fail_counter = 0;
  return pkt_length;
}

int datastream::serial_read_any() {
  int read_bytes = 0;
  int buf_in_cursor = 0; // redundant ??
  forever {
    port->waitForReadyRead(4);
    if(MAX_READ_BUF - buf_in_cursor < 1) buf_in_cursor -= 50; // prevent limit @ overrun
    read_bytes = port->read((char *)&buf_in[buf_in_cursor], MAX_READ_BUF - buf_in_cursor);
    if(read_bytes > 0) { // got some data
      buf_in_cursor += read_bytes;
      msleep(4);
    } else {
      return buf_in_cursor;
    }
  }
  return buf_in_cursor;
}

bool datastream::serial_read_bytestream(byte *dest, int n_bytes, int timeout) {
  if(n_bytes > MAX_READ_BUF) return false;
  QElapsedTimer time_spent;
  time_spent.start();
  int read_bytes = 0;
  int buf_in_cursor = 0; // redundant ??
  if(timeout < 25) timeout = 25; // 25ms timeout minimum
  while(buf_in_cursor < n_bytes) {
    read_bytes = port->read((char *)&dest[buf_in_cursor], n_bytes - buf_in_cursor);
    if(read_bytes > 0) { // got some data
      buf_in_cursor += read_bytes;
      time_spent.start(); // reset timeout !!
    } else if(read_bytes == -1) { // read error
      debug.print("SERIAL ERR: Serial driver error while reading buffer.");
      return false;
    } else if(read_bytes == 0) {
      if(time_spent.hasExpired(timeout) == true) {
        debug.print("DATASTREAM ERR: Timeout waiting for incoming packet.");
        return false;
      } else {
        int data_remaining = n_bytes - buf_in_cursor;
        // extra delay if there's lots of data left, to increase block size
        // this dynamic throttling may be unnecessary, requires testing.
        if(data_remaining > 12) {
          msleep(THROTTLE_MS);
          port->waitForReadyRead(THROTTLE_MS * 2);
        } else if(data_remaining > 3) {
          port->waitForReadyRead(THROTTLE_MS);
        } else {
          port->waitForReadyRead(1);
        }
      }
    }
  }
  debug.print_verbose("READ: ",dest,n_bytes);
  post_delay_timer.start(); // restart delay timer here.
  keepalive_timer.start();
  return true;
}

void datastream::aldl_reqdup(byte *req, int req_len) {
  // cp request to buf_out
  if(req != buf_out && req != nullptr) for(int x=0;x<req_len;x++) buf_out[x] = req[x];
}

bool datastream::aldl_request_reply_specific(byte *req, int req_len, byte *repl, int repl_len, int timeout, int post_delay) {
  aldl_reqdup(req,req_len);
  return aldl_request_reply_specific(req_len, repl, repl_len, timeout, post_delay);
}

bool datastream::aldl_request_reply_specific(int req_len, byte *repl, int repl_len, int timeout, int post_delay) {
  if(aldl_request_reply_length(req_len,repl_len,timeout,post_delay) == false) {
    debug.print("DATASTREAM ERR: No reply or incorrect reply length.");
    debug.print("DATASTREAM ERR: Expected: ",repl,repl_len);
    return false;
  }
  if(repl == nullptr) return true; // skip reply
  // don't compare length/mode/device/checksum, they are compared earlier!
  for(int x=BMSG;x<repl_len - 1;x++) {
    if(repl[x] != buf_in[x]) {
      debug.print("DATASTREAM ERR: Expected reply: ",repl,repl_len);
      debug.print("DATASTREAM ERR: Got reply: ",buf_in,repl_len);
      serial_error_occurred();
      return false;
    }
  }
  return true;
}

bool datastream::aldl_request_reply_length(int req_len, int repl_len, int timeout, int post_delay) {
  // send request and get reply
  int in_length = aldl_request_reply_undefined(req_len, timeout, post_delay);
  // check for err
  if(in_length == 0) {
    debug.print("DATASTREAM ERR: No reply.");
    return false;
  }
  // check for length mismatch
  if(in_length != repl_len) {
    debug.print("DATASTREAM ERR: Reply length incorrect.: ",buf_in,in_length);
    serial_error_occurred();
    return false;
  }
  return true;
}

int datastream::aldl_request_reply_undefined(byte *req, int req_len, int timeout, int post_delay) {
  aldl_reqdup(req,req_len);
  return aldl_request_reply_undefined(req_len,timeout,post_delay);
}

int datastream::aldl_request_reply_undefined(int req_len, int timeout, int post_delay) {
  // send request
  bool send_result = aldl_send_request(req_len);
  // request err
  if(send_result == false) {
    post_delay_amount = 100; // backup post delay
    return 0;
  }
  // get reply
  int in_length = aldl_read_reply(timeout); // this verifies checksum
  // no data?
  if(in_length == 0) {
    post_delay_amount = 100; // backup post delay
    return 0; // skip sleep step if no data
  }
  // verify device and mode are equal
  if(buf_in[BDEVICE] != buf_out[BDEVICE] || buf_in[BMODE] != buf_out[BMODE]) {
    debug.print("DATASTREAM ERR: Device/mode mismatch: ",buf_in,in_length);
    post_delay_amount = 100; // backup post delay
    serial_error_occurred();
    return 0;
  }
  // data ok
  if(post_delay == 0) {
    post_delay_amount = mode_postdelay(buf_out[2]);
  } else {
    post_delay_amount = post_delay;
  }
  return in_length;
}

bool datastream::aldl_request_reply_none(byte *req, int req_len, int post_delay) {
  aldl_reqdup(req,req_len);
  return aldl_request_reply_none(req_len, post_delay);
}

bool datastream::aldl_request_reply_none(int req_len, int post_delay) {
  // send request
  bool result = aldl_send_request(req_len);
  // data ok
  if(post_delay == 0) {
    post_delay_amount = mode_postdelay(buf_out[2]);
  } else {
    post_delay_amount = post_delay;
  }
  return result;
}

bool datastream::aldl_send_request(int req_len) {
#ifdef ECHO_NO_ERR
  serial_discard_all(50);
#else
  if(garbage_collection_amount > 0) serial_discard_all(garbage_collection_amount); // flush potential garbage
#endif
  //--------- post/delay routine!!
  if(post_delay_timer.isValid() == true && post_delay_amount > 0) { // do post delay
    // do post delay relative to time elapsed since last command
    timestamp_t time_elapsed = post_delay_timer.elapsed();
    post_delay_timer.invalidate(); // invalidate for next time
    post_delay_amount = clamp(post_delay_amount - time_elapsed,0,1000);
    if(post_delay_amount > 0) sleep_ms(post_delay_amount); // minus discard time
  } else if(post_delay_amount > 0) { // no clock ... we'll just go from right now.
    sleep_ms(post_delay_amount);
  }
  post_delay_amount = 0; // reset for next iteration
  //-------- send command
  // set length byte and checksum
  buf_out[1] = calc_msglength(req_len);
  checksum_generate();
  // send request
  bool result = serial_write(req_len);
  // return result
  return result;
}

void datastream::sleep_ms(int timeout) {
  msleep(timeout);
}

void datastream::sleep_us(int timeout) {
  usleep(timeout);
}

bool datastream::serial_discard_all(int timeout) {
  int bytes_read = 0;
  QElapsedTimer t;
  t.start();
  bytes_read = port->read((char *)&buf_in[0], MAX_READ_BUF);
  if(t.hasExpired(timeout)) return true;
  if(bytes_read > 0) debug.print_verbose("DATASTREAM: Ignored incoming data for " + QString::number(timeout) + "ms");
  while(port->waitForReadyRead(timeout - t.elapsed()) == true) {
    if(control->connection.get() == false) return false;
    bytes_read = port->read((char *)&buf_in[0], MAX_READ_BUF);
    if(bytes_read == 0) return true;
    if(t.hasExpired(timeout)) return false;
  }
  return true;
}

int datastream::mode_postdelay(byte mode) {
  // list of post delays for particular instructions
  switch(mode) {
  case 0x01:
    return 2;
  case 0x02:
    return 10;
  case 0x03:
    return 10;
  case 0x04:
    return 8;
  case 0x05:
    return 60;
  case 0x06:
    return 60;
  case 0x0D:
    return 60;
  default: // unknown mode
    return 35;
  }
}

void datastream::serial_error_occurred() {
  increase_garbage_collection(); // incr garbage collection since datastream might have slag in it
  if(control->enable_errors.is_set() == true) emit error_occurred();
}

bool datastream::serial_write(int len) {
  // send data
  if(serial_write_raw(len) == false) return false;
  // delay for echo
  sleep_ms(approx_echo_time(len));
  bool echo_result = serial_discard_echo(len);;
  return echo_result;
}

bool datastream::serial_discard_echo(int len) {
#ifdef ECHO_NO_ERR
  serial_read_bytestream(buf_echo, len, 350);
  return true;
#else
  debug.print_verbose("SERIAL: Discarded serial echo.");
  // process echo
  if(serial_read_bytestream(buf_echo, len, 350) == false) { // expect echo
    debug.print(QStringLiteral("SERIAL ERR: No echo! Possibly a bad serial interface."));
    return false;
  }
  // compare echo
  for(int x=0;x<len;x++) {
    if(buf_out[x] != buf_echo[x]) {
      pkt_fail_counter++;
      serial_error_occurred();
      debug.print(QStringLiteral("SERIAL ERR: Corrupt data in echo! Possibly a bad serial interface or noise on the bus."));
      return false;
    }
  }
  return true;
#endif
}

int datastream::approx_echo_time(int len) {
  return(clamp(len / 3,1,20));
}

int datastream::approx_reply_time(int len) {
  return(clamp(len / 3,1,250));
}

bool datastream::serial_write_raw(int len) {
  if(len > MAX_WRITE_BUF) return false;
  // write
  if(port->write((char *)buf_out,len) != len) return false;
  debug.print_verbose("WRITE: ",buf_out,len);
  // synchronize
  return serial_wait_for_write_complete(len);
}

bool datastream::serial_wait_for_write_complete(int len) {
  // calculate timeout
  int timeout_ms = clamp((len * 3),100,1000);
  bool result = port->waitForBytesWritten(timeout_ms);
  if(result == true) {
    return true;
  } else {
    debug.print("SERIAL ERR: Write could not be completed.");
    return false;
  }
}

bool datastream::open_serial_port() {
  if(port->isOpen() == true) {
    port->close();
  }
  port->setPortName(control->port_name.get());
  //-----------------
  port->setBaudRate(8192);
  port->setDataBits(QSerialPort::Data8);
  port->setParity(QSerialPort::NoParity);
  port->setStopBits(QSerialPort::OneStop);
  port->setFlowControl(QSerialPort::NoFlowControl);
  QSerialPortInfo info(port->portName());
  //-----------------
  if(port->open(QIODevice::ReadWrite)) {
    debug.print("SERIAL: Opened port " + port->portName());
    debug.print("SERIAL: Port info: " + info.description() +
                " MFR " + info.manufacturer() +
                " SER#" + info.serialNumber() +
                " ID#" + info.vendorIdentifier() + ":" + info.productIdentifier());
    return true;
  } else { // fail
    debug.print("SERIAL ERR: Failed to open port " +
                port->portName() + ": " +
                port->errorString());
    return false;
  }
}

void datastream::close_serial_port() {
  if(port->isOpen() == true) {
    port->close();
  }
}

bool datastream::m4_send_if_requested() {
  if(control->m4_get_request(buf_out) == true) {
    return aldl_request_reply_length(16,4);
  } else {
    return true;
  }
}

bool datastream::m4_send_force() {
  control->m4_get_raw(buf_out);
  return aldl_request_reply_length(16,4,100);
}

void datastream::m4_force_kill() {
  control->m4_comm_init();
  m4_send_force();
}

