#include "config.h"
#include "datastream.h"
#include "datalog.h"
#include "common.h"

datalog::datalog(QString name_in, QString path_in, datalog *tail_record = nullptr) {
  timestamp.invalidate();
  p = nullptr;
  filepath.set(path_in);
  name.set(name_in);
  next = nullptr; // this is the most recent record for sure.
  if(tail_record == nullptr) { // probably first record, or will link manually later
    prev = nullptr;
  } else { // link to existing list
    m.lock();
    tail_record->m.lock();
    tail_record->next = this;
    prev = tail_record;
    tail_record->m.unlock();
    m.unlock();
  }
}

datalog::~datalog() {
  del_lock.lock(); // lock before trashing data
  if(get_next() != nullptr) get_next()->set_prev(get_prev()); // unlink
  if(get_prev() != nullptr) get_prev()->set_next(get_next()); // unlink
  restart_log(); // delete data
  del_lock.unlock();
}

datalog_packet *datalog::add_mode1_packet(byte *raw_packet, datalog_definition *msg, int size) {
  if(msg == nullptr) return nullptr;
  // create packet
  datalog_packet *new_packet = new datalog_packet(msg,this,size);
  // timestamp as offset from latest timestamp
  new_packet->timestamp = get_and_reset_timestamp() + newest_timestamp();
  // create packet data (copy from offset +3 @ source)
  if(size > MAX_M1_DATA_SIZE) size = MAX_M1_DATA_SIZE;
  for(int x=0;x<size;x++) new_packet->set_raw_byte(x,raw_packet[x + 3]);
  add_packet(new_packet);
  return new_packet;
}

bool datalog::does_message_data_exist(datalog_definition *msg) {
  if(msg == nullptr || is_data() == false) return false;
  datalog_packet *x = oldest_packet();
  while(x->is_last_packet() == false) {
    if(x->def == msg) return true;
    x = x->get_next();
  }
  return false;
}

void datalog::add_packet(datalog_packet *new_packet) {
  new_packet->set_next(nullptr); // this will be the newest packet
  if(p == nullptr) { // no data yet, this is the first packet
    // origin sequence 1, timestamp 0.
    new_packet->sequence = 1;
    new_packet->timestamp = 0;
  } else {
    new_packet->sequence = p->sequence + 1; // increment sequence
    // restrict timestamp to thresholds
    if(new_packet->timestamp < newest_timestamp() + MIN_TIMESTAMP_GAP) {
      new_packet->timestamp = newest_timestamp() + MIN_TIMESTAMP_GAP;
    } else if(new_packet->timestamp > newest_timestamp() + MAX_TIMESTAMP_GAP) {
      new_packet->timestamp = newest_timestamp() + MAX_TIMESTAMP_GAP;
    }
    p->set_next(new_packet); // set ptr to next unless this is the first packet
  }
  new_packet->set_prev(p); // link to last packet, if there is one...
  p = new_packet; // set list head
}

timestamp_t datalog::get_and_reset_timestamp() {
  if(timestamp.isValid() == false) {
    timestamp.start();
    return DEFAULT_TIMESTAMP_GAP;
  } else {
    return timestamp.restart();
  }
}

void datalog::generate_testing_data() {
  // create packet
  datalog_packet *new_packet = new datalog_packet(0x00,this,MAX_M1_DATA_SIZE);
  // timestamp
  new_packet->timestamp = newest_timestamp() + 50;
  // copy packet data
  for(int x=3;x<MAX_M1_DATA_SIZE;x++) new_packet->set_raw_byte(x-3,random_byte());
  // link old packet as previous packet
  add_packet(new_packet);
}

datalog_packet *datalog::latest_packet(int msg_no) {
  if(p == nullptr) return nullptr;
  datalog_packet *x = p;
  while(x->get_msgnumber() != msg_no) {
    x = x->get_prev();
    if(x->get_prev() == nullptr) return nullptr;
  }
  return x;
}

datalog_packet *datalog::latest_packet() {
  if(is_data() == false) return nullptr;
  return p;
}

datalog_packet *datalog::oldest_packet(int msgnumber) {
  datalog_packet *oldest = p; // start at top
  if(oldest == nullptr) return nullptr;
  while(oldest->get_prev() != nullptr) oldest = oldest->get_prev();
  while(oldest->get_msgnumber() != msgnumber && oldest->get_next() != nullptr) oldest = oldest->get_next();
  if(oldest->get_msgnumber() != msgnumber) return nullptr;
  return oldest;
}

datalog_packet *datalog::oldest_packet() {
  datalog_packet *oldest = p; // start at top
  if(oldest == nullptr) return nullptr;
  while(oldest->get_prev() != nullptr) oldest = oldest->get_prev();
  return oldest;
}

unsigned int datalog::n_packets() {
  if(p == nullptr) return 0;
  return p->sequence;
}

unsigned int datalog::n_packets(datalog_element *e) {
  datalog_packet *p = oldest_packet();
  if(p == nullptr) return 0;
  unsigned int count = 0;
  while(p != nullptr) {
    if(e->is_associated(p) == true) count++;
    p = p->get_next();
  }
  return count;
}

timestamp_t datalog::oldest_timestamp() {
  return oldest_packet()->timestamp;
}

timestamp_t datalog::newest_timestamp() {
  if(is_data() == false) {
    return 0;
  } else {
   return p->timestamp;
  }
}

void datalog::restart_log() {
  invalidate_log_timers();
  if(p == nullptr) return; // no data
  datalog_packet *oldest = p;
  while(oldest->get_prev() != nullptr) oldest = oldest->get_prev();
  datalog_packet *delete_tmp = oldest;
  while(delete_tmp != nullptr) {
    datalog_packet *next_pkt = delete_tmp->get_next();
    delete delete_tmp;
    delete_tmp = next_pkt;
  }
  p = nullptr;
}

datalog *datalog::get_next() {
  m.lock();
  datalog *x = next;
  m.unlock();
  return x;
}

datalog *datalog::get_prev() {
  m.lock();
  datalog *x = prev;
  m.unlock();
  return x;
}

datalog *datalog::get_first() {
  m.lock();
  datalog *x = this;
  while(x->prev != nullptr) x = x->prev;
  m.unlock();
  return x;
}

datalog *datalog::get_last() {
  m.lock();
  datalog *x = this;
  while(x->next != nullptr) x = x->next;
  m.unlock();
  return x;
}

datalog *datalog::get_by_name(QString search_name) {
  datalog *x = get_first();
  if(x->name.get() == search_name) {
    return x;
  }
  while(x->next != nullptr) {
    x = x->get_next();
    if(x->name.get() == search_name) {
      return x;
    }
  }
  return nullptr;
}

datalog *datalog::get_by_index(int idx) {
  datalog *x = get_first();
  int y = 0;
  while(x != nullptr) {
    if(y == idx) {
       return x;
    } else {
      x = x->get_next();
    }
  }
  return nullptr;
}

void datalog::set_next(datalog *x) {
  m.lock();
  next = x;
  m.unlock();
}

void datalog::set_prev(datalog *x) {
  m.lock();
  prev = x;
  m.unlock();
}

int datalog::get_pkt_count() {
  if(p == nullptr) return 0;
  return p->sequence;
}

datalog_packet *datalog::get_packet_by_timestamp(timestamp_t timestamp) {
  datalog_packet *x = p;
  if(x == nullptr) return nullptr;
  while(x->timestamp >= timestamp && x->get_prev() != nullptr) x = x->get_prev();
  return x;
}

datalog_packet *datalog::get_packet_by_sequence(unsigned int sequence) {
  if(p == nullptr) return nullptr;
  if(sequence > n_packets()) return nullptr;
  datalog_packet *x = p;
  while(x->sequence > sequence && x->get_prev() != nullptr) x = x->get_prev();
  return x;
}

bool datalog::is_data() {
  if(p == nullptr) return false;
  return true;
}

bool datalog::verify_datalog_timestamps() {
  if(is_data() == false) return true;
  timestamp_t prev_timestamp = 0;
  datalog_packet *current_pkt = oldest_packet();
  while(current_pkt->is_last_packet() == false) {
    if(current_pkt->timestamp < prev_timestamp) {
      return false;
    }
    if(current_pkt->timestamp > LOG_TIMESTAMP_LIMIT) {
      return false;
    }
    current_pkt = current_pkt->get_next();
  }
  return true;
}

byte datalog::random_byte() {
  byte out = (byte)(qrand() % 256);
  return out;
}

void datalog::invalidate_log_timers() {
  timestamp.invalidate();
}

