#ifndef DATASTREAM_H
#define DATASTREAM_H

#include <QtSerialPort>
#include <QElapsedTimer>

#include "config.h"
#include "safetypes.h"
#include "settings.h"
#include "debuglog_window.h"
#include "bin_file.h"
#include "datalog.h"
#include "datastream.h"

//-------------------------------------

class error_result {
public:
  error_result();
  void reset();
  void set_errstring(QString err);
  void set_errstring(QString err, byte b);
  QString get_errstring();
private:
  QString errstring;
};

//-------------------------------------

typedef enum {
  STATE_CONNECTED,
  STATE_CONNECTING,
  STATE_DISCONNECTED,
  STATE_DISCONNECTING
} conn_state;

//-------------------------------------

typedef enum { BDEVICE = 0x00, BLENGTH = 0x01, BMODE = 0x02, BMSG = 0x03 } bytepos;
typedef enum { PRIMARY_ECM = 0xF4, SECONDARY_ECM = 0xE4, MOD_CCM = 0xF1, MOD_ABS = 0xF9} device;

//-------------------------------------

class datastream_control : public QObject {
  Q_OBJECT
public:
  datastream_control();
  safe_switch connection;
  safe_switch quit;
  //-------------
  bool is_connected(); // is completely connected?
  bool is_disconnected(); // is completely disconnected?
  safe_int connection_state;
  //-----------------
  datalog_infoheader info;
  //-------------
  safe_string port_name; // serial port string
  //-------------
  safe_switch mode1speedhack;
  //-------------
  safe_switch cyldroptest;
  safe_switch blmcell_dump;
  safe_switch clear_dtc;
  safe_switch ecminfo;
  safe_switch m4_keepalive;
  //-------------
  safe_switch set_vin;
  safe_switch set_calid;
  //-------------
  safe_raw_command custom_a;
  safe_raw_command custom_b;
  safe_switch custom_idle;
  //-----------------
  safe_switch enable_errors;
  //-----------------
  datalog_definition *def;
  //-----------------
  // Mode4 commands
  safe_switch m4_updated;               // set if new mode4 string is available
  void m4_comm_init();                  // zero entire mode4 string (disable all hacks)
  void m4_comm_afr(byte afr);           // command an AFR
  void m4_drop_cyl(int cyl);            // drop a cyl or 0 = dont drop
  void m4_comm_idle(int rpm, int mode); // command an idle, mode 1=rpm 2=steps
  void m4_comm_spark(int advance, int absolute);  // command a spark, absolute =
  int m4_get_spark();
  void m4_force_blm(bool set, bool enable);
  void m4_force_cl(bool set, bool enable);
  void m4_reset_blm(bool set);
  void m4_clear_blm_reset();            // doesn't do any locking, for event clear.  use m4_reset_blm instead
  void m4_force_gear(int gear);         // commanded gear or 0=disable
  void m4_fan(int n, bool set, bool enable);         // mode::0=auto,1=on,2=off n::1=fan1 2=fan2
  void m4_ccp(bool set, bool enable);
  void m4_air(bool set, bool enable);
  void m4_ac(bool set, bool enable);
  void m4_tcc(bool set, bool enable);
  void m4_actuator(int enable_byte, int enable_bit, int actuator_byte, int actuator_bit, bool set, bool enable);
  QString m4_getpkt(); // as a hex string
  void m4_get_raw(byte *buf);
  void m4_set_raw(byte *buf);
  bool m4_get_request(byte *buf); // cp to buf if requested, else return false
  void m4_mil(bool set);
  void m4_comm_linepressure(bool enable, byte pressure);
  void m4_comm_egr(bool enable, byte pressure);
  //-----------------
  void reset();
  safe_switch start_flash_write;
  safe_switch start_flash_read;
  safe_switch start_flash_stability_test;
  safe_switch flash_exit;
  safe_switch flash_tside;
  safe_switch flash_eside;
  safe_switch skip_unused_regions;
  safe_switch enable_patches;
  safe_switch dump_ram;
  //-----------------
  bin_file write_bin;
  bin_file prev_bin;
  bin_file read_bin;
  QElapsedTimer time_connected;
private:
  //-----------------
  byte cylnum_to_m4ref(int cyl);
  byte m4_process_afr(int afr);
  //-----------------
  QMutex m4_lock;
  byte construct_mode4[16]; // mode4 buffer
};

//-------------------------------------

class datastream : public QThread {
  Q_OBJECT
public:
  //--------------------
  void run() Q_DECL_OVERRIDE {
    post_thread_create_init(); // create in-thread structs
    disconnected_event_loop(); // start disconnected loop
    QThread::exit(0);
  }
  //--------------------
  datastream(datastream_control *control_in, datalog *log_in, datalog_definition *def_in); // the all important intermediary control struct
  //--------------------
  // static pubs
  static byte checksum_generate(byte *buf, int len);  // generate a checksum
  static bool checksum_test(byte *buf, int len); // test a checksum

signals:
  void error_occurred();
  void reset_error_count();
  void new_packet_ready(datalog_packet*);
  void connection_state_changed(int);
  void acq_event_display(QString);
  void invalid_serial_port();
  void display_critical_error(QString);
  void updated_ecm_info();
  void got_ecm_info(QByteArray);
  void update_pktrate(float);
  //--------------------
  void update_blm_cells(QByteArray);
  //--------------------
  void custom_log_append(QString);
  void custom_a_copied();
  void custom_b_copied();
  //--------------------
  void start_flashprogress();
  void end_flashprogress();
  void update_tside_progress(int);
  void update_eside_progress(int);
  void new_flash_message(QString);
  void new_flash_divider(QString);
  void new_flash_error(QString);
  void enable_flash_exit(bool);
  void flash_error_occurred();
  void update_flash_speed(float);
  void flash_write_successful();
  void flash_read_successful();
  //--------------------
  void new_cylbalance_progress(QString,int);
  void new_cylbalance_result(int,int,float);

private:
  void serial_error_occurred();
  int pkt_fail_counter;
  void disconnected_event_loop();
  void connected_event_loop();
  //--------------------
  datastream_control *control;
  //--------------------
  byte get_raw_data(byte offset); // get a byte from buf_in
  //--------------------
  byte custom_cmd_a[MAX_WRITE_BUF]; // custom command a
  byte custom_cmd_b[MAX_WRITE_BUF]; // custom command b
  //--------------------
  bool is_serial_port_open();
  bool open_serial_port();
  void close_serial_port();
  //--------------------
  bool silence_modules();      // aka shutup request, mode 0x08
  bool m8_silence_request();     // wait for reply
  bool send_keepalive();        // don't let ecm return to idle traffic
  void request_resume();
  //--------------------
  datalog_packet *acquire_mode1_packet(datalog_definition *def);
  bool get_ecminfo();
  //--------------------
  int m3_get(byte addr_a, byte addr_b, byte side = 0xF4); // get single byte
  QString m3_get_str(byte addr_a, byte addr_b); // get single byte as string
  bool m2_get(byte addr_a, byte addr_b, byte side = 0xF4); // get 64 sequential bytes
  bool m2_get_cp(int addr, byte side, byte *buf); // get length bytes in buf
  bool set_vin(QByteArray vin);
  bool set_calid(quint32 calid);
  //--------------------
  bool m4_send_force();
  bool m4_send_if_requested();
  void m4_force_kill();                 // try to force m4 to null
  //--------------------
  QString m4_getpkt();                  // get m4 packet as a qstring ...
  byte m4_process_afr(int afr);         // conv AFR int to byte
  byte cylnum_to_m4ref(int cyl);         // post process cyl id
  //--------------------
  bool sched_m4_update;         // set if mode 4 packet is changed... it'll be re-sent.
  //--------------------
  bool clear_dtc();
  //--------------------
  int garbage_collection_amount;
  void restart_garbage_collection();
  void increase_garbage_collection();
  void decrease_garbage_collection();
  //--------------------
  bool upload_tside_development_routine();
  bool upload_eside_development_routine();
  bool unlock_security(byte side);
  bool enter_mode5(byte side);
  bool upload_read_routine_eside();
  bool upload_read_routine_tside();
  bool read_flash_section(byte side, byte *bin, quint16 offset);
  bool reset_ecm_devmode();
  bool
  apply_vpp_voltage();
  bool remove_vpp_voltage();
  bool upload_erase_program(byte side);
  byte wait_for_erase_complete();
  bool upload_flash_program(byte side);
  bool get_post_status();
  error_result flash_err;
  //--------------------
  // for the following functions, len is the packet length minus checksum
  // this is for legacy purposes.
  byte checksum_generate(int len); // generate with buf_out
  byte checksum_generate(); // generate assuming length byte is correct
  bool checksum_test();
  //--------------------
  QString custom_command_as_hexstring(int postdelay);
  int mode_postdelay(byte mode); // get appropriate post delay for a mode
  //--------------------
  bool write_flash(byte side, bin_file bin, quint16 offset, byte length, int postdelay);
  //--------------------
  QByteArray get_buf_in(int length);
  debuglog_controller debug;
  //--------------------
  datalog *log;
  void sleep_ms(int timeout); // delay for milliseconds, process ui while delaying.
  //--------------------
  datalog_definition *def;
  //--------------------
  eehack_settings config;
  QSerialPort *port;  // serial instance ptr
  //--------------------
  // buffers
  byte buf_in[MAX_READ_BUF];  // input buffer
  byte buf_out[MAX_WRITE_BUF]; // output buffer, requests are constructed here...
  byte buf_echo[MAX_READ_BUF];  // echo reply discard buffer
  //--------------------
  void set_device(byte d);
  void set_mode(byte m);
  void set_msg(byte m);
  void set_length(int len);
  //--------------------
  // raw serial i/o
  bool serial_write(int len); // also discards echo
  bool serial_write_raw(int len); // does not check for echo or wait for write complete
  bool serial_discard_echo(int len);
  bool serial_wait_for_write_complete(int len);
  bool serial_discard_all(int timeout = 2); // flush incoming
  int approx_echo_time(int len);
  int approx_reply_time(int len);
  //--------------------
  // aldl formatted i/o basic
  int aldl_read_reply(int timeout); // read a reply to buf_in
  bool aldl_send_request(int req_len); // send a reply from buf_out, calcs checksum and length etc.
  bool serial_read_bytestream(byte *dest, int n_bytes, int timeout); // read packet of specific length
  //--------------------
  void aldl_reqdup(byte *req, int req_len); // duplicate a request to buf_out
  //--------------------
  // aldl formatted i/o advanced, these calc msg length and checksums and verify mode bytes etc.
  // postdelay = 0 to automatically calculate post delay
  // strict match on reply or fails.  copies request to buf_out, leaving req untouched.
  bool aldl_request_reply_specific(byte *req, int req_len, byte *repl, int repl_len,
                                               int timeout = 1000, int post_delay = 0);
  // strict match on reply, using buf_out for request.  stores reply in buf_in.
  bool aldl_request_reply_specific(int req_len, byte *repl, int repl_len, int timeout = 1000, int post_delay = 0);
  // strict length match or fail, stores reply in buf_in
  bool aldl_request_reply_length(int req_len, int repl_len, int timeout = 1000, int post_delay = 0);
  // undefined reply length, but requires mode to match the request mode.
  int aldl_request_reply_undefined(int req_len, int timeout = 1000, int post_delay = 0);
  int aldl_request_reply_undefined(byte *req, int req_len, int timeout = 1000, int post_delay = 0);
  // no expected reply at all.  rare.
  bool aldl_request_reply_none(int req_len, int post_delay = 0);
  bool aldl_request_reply_none(byte *req, int req_len, int post_delay = 0);
  //--------------------
  QElapsedTimer keepalive_timer;
  QElapsedTimer post_delay_timer;
  int post_delay_amount;
  //--------------------
  QString aldl_as_colored_hexstring(byte *buf); // uses aldl length byte
  QString buf_as_colored_hexstring(byte *buf, int len); // uses defined byte
  void calc_avg_pktrate(qint64 pktrate_in, int size);
  void change_connection_state(int c);
  void critical_error_message(QString msg);
  bool get_main_data_packet();
  void post_thread_create_init();
  void sleep_us(int timeout);
  bool aldl_init_connection();
  void aldl_close_connection();
  void read_flash_procedure();
  bool check_if_ok_to_flash();
  void write_flash_procedure();
  void flash_stability_test();
  void flash_message(QString msg);
  void flash_error_message(QString msg);
  void flash_divider(QString msg);
  void flash_cancel_enable(bool value);
  void flash_error();
  void flash_speed(float value);
  void flash_tside_progress(int value);
  void flash_eside_progress(int value);
  void flash_close_window();
  QByteArray process_blm_cell_memory();
  bool get_blm_cell_memory();
  void cyl_drop_test();
  float get_rpm_map_diff(int quality);
  void display_event(QString str);
  bool acq_openserialport();
  void force_raw_resume_request(byte device);
  void scan_idle();
  int serial_read_any();
  byte wait_for_heartbeat(int msTimeout = BROADCAST_WAIT_TIME);
  void tickle_ecm(); // poke ecm a few times
  bool serial_discard_new(int gap, int timeout);
  bool silence_module_new(unsigned char device);
  unsigned int master_id = 0;
};

#endif

