#include <math.h>
#include <QElapsedTimer>
#include <QProgressDialog>
#include <QSettings>
#include <QGraphicsEllipseItem>
#include <QGraphicsSimpleTextItem>

#include "analyzer.h"
#include "ui_analyzer.h"
#include "definition.h"
#include "table_config.h"
#include "general.h"

void analyzer::reload_all_presets() {
  // clear old entries
  for(int x=0;x<preset_list.size();x++) {
    table_geometry *layout = preset_list.value(x);
    delete layout;
  }
  preset_list.clear();
  // clear old ui entries
  ui->preset_selector->clear();
  int n_user_presets = config.conf->value("N_USER_PRESETS",0).toInt();
  for(int x=0;x<n_user_presets;x++) {
    add_preset(get_user_preset(x));
  }
  // set default
  add_preset("EE","Lower VE Table (400-2000RPM)",
                "20,25,30,35,40,45,50,55,60,65,70,75,80,85,90,95,100",
                "400,600,800,1000,1200,1400,1600,1800,2000",false);
  add_preset("EE","Upper VE Table (2000-7000RPM)",
                "20,25,30,35,40,45,50,55,60,65,70,75,80,85,90,95,100",
                "2000,2200,2400,2600,2800,3000,3200,3400,3600,3800,4000,4500,5000,5500,6000,6500,7000",false);
  add_preset("0D/0E/E6","Main VE Table",
                "20,25,30,35,40,50,60,70,80,90,100",
                "400,800,1200,1600,2000,2400,2800,3200,3600,4000,4400,4800,5200,5600,6000,6400",false);
  add_preset("61","Main VE Table",
                "20,30,40,50,60,70,80,90,100",
                "400,800,1200,1600,2000,2400,2800,3200",true);
  current_layout = add_preset("Generic","Default",
                                     "10,15,20,25,30,35,40,45,50,55,60,65,70,75,80,85,90,95,100,105,110",
                                     "200,400,600,800,1000,1200,1400,1600,1800,2000,2200,2400,2600,2800,3000,3500,4000,4500,5000,5500,6000,6500,7000,8000,9000,10000",
                            false);
  add_preset("Generic","Default",
                "2,4,6,8,10,12,14,16,18,20,24,28,32,36,40,45,50,55,60,65,70,75,80,85,90,95,100,110,120,130,140,150,160,170,180,200,220,240,260,280,300,325,350,375,400");
  select_preset(current_layout);
  analyze();
}

void analyzer::on_table_edit_btn_clicked() {
  if(current_layout->user_preset_number == -1) return;
  table_config *editor = new table_config(current_layout);
  if(editor->exec() == QDialog::Accepted) {
    set_user_preset(current_layout->user_preset_number,current_layout);
  }
  delete editor;
  analyze();
}

void analyzer::on_maf_table_new_btn_clicked() {
  new_user_preset_ui(true);
}

void analyzer::on_ve_table_new_btn_clicked() {
  new_user_preset_ui(false);
}

void analyzer::new_user_preset_ui(bool maf) {
  table_geometry *new_layout = new table_geometry();
  new_layout->maf = maf;
  table_config *editor = new table_config(new_layout);
  if(editor->exec() == QDialog::Accepted) {
    add_user_preset(new_layout);
  } else {
    delete new_layout;
  }
}

void analyzer::on_table_delete_btn_clicked() {
   delete_user_preset(current_layout);
}

table_geometry *analyzer::add_user_preset(table_geometry *layout) {
  int n_presets = config.conf->value("N_USER_PRESETS",0).toInt();
  set_user_preset(n_presets,layout);
  layout->user_preset_number = n_presets;
  n_presets++;
  config.conf->setValue("N_USER_PRESETS",n_presets);
  add_preset(layout);
  select_preset(layout);
  return layout;
}

void analyzer::delete_user_preset(table_geometry *layout) {
  if(layout == NULL) return;
  int n_presets = config.conf->value("N_USER_PRESETS",0).toInt();
  int preset_number = layout->user_preset_number;
  if(preset_number == -1) return;
  // move all presets in the configurator down one
  for(int x=preset_number+1;x<n_presets;x++) {
    table_geometry *layout = find_user_preset(x);
    int new_preset_number = x - 1;
    layout->user_preset_number = new_preset_number;
    set_user_preset(new_preset_number,layout);
  }
  config.conf->setValue("N_USER_PRESETS",n_presets - 1);
  reload_all_presets(); // reload all
}

table_geometry *analyzer::find_user_preset(int preset_number) {
  for(int x=0;x<preset_list.size();x++) {
    table_geometry *layout = preset_list.value(x);
    if(layout->user_preset_number == preset_number) return layout;
  }
  return NULL;
}

void analyzer::select_preset(table_geometry *layout) {
  int ref = preset_list.indexOf(layout);
  int i = ui->preset_selector->findData(ref);
  ui->preset_selector->setCurrentIndex(i);
}

table_geometry *analyzer::get_user_preset(int preset_number) {
  QString preset_base_str("USER_PRESET" + QString::number(preset_number));
  bool maf_mode = config.conf->value(preset_base_str + "_MAF",false).toBool();
  table_geometry *layout;
  if(maf_mode == true) {
    if(conf_maf_analysis == false) return NULL; // bail if maf not enabled
    layout = process_csv_preset(
          config.conf->value(preset_base_str + "_NAME","User Preset").toString(),
          config.conf->value(preset_base_str + "_AFGS","10,20,30,40,50,80,100").toString());
  } else {
    layout = process_csv_preset(
          config.conf->value(preset_base_str + "_NAME","User Preset").toString(),
          config.conf->value(preset_base_str + "_MAP","10,20,30,40,50,80,100").toString(),
          config.conf->value(preset_base_str + "_RPM","1000,2000,3000,4000,5000,6000").toString(),
          config.conf->value(preset_base_str + "_FLIPPED",false).toBool());
  }
  layout->user_preset_number = preset_number;
  return layout;
}

void analyzer::set_user_preset(int preset_number, table_geometry *layout) {
  QString preset_base_str("USER_PRESET" + QString::number(preset_number));
  if(layout->maf == true) {
    config.conf->setValue(preset_base_str + "_NAME",layout->name);
    config.conf->setValue(preset_base_str + "_AFGS",layout->row.to_csv());
    config.conf->setValue(preset_base_str + "_MAF",true);
  } else {
    config.conf->setValue(preset_base_str + "_NAME",layout->name);
    config.conf->setValue(preset_base_str + "_MAP",layout->col.to_csv());
    config.conf->setValue(preset_base_str + "_RPM",layout->row.to_csv());
    config.conf->setValue(preset_base_str + "_FLIPPED",layout->flipped);
    config.conf->setValue(preset_base_str + "_MAF",false);
  }
}

void analyzer::on_preset_selector_currentIndexChanged(int index) {
  if(index == -1) return;
  current_layout = preset_list.value(index);
  if(current_layout == NULL) return;
  // set interface state appropriate for fixed or user preset
  if(current_layout->user_preset_number == -1) {
    ui->table_edit_btn->setEnabled(false);
    ui->table_delete_btn->setEnabled(false);
  } else {
    ui->table_edit_btn->setEnabled(true);
    ui->table_delete_btn->setEnabled(true);
  }
  populate_engine_select_box();
  switch_maf_ui(current_layout->maf);
  analyze();
}

table_geometry *analyzer::add_preset(QString mask, QString table, QString map, QString rpm, bool flipped) {
  return add_preset("VE $" + mask + " : " + table,map,rpm,flipped);
}

table_geometry *analyzer::add_preset(QString mask, QString table, QString afgs) {
  if(conf_maf_analysis == false) return NULL;
  return add_preset("MAF $" + mask + " : " + table,afgs);
}

table_geometry *analyzer::add_preset(table_geometry *layout) {
  if(layout == NULL) return NULL;
  if(layout->maf == true && conf_maf_analysis == false) return NULL;
  preset_list.append(layout);
  int new_item_index = preset_list.indexOf(layout);
  QString modified_name = layout->name;
  if(layout->user_preset_number != -1) {
    if(layout->maf == true) {
      modified_name.prepend("MAF *");
    } else {
      modified_name.prepend("VE *");
    }
  }
  ui->preset_selector->addItem(modified_name,new_item_index);
  return layout;
}

table_geometry *analyzer::add_preset(QString name, QString map, QString rpm, bool flipped) {
  table_geometry *layout = process_csv_preset(name,map,rpm,flipped);
  if(layout->verify() == false) return NULL;
  add_preset(layout);
  return layout;
}

table_geometry *analyzer::add_preset(QString name, QString afgs) {
  if(conf_maf_analysis == false) return NULL;
  table_geometry *layout = process_csv_preset(name,afgs);
  if(layout->verify() == false) return NULL;
  add_preset(layout);
  return layout;
}

table_geometry *analyzer::process_csv_preset(QString name, QString map, QString rpm, bool flipped) {
  QStringList map_list = split_csv_line(map,false,false);
  table_geometry *layout = new table_geometry();
  for(int x=0;x<map_list.size();x++) {
    QString str_in = map_list.at(x);
    layout->col.list.append(str_in.toInt());
  }
  QStringList rpm_list = split_csv_line(rpm,false,false);
  for(int x=0;x<rpm_list.size();x++) {
    QString str_in = rpm_list.at(x);
    layout->row.list.append(str_in.toInt());
  }
  layout->user_preset_number = -1; // locked preset
  layout->flipped = flipped;
  layout->name = name;
  return layout;
}

table_geometry *analyzer::process_csv_preset(QString name, QString afgs) {
  if(conf_maf_analysis == false) return NULL;
  QStringList afgs_list = split_csv_line(afgs,false,false);
  table_geometry *layout = new table_geometry();
  for(int x=0;x<afgs_list.size();x++) {
    QString str_in = afgs_list.at(x);
    layout->row.list.append(str_in.toInt());
  }
  layout->user_preset_number = -1; // locked preset
  layout->maf = true;
  layout->flipped = false;
  layout->name = name;
  return layout;
}


