#include <math.h>
#include <QElapsedTimer>
#include <QProgressDialog>
#include <QSettings>
#include <QGraphicsEllipseItem>
#include <QGraphicsSimpleTextItem>
#include <QDebug>

#include "analyzer.h"
#include "ui_analyzer.h"
#include "definition.h"
#include "general.h"

void analyzer::process_row(QStringList *row) {
  total_lines++;
  if(process_filter_set(row) == true) {
    if(conf_blm_analysis == true) {
      get_trim_cell(row);
    }
  }
  if(conf_knock_analysis == true) {
    get_knock_cell(row);
  }
}

void analyzer::get_trim_cell(QStringList *row) {
  double trim = 0.00;
  // BLM VALUE
  trim = log_trim_to_multiplier(get_value(row,definition::STDMAP_LBLM)); // get bank a
  if(conf_dual_bank == true && conf_arbitrary_input == false) {
    if(leanest_bank == true) {
      trim = highest_of(trim,log_trim_to_multiplier(get_value(row,definition::STDMAP_RBLM)));
    } else {
      trim += log_trim_to_multiplier(get_value(row,definition::STDMAP_RBLM)); // avg w/ bank b
      trim = trim / 2.00;
    }
  }
  // INTEGRATOR VALUE
  if(conf_integrator == true && conf_arbitrary_input == false) {
    double integrator = log_trim_to_multiplier(get_value(row,definition::STDMAP_LINT));
    if(conf_dual_bank == true) {
      if(leanest_bank == true) {
        integrator = highest_of(integrator,log_trim_to_multiplier(get_value(row,definition::STDMAP_RINT)));
      } else {
        integrator += log_trim_to_multiplier(get_value(row,definition::STDMAP_RINT));
        integrator = integrator / 2.00;
      }
    }
    integrator = clamp(integrator,config.integrator_min,config.integrator_max);
    if(integrator < 0.00) {
      trim = trim - ( ( 1.00 - integrator ) * config.integrator_reduction );
    } else if(integrator > 0.00) {
      trim = trim + ( ( integrator - 1.00 ) * config.integrator_reduction );
    }
  }
  // get afgs only if necessary
  double afgs = 0.00; // .. otherwise afgs will be zero
  if(conf_maf_analysis == true) afgs = get_value(row,definition::STDMAP_MAF);
  // add to trim list
  if(conf_arbitrary_input == false) trim = trim * 100.00;
  add_trim_vector(trim,
                  get_value(row,definition::STDMAP_RPM),
                  get_value(row,definition::STDMAP_MAP),
                  afgs);
}

inline double analyzer::log_trim_to_multiplier(double trim_in) {
  switch(def->trim_type) {
  case definition::TRIM_FORMAT_PERCENT_SIGNED:
    return ( trim_in + 100.00 ) / 100.00;
    break;
  case definition::TRIM_FORMAT_PERCENT_WHOLE:
    return trim_in / 100.00;
    break;
  case definition::TRIM_FORMAT_RAW:
    return trim_in / 128.00;
    break;
  case definition::TRIM_FORMAT_ARBITRARY:
    return trim_in;
    break;
  default:
    fatal_error("Unknown input trim format in log_trim_to_multiplier()");
  }
  return 0.00; // never reached
}

void analyzer::get_knock_cell(QStringList *row) {
  quint32 current_knock_count = get_value(row,definition::STDMAP_KNOCKCOUNT);
  if(current_knock_count == last_knock_count) { // no counter increase
    return; // bail quickly for minimum overhead
  } else if(current_knock_count < last_knock_count) { // it wrapped around or something, ignore event
    last_knock_count = current_knock_count; // reset counter to current
    return;
  } else { // a real knock event
    quint32 counts = current_knock_count - last_knock_count;
    // we need to check for a timestamp here .....
    // this magnitude is time variable.
    // this is shitty.
    add_knock_vector(counts,
                     get_value(row,definition::STDMAP_RPM),
                     get_value(row,definition::STDMAP_MAP));
    last_knock_count = current_knock_count;
  }
}

void analyzer::add_trim_vector(float trim, float rpm, float map, float afgs) {
  trim_point x;
  x.trim = trim;
  x.rpm = rpm;
  x.map = map;
  x.afgs = afgs;
  trim_list.append(x);
}

void analyzer::add_knock_vector(quint32 count, float rpm, float map) {
  knock_point x;
  x.rpm = rpm;
  x.counts = count;
  x.map = map;
  knock_list.append(x);
}

void analyzer::restart_knock_list() {
  knock_list.clear();
  knock_list.reserve(10000);
}

bool analyzer::parameter_is_defined(int stdvalue) {
  if(get_mapping(stdvalue) == -1) return false;
  return true;
}

inline double analyzer::get_value(QStringList *row, int stdvalue) {
  QString x(row->at(get_mapping(stdvalue)));
  x.remove('%');
  x.remove('+');
  return x.toDouble();
}

inline int analyzer::get_mapping(int stdvalue) {
  field_mapping *m = mapping_list.value(stdvalue);
  return m->mapped_column;
}

bool analyzer::process_filter_set(QStringList *data) {
  for(int x=0;x<filter_list.size();x++) {
    if(filter_list.at(x)->filter_match(data) == false) return false;
  }
  return true;
}

inline double analyzer::linear_distance(double xa, double xb, double ya, double yb) {
  double x = pow(xb - xa,2) + pow(yb - ya,2);
  return sqrt(x);
}

inline double analyzer::linear_distance(trim_point a, trim_point b) {
  return linear_distance(a.rpm / config.distance_rpm, b.rpm / config.distance_rpm,
                         a.map / config.distance_map, b.map / config.distance_map);
}

void analyzer::process_ve_strict(int row, int col) {
  /* this is a traditional restricted cell routine.. */
  int max_rpm = current_layout->row.high_border(row);
  int max_map = current_layout->col.high_border(col);
  int min_rpm = current_layout->row.low_border(row);
  int min_map = current_layout->col.low_border(col);
  double avg = 0;
  int n_recs = 0;
  for(int x=0;x<trim_list.size();x++) {
    if(trim_list[x].map < min_map || trim_list[x].map > max_map) continue;
    if(trim_list[x].rpm < min_rpm || trim_list[x].rpm > max_rpm) continue;
    avg += trim_list[x].trim;
    n_recs++;
  }
  display_cell(row,col,avg/(double)n_recs,n_recs);
}

void analyzer::process_ve_loose(int row, int col) {
  /* this is a cell-based ve averaging routine, but with hysteresis, so that data
   * points near a cell border affect both cells equally. */
  double max_rpm = current_layout->row.high_border(row) + config.cell_hystersis_rpm;
  double max_map = current_layout->col.high_border(col) + config.cell_hystersis_map;
  double min_rpm = current_layout->row.low_border(row) - config.cell_hystersis_rpm;
  double min_map = current_layout->col.low_border(col) - config.cell_hystersis_map;
  double avg = 0;
  int n_recs = 0;
  for(int x=0;x<trim_list.size();x++) {
    if(trim_list[x].map < min_map || trim_list[x].map > max_map) continue;
    if(trim_list[x].rpm < min_rpm || trim_list[x].rpm > max_rpm) continue;
    avg += trim_list[x].trim;
    n_recs++;
  }
  display_cell(row,col,avg/(double)n_recs,n_recs);
}

void analyzer::process_maf_strict(int row) {
  /* this is a traditional restricted cell routine.. */
  int max_maf = current_layout->row.high_border(row);
  int min_maf = current_layout->row.low_border(row);
  double avg = 0;
  int n_recs = 0;
  for(int x=0;x<trim_list.size();x++) {
    if(trim_list[x].afgs < min_maf || trim_list[x].afgs > max_maf) continue;
    avg += trim_list[x].trim;
    n_recs++;
  }
  display_cell(row,avg/(double)n_recs,n_recs);
}

void analyzer::process_maf_loose(int row) {
  double max_maf = current_layout->row.high_border(row) + config.cell_hystersis_maf;
  double min_maf = current_layout->row.low_border(row) - config.cell_hystersis_maf;
  double avg = 0;
  int n_recs = 0;
  for(int x=0;x<trim_list.size();x++) {
    if(trim_list[x].afgs < min_maf || trim_list[x].afgs > max_maf) continue;
    avg += trim_list[x].trim;
    n_recs++;
  }
  display_cell(row,avg/(double)n_recs,n_recs);
}

void analyzer::process_ve_radial(int row, int col) {
  /* This analysis method attempts to determine the geometric relationship between
   * the map/rpm of the center of this cell and the map/rpm of all other data points...
   * The map to rpm geometry is transformed based on two divisors that attempt to square
   * the chart, allowing map or rpm to have various magntitudes.
   * It then determines a 'strength' ratio based directly on that distance.
   * Gain, clipping, and highpass filtering is applied to that strength value to tighten
   * localzation of strong data, and to ensure that far-away points don't have influence.
   * The modified strength is used as 'weight' when calculating the average for this cell.
   * This approach removes restrictive cell borders, and allows one data point to feed
   * multiple cells effectively, creating stronger data and inherently smoothing output. */
  // radial mode only supports processed values, so in 'arbitrary' value mode, use ve_loose intead.
  if(conf_arbitrary_input == true) {
    process_ve_loose(row,col);
    return;
  }
  trim_point this_point;
  this_point.rpm = current_layout->row.center(row);
  this_point.map = current_layout->col.center(col);
  double avg = 0;
  double divisor = 0;
  for(int x=0;x<trim_list.size();x++) {
    double distance = linear_distance(this_point,trim_list.at(x));
    double strength = ( 1 - distance / config.interpolator_slope ) + config.interpolator_gain;
    if(strength < config.interpolator_noisefloor) continue; // drop noise floor
    if(strength > 1.00) strength = 1.00; // clip strength @ 100%
    avg += ( strength * trim_list[x].trim );
    divisor += strength;
  }
  display_cell(row,col,avg/divisor,divisor);
}

void analyzer::add_knock_point(int rpm, int map, int count, int width, int height) {
  float rpm_range = 8000;
  float map_range = 120;
  float circle_size = clamp(count / config.knock_count_divisor,
                            config.min_knock_circle_radius * 2,
                            config.max_knock_circle_radius * 2);
  float scale_x = width * ((float)map / map_range);
  float scale_y = height * ((float)rpm / rpm_range);
  qreal x = clamp(scale_x - (circle_size / 2),1,width - 1);
  qreal y = clamp(scale_y - (circle_size / 2),1,height - 1);
  qreal w = clamp(circle_size,1,width - circle_size - 1);
  qreal h = clamp(circle_size,1,height - circle_size - 1);
  QGraphicsEllipseItem *point = knockmap.addEllipse(x,y,w,h);
  point->setBrush(QColor(200,0,0,180));
  point->setToolTip("Knock count = " + QString::number(count) + "\nRPM = " +
                    QString::number(rpm) + "\nMAP = " + QString::number(map));
  QCoreApplication::processEvents();
}

void analyzer::draw_knock_grid(int width, int height) {
  float rpm_range = 8000;
  float map_range = 120;
  int n_map_cells = 12;
  int n_rpm_cells = 16;
  if(config.dark_ui == true) knockmap.setBackgroundBrush(Qt::lightGray);
  for(int x=1;x<n_map_cells;x++) { // draw vertical lines
    qreal xpos = ((float)width/n_map_cells)*(float)x;
    qreal h = height - 1;
    knockmap.addLine(xpos,1,xpos,h);
    int current_map = ( map_range / n_map_cells ) * x;
    QGraphicsSimpleTextItem *label = knockmap.addSimpleText(QString::number(current_map));
    label->setPos(xpos + 2,1);
  }
  for(int x=1;x<n_rpm_cells;x++) { // hz lines
    qreal ypos = ((float)height/n_rpm_cells)*(float)x;
    qreal w = width - 1;
    knockmap.addLine(1,ypos,w,ypos);
    int current_rpm = ( rpm_range / n_rpm_cells ) * x;
    QGraphicsSimpleTextItem *label = knockmap.addSimpleText(QString::number(current_rpm));
    label->setPos(2,ypos);
  }
}
