#include <math.h>
#include <QElapsedTimer>
#include <QProgressDialog>
#include <QSettings>
#include <QGraphicsEllipseItem>
#include <QGraphicsSimpleTextItem>
#include <QMessageBox>
#include <QClipboard>

#include "config.h"
#include "analyzer.h"
#include "ui_analyzer.h"
#include "definition.h"
#include "table_config.h"
#include "general.h"

analyzer::analyzer(QWidget *parent) :
  QWidget(parent),
  ui(new Ui::analyzer) {
  ui->setupUi(this);
  ui->knockscope->setScene(&knockmap);
  table_redraw_lock = false;
  leanest_bank = false;
  ignore_int = false;
  conf_trim_skew = ui->trim_skew_box->value();
  conf_trim_max = ui->trim_max_box->value();
  conf_trim_min = ui->trim_min_box->value();
}

void analyzer::set_busy_cursor(bool busy) {
  if(busy) {
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    QApplication::processEvents();
  } else {
    QApplication::restoreOverrideCursor();
  }
}

analyzer::~analyzer() {
  delete ui;
}

void analyzer::prepare_interface() {
  //--------------
  ui->table_to_clipboard_btn->setEnabled(!conf_arbitrary_input);
  //--------------
  ui->display_selector->clear();
  ui->maf_table_new_btn->setHidden(!conf_maf_analysis); // hide maf btn
  if(conf_arbitrary_input == false) {
    ui->display_selector->addItem(QStringLiteral("Whole Percentage (100=Good)"),definition::TRIM_FORMAT_PERCENT_WHOLE);
    ui->display_selector->addItem(QStringLiteral("+/- Percentage (0=Good)"),definition::TRIM_FORMAT_PERCENT_SIGNED);
    ui->display_selector->addItem(QStringLiteral("0-255 (128=Good)"),definition::TRIM_FORMAT_RAW);
    ui->display_selector->setCurrentIndex(ui->display_selector->findData(definition::TRIM_FORMAT_PERCENT_SIGNED));
    ui->target_setup_box->setHidden(false);
  } else { // arbitrary input true
    ui->display_selector->addItem(QStringLiteral("Arbitrary Value (AFR, Etc)"),definition::TRIM_FORMAT_ARBITRARY);
    ui->display_selector->setCurrentIndex(ui->display_selector->findData(definition::TRIM_FORMAT_ARBITRARY));
    ui->target_setup_box->setHidden(true);
  }
}

void analyzer::populate_engine_select_box() {
  if(current_layout == nullptr) { // no layout present
    ui->engine_select->clear();
    return;
  }
  //---------------
  // we keep track of the previous layout's maf flag so we know whether
  // to redraw the list (and lose previous selection ... )
  static bool first_draw = false;
  static bool maf_already_enabled = true;
  if(maf_already_enabled == current_layout->maf && first_draw == true) return;
  maf_already_enabled = current_layout->maf;
  first_draw = true;
  //---------------
  table_redraw_lock = true;
  ui->engine_select->clear();
  // only enable geometric interpolation for VE and never for arbitrary input
  if(conf_arbitrary_input == false && current_layout->maf == false) {
    ui->engine_select->addItem(ENGINE_TEXT_RADIAL,ENGINE_RADIAL);
    ui->engine_select->addItem(ENGINE_TEXT_LOOSE,ENGINE_LOOSE);
    ui->engine_select->addItem(ENGINE_TEXT_STRICT,ENGINE_STRICT);
  } else {
    ui->engine_select->addItem(ENGINE_TEXT_LOOSE,ENGINE_LOOSE);
    ui->engine_select->addItem(ENGINE_TEXT_STRICT,ENGINE_STRICT);
  }
  ui->engine_select->setCurrentIndex(0);
  table_redraw_lock = false;
}

int analyzer::selected_engine() {
  if(ui->engine_select->currentText() == ENGINE_TEXT_RADIAL) return ENGINE_RADIAL;
  if(ui->engine_select->currentText() == ENGINE_TEXT_LOOSE) return ENGINE_LOOSE;
  if(ui->engine_select->currentText() == ENGINE_TEXT_STRICT) return ENGINE_STRICT;
  return ENGINE_UNDEFINED;
}

void analyzer::load_new_analysis() {
  table_redraw_lock = true;
  set_busy_cursor(true);
  total_lines = 0;
  init_progress();
  config.get();
  if(configure_analyzer() == false) {
    set_busy_cursor(false);
    table_redraw_lock = false;
    return;
  }
  restart_filter_counters();
  prepare_interface();
  reload_all_presets();
  restart_trim_list();
  restart_knock_list();
  process_all_files();
  progress_line();
  progress("Parsed " + QString::number(total_lines) + " lines.");
  progress("Created " + QString::number(trim_list.size()) + " trim vectors.");
  progress("Created " + QString::number(knock_list.size()) + " knock points.");
  progress_line();
  progress("Data is ready for processing.");
  get_filter_results();
  table_redraw_lock = false;
  set_busy_cursor(false);
  analyze();
  ui->main_switcher->setCurrentIndex(1);
}

QString analyzer::get_filter_results() {
  QString out;
  quint64 total_rejected = 0;
  for(int x=0;x<filter_list.size();x++) {
    data_filter *f = filter_list.at(x);
    if(f->n_filtered > 0) {
      out.append(QString::number(f->n_filtered) + " Blocked @ ");
      out.append(f->filter_description());
      total_rejected += f->n_filtered;
      out.append("\n");
    }
  }
  out.append("Discarded " + QString::number(total_rejected) + "/" +
         QString::number(total_lines) + " records due to filtering.");
  ui->filter_display->clear();
  ui->filter_display->setPlainText(out);
  return out;
}

void analyzer::restart_filter_counters() {
  for(int x=0;x<filter_list.size();x++) {
    data_filter *f = filter_list.at(x);
    f->reset_counters();
  }
}

void analyzer::process_all_files() {
  QList<QString> key_list = log_list.keys();
  for(int x=0;x<key_list.size();x++) { // iterate over all logs
    datalog *y = log_list.value(key_list.at(x));
    process_file(y);
  }
}

void analyzer::process_file(datalog *log) {
  QElapsedTimer analysis_time;
  analysis_time.start();
  progress_line();
  last_knock_count = 65535; // let 'er wrap .... (don't ask)
  //-------------------
  progress("Parsing file " + log->file_path + "...");
  log->seek_to_data();
  forever {
    QStringList row_in = log->get_next_line();
    if(row_in.isEmpty() == true) break;
    process_row(&row_in);
  }
  //-------------------
  progress("Completed processing log in " + QString::number(analysis_time.elapsed()) + " msec.");
  if(log->rejected_lines > 0) {
    progress("Rejected " + QString::number(log->rejected_lines) + " malformed lines.");
  }
}

bool analyzer::configure_analyzer() {
  // this is an ugly if/then/else dependancy tree to enable various analyzer modules
  // depending on whether certain parameters are defined.
  // perhaps some kind of smarter solution will be necessary when there are more than
  // two major analyzers, but for now this hellish kludge will do.
  bool err = false;
  if(! parameter_is_defined(definition::STDMAP_RPM) ||
     ! parameter_is_defined(definition::STDMAP_MAP)) {
    progress_error("The analyzer can't function without basic RPM and MAP values.");
    err = true;
  }
  if(def->trim_type == definition::TRIM_FORMAT_ARBITRARY) {
    conf_arbitrary_input = true;
  } else {
    conf_arbitrary_input = false;
  }
  if(parameter_is_defined(definition::STDMAP_LBLM) == true) {
    conf_blm_analysis = true;
    if(parameter_is_defined(definition::STDMAP_RBLM) == true) {
      conf_dual_bank = true;
      if(ignore_int == false && parameter_is_defined(definition::STDMAP_LINT) &&
         parameter_is_defined(definition::STDMAP_RINT)) {
        conf_integrator = true;
      } else {
        conf_integrator = false;
      }
    } else {
      conf_dual_bank = false;
      if(ignore_int == false && parameter_is_defined(definition::STDMAP_LINT)) {
        conf_integrator = true;
      } else {
        conf_integrator = false;
      }
    }
    if(parameter_is_defined(definition::STDMAP_MAF) == true) {
      conf_maf_analysis = true;
    } else {
      conf_maf_analysis = false;
    }
  } else {
    conf_integrator = false;
    conf_maf_analysis = false;
    conf_blm_analysis = false;
    conf_dual_bank = false;
  }
  if(parameter_is_defined(definition::STDMAP_KNOCKCOUNT) == true) {
    conf_knock_analysis = true;
  } else {
    conf_knock_analysis = false;
  }
  if(conf_blm_analysis == false &&
     conf_knock_analysis == false) {
    progress_error(QStringLiteral("No analyzers could be enabled."));
    err = true;
  }
  progress_line();
  // overrides for arbitrary trim mode
  if(conf_arbitrary_input == true) {
    progress_error(QStringLiteral("Some features are being disabled to use arbitrary trim mode."));
    conf_integrator = false;
  }
  if(err == true) {
    progress_error(QStringLiteral("Major parameter mapping is missing, cannot analyze!"));
    warning("At least one critical parameter that is necessary for analysis is not mapped.\nAnalyzer disabled.");
    ui->main_switcher->setEnabled(false);
    return false;
  } else {
    ui->main_switcher->setEnabled(true);
    print_configuration(QStringLiteral("BLM Analysis"),conf_blm_analysis);
    print_configuration(QStringLiteral("INT Analysis"),conf_integrator);
    print_configuration(QStringLiteral("Dual Bank Mode"),conf_dual_bank);
    print_configuration(QStringLiteral("Knock Analysis"),conf_knock_analysis);
    print_configuration(QStringLiteral("MAF Mode"),conf_maf_analysis);
    ui->table_page->setEnabled(conf_blm_analysis);
    return true;
  }
}

void analyzer::print_configuration(QString module, bool enabled) {
  if(enabled == true) {
    progress("Option '" + module + "' is [ ENABLED ]");
  } else {
    progress("Option '" + module + "' is [ DISABLED ]");
  }
}

void analyzer::progress(QString str) {
  ui->status_display->appendPlainText(str);
  QCoreApplication::processEvents();
}

void analyzer::progress_error(QString str) {
  progress("ERROR: " + str);
}

void analyzer::progress_line() {
  progress(QStringLiteral("-----------------------------"));
}

void analyzer::init_progress() {
  ui->main_switcher->setCurrentIndex(0);
  ui->status_display->clear();
  QCoreApplication::processEvents();
}

void analyzer::restart_trim_list() {
  trim_list.clear();
  trim_list.reserve(TRIM_LIST_RESERVE_SIZE);
}

int analyzer::get_analyzer_fuel_mode_selection() {
  if(current_layout->maf == false) {
    return FUEL_MODE_VE;
  } else {
    return FUEL_MODE_MAF;
  }
}

void analyzer::switch_maf_ui(bool maf) {
  // enable/disable the clipboard button, but check for arbitrary input case
  if(maf == false) {
    if(conf_arbitrary_input == false) ui->table_to_clipboard_btn->setEnabled(true);
  } else {
    ui->table_to_clipboard_btn->setEnabled(false);
  }
}

void analyzer::analyze() {
  if(table_redraw_lock == true) return;
  if(current_layout == NULL) return;
  switch(get_analyzer_fuel_mode_selection()) {
  case FUEL_MODE_VE:
    analyze_ve();
    break;
  case FUEL_MODE_MAF:
    analyze_maf();
    break;
  default:
    fatal_error("Undefined fuel mode in analyze()");
  }
}

void analyzer::analyze_maf() {
  if(table_redraw_lock == true) return;
  if(current_layout == NULL) return;
  set_busy_cursor(true);
  // get configuration from ui
  int engine = selected_engine();
  //--------------
  reset_table_layout(); // prepare table
  //--------------
  for(int row=0;row < current_layout->row.list.size();row++) {
    switch(engine) {
    case ENGINE_STRICT:
      process_maf_strict(row);
      break;
    case ENGINE_LOOSE:
      process_maf_loose(row);
      break;
    default:
      fatal_error("Inappropriate engine in analyze_maf()");
    }
  }
  ui->analysis_table->resizeColumnsToContents();
  set_busy_cursor(false);
}

void analyzer::analyze_ve() {
  if(table_redraw_lock == true) return;
  if(current_layout == NULL) return;
  set_busy_cursor(true);
  // get configuration from ui
  int engine = selected_engine();
  //--------------
  reset_table_layout(); // prepare table
  //--------------
  for(int col=0;col < current_layout->col.list.size();col++) {
    for(int row=0;row < current_layout->row.list.size();row++) {
      switch(engine) {
      case ENGINE_RADIAL:
        process_ve_radial(row, col);
        break;
      case ENGINE_STRICT:
        process_ve_strict(row, col);
        break;
      case ENGINE_LOOSE:
        process_ve_loose(row, col);
        break;
      default:
        fatal_error("Inappropriate engine in analyze_ve()");
      }
    }
  }
  ui->analysis_table->resizeColumnsToContents();
  set_busy_cursor(false);
}

void analyzer::reset_table_layout() {
  ui->analysis_table->clear();
  switch(get_analyzer_fuel_mode_selection()) {
  case FUEL_MODE_VE:
    reset_ve_table_layout();
    break;
  case FUEL_MODE_MAF:
    reset_maf_table_layout();
    break;
  default:
    fatal_error("Undefined fuel mode in reset_table_layout()");
  }
}

void analyzer::reset_maf_table_layout() {
  if(current_layout == NULL) return;
  ui->analysis_table->setColumnCount(1);
  ui->analysis_table->setRowCount(current_layout->row.list.count());
  ui->analysis_table->setVerticalHeaderLabels(current_layout->row.get_headers());
  QStringList header_label;
  header_label.append(QString("Trim"));
  ui->analysis_table->setHorizontalHeaderLabels(header_label);
}

void analyzer::reset_ve_table_layout() {
  if(current_layout == NULL) return;
  //---------------------
  if(current_layout->flipped == false) {
    ui->analysis_table->setColumnCount(current_layout->col.list.count());
    ui->analysis_table->setRowCount(current_layout->row.list.count());
    ui->analysis_table->setVerticalHeaderLabels(current_layout->row.get_headers());
    ui->analysis_table->setHorizontalHeaderLabels(current_layout->col.get_headers());
  } else {
    ui->analysis_table->setColumnCount(current_layout->row.list.count());
    ui->analysis_table->setRowCount(current_layout->col.list.count());
    ui->analysis_table->setVerticalHeaderLabels(current_layout->col.get_headers());
    ui->analysis_table->setHorizontalHeaderLabels(current_layout->row.get_headers());
  }
  //---------------------
  ui->analysis_table->clearContents();
}

void analyzer::refresh_table_from_existing_data() {
  // this is a way faster way to simply change display modes,
  // assuming the data stored in the table is still valid.
  // initial draw is still done by display_ve_cell
  // not great design but works just fine
  if(table_redraw_lock == true) return;
  for(int x=0;x<ui->analysis_table->columnCount();x++) {
    for(int y=0;y<ui->analysis_table->rowCount();y++) {
      QTableWidgetItem *i = ui->analysis_table->item(y,x);
      if(i == NULL) continue;
      if(i->text() == QStringLiteral("---")) continue;
      double trim = i->data(Qt::UserRole).toDouble();
      i->setText(get_trim_display_string(trim));
      i->setBackgroundColor(get_trim_shade(trim));
    }
  }
  ui->analysis_table->resizeColumnsToContents();
}

void analyzer::display_cell(int row,int col,QString str,QString tooltip) {
#ifdef EXTRA_CHECKS
  if(row < 0 || row > current_layout->row.n_cells()) {
    warning("Attempt to display cell outside of row boundary");
    return;
  }
  if(col < 0 || col > current_layout->col.n_cells()) {
    warning("Attempt to display cell outside of column boundary");
    return;
  }
#endif
  QTableWidgetItem *item = new QTableWidgetItem(str);
  item->setToolTip(tooltip);
  item->setData(Qt::UserRole,(double)100.00); // string cell is 100%
  if(config.dark_ui == true) item->setTextColor(QColor(0,0,0));
  if(current_layout->flipped == false) {
    ui->analysis_table->setItem(row,col,item);
  } else {
    ui->analysis_table->setItem(col,row,item);
  }
}

void analyzer::display_cell(int row,int col,double value,double n_recs) {
#ifdef EXTRA_CHECKS
  if(row < 0 || row > current_layout->row.n_cells()) {
    warning("Attempt to display cell outside of row boundary");
    return;
  }
  if(col < 0 || col > current_layout->col.n_cells()) {
    warning("Attempt to display cell outside of column boundary");
    return;
  }
#endif
  if(n_recs < config.minimum_records) {
    display_cell(row,col,QStringLiteral("---"),
                 "Not enough records (" + QString::number(n_recs) + ") for accurate result");
    return;
  }
  QTableWidgetItem *item = new QTableWidgetItem();
  item->setText(get_trim_display_string(value));
  item->setToolTip("Based on " + QString::number(n_recs,'f',2) + " records...");
  item->setData(Qt::UserRole,(double)value);
  if(conf_arbitrary_input == false) item->setBackgroundColor(get_trim_shade(value));
  if(config.dark_ui == true) item->setTextColor(QColor(0,0,0));
  if(current_layout->flipped == false) {
    ui->analysis_table->setItem(row,col,item);
  } else {
    ui->analysis_table->setItem(col,row,item);
  }
}

void analyzer::display_cell(int row,double value,double n_recs) {
#ifdef EXTRA_CHECKS
  if(row < 0 || row > current_layout->row.n_cells() - 1) {
    warning("Attempt to display 2d cell outside of row boundary");
    return;
  }
#endif
  if(n_recs < config.minimum_records) {
    display_cell(row,0,QStringLiteral("---")
                 ,"Not enough records (" + QString::number(n_recs) + ") for accurate result");
    return;
  }
  QTableWidgetItem *item = new QTableWidgetItem();
  item->setText(get_trim_display_string(value));
  item->setToolTip("Based on " + QString::number(n_recs,'f',2) + " records...");
  item->setData(Qt::UserRole,(double)value);
  if(conf_arbitrary_input == false) item->setBackgroundColor(get_trim_shade(value));
  if(config.dark_ui == true) item->setTextColor(QColor(0,0,0));
  ui->analysis_table->setItem(row,0,item);
}

QString analyzer::get_trim_display_string(double value) {
  // format will probably be in whole percentage
  double trim = skew_trim(value);
  switch(conf_output_format) {
  case definition::TRIM_FORMAT_PERCENT_WHOLE:
    return QString(QString::number(trim,'f',0) + "%");
    break;
  case definition::TRIM_FORMAT_PERCENT_SIGNED:
    if(trim < 100) {
      return QString("-" + QString::number(trim - 100,'f',1) + "%");
    } else {
      return QString("+" + QString::number(trim - 100,'f',1) + "%");
    }
    break;
  case definition::TRIM_FORMAT_RAW:
    return QString::number(trim * 1.28,'f',1);
    break;
  case definition::TRIM_FORMAT_ARBITRARY:
    return QString::number(trim,'f',1);
    break;
  default:
    fatal_error("Inappropriate trim output format in get_trim_display_string()");
  }
  return QString::null; // should never be reached
}

QColor analyzer::get_trim_shade(double trim_in) {
  double trim = skew_trim(trim_in);
  QColor x;
  if(trim <= config.ideal_low_trim) {
    x.setRgb(160,50,50);
  } else if(trim >= config.ideal_high_trim) {
    x.setRgb(60,160,75);
  } else {
    x.setRgb(180,180,220);
  }
  return x;
}

void analyzer::on_knock_exec_btn_clicked() {
  set_busy_cursor(true);
  knockmap.clear();
  config.get();
  int graph_width = ui->knockscope->width();
  int graph_height = ui->knockscope->height();
  draw_knock_grid(graph_width,graph_height);
  for(int x=0;x<knock_list.size();x++) {
    add_knock_point(knock_list[x].rpm,knock_list[x].map,
                    knock_list[x].counts,
                    graph_width,graph_height);
    if(x > MAX_KNOCK_EVENTS) {
      warning("There are too many knock events, only the first "
                     + QString::number(MAX_KNOCK_EVENTS) + " are shown.");
      break;
    }
  }
  set_busy_cursor(false);
}

void analyzer::on_table_to_clipboard_btn_clicked() {
#ifdef WARN_BEFORE_CLIPBOARD
  QMessageBox premsgbox;
  premsgbox.setText("You are about to apply changes to the table information in the clipboard.");
  premsgbox.setInformativeText("If you have not done so already, copy the appropriate data from your tuning utility into your clipboard and press APPLY.\nThe layout of the table must match.");
  premsgbox.setStandardButtons(QMessageBox::Apply | QMessageBox::Abort);
  premsgbox.setDefaultButton(QMessageBox::Apply);
  if(premsgbox.exec() == QMessageBox::Abort) return;
#endif
  //------------
  // get text from clipboard
  QClipboard *clipboard = QApplication::clipboard();
  QString text_in = clipboard->text();
  //------------
  // split lines into rows
  QStringList rows_in = text_in.split(QRegExp("[\r\n]"),QString::SkipEmptyParts);
  if(rows_in.size() != ui->analysis_table->rowCount()) { // row size mismatch
    QMessageBox failmsg;
    failmsg.setText("The number of rows in the clipboard data doesn't match the current layout."
                    "\nNo modifications were made.");
    failmsg.setStandardButtons(QMessageBox::Abort);
    failmsg.exec();
    return;
  }
  //------------
  // operate on each row
  QString text_out;
  for(int row=0;row<rows_in.size();row++) {
    QString string_in = rows_in.at(row);
    QStringList cols_in = string_in.split(QRegExp("\\s+"),QString::SkipEmptyParts);
    if(cols_in.size() != ui->analysis_table->columnCount()) { // col size mismatch
      QMessageBox failmsg;
      failmsg.setText("The number of columns in "
                      "the clipboard data doesn't match the current layout."
                      "\nNo modifications were made.");
      failmsg.setStandardButtons(QMessageBox::Abort);
      failmsg.exec();
      return;
    }
    //------------
    // operate on each column
    for(int col=0;col<cols_in.size();col++) {
      double cell_modifier = 100; // current table cell as double percentage
      //------------
      // get table cell
      QTableWidgetItem *cell = ui->analysis_table->item(row,col);
      if(cell == NULL) {
        cell_modifier = 100; // no multiplier if no cell contents
      } else {
        cell_modifier = skew_trim(cell->data(Qt::UserRole).toDouble());
      }
      //------------
      // get clipboard cell
      QString value_in_text = cols_in.at(col);
      bool conversion_success = false;
      double value_in = value_in_text.toDouble(&conversion_success);
      if(conversion_success == false) { // bad data in clipboard
        QMessageBox failmsg;
        failmsg.setText("There is bad data in the clipboard."
                        "\nNo modifications were made.");
        failmsg.setStandardButtons(QMessageBox::Abort);
        failmsg.exec();
        return;
      }
      //------------
      // modify and output
      double modified_value = value_in * ( cell_modifier / 100 );
      text_out.append(QString::number(modified_value,'f',3));
      if(col < cols_in.size() - 1) text_out.append("\t");
    } // end col
    text_out.append("\n");
  } // end row
  clipboard->setText(text_out);
  QMessageBox postmsgbox;
  postmsgbox.setText("The table in the clipboard has been modified.  Go to your tuning software and paste the changes.");
  postmsgbox.setStandardButtons(QMessageBox::Ok);
  postmsgbox.setDefaultButton(QMessageBox::Ok);
  postmsgbox.exec();
}

void analyzer::on_engine_select_currentIndexChanged(int) {
  analyze();
}

analyzer_parameters::analyzer_parameters() {
  conf = new QSettings(CONFIG_SETTINGS_DOMAIN,CONFIG_SETTINGS_GROUP);
}

void analyzer_parameters::get() {
  distance_rpm = conf->value("TUNE_DISTANCE_RPM").toDouble();
  distance_map = conf->value("TUNE_DISTANCE_MAP").toDouble();
  minimum_records = conf->value("TUNE_MINIMUM_RECORDS").toDouble();
  ideal_low_trim = conf->value("TUNE_VISUAL_RICH_TRIM").toDouble();
  ideal_high_trim = conf->value("TUNE_VISUAL_LEAN_TRIM").toDouble();
  integrator_reduction = conf->value("TUNE_INTEGRATOR_REDUCTION").toDouble();
  integrator_max = conf->value("TUNE_INTEGRATOR_MAX").toDouble() / 100.00;
  integrator_min = conf->value("TUNE_INTEGRATOR_MIN").toDouble() / 100.00;
  interpolator_slope = conf->value("TUNE_INTERPOLATOR_SLOPE").toDouble();
  interpolator_gain = conf->value("TUNE_INTERPOLATOR_GAIN").toDouble();
  interpolator_noisefloor = conf->value("TUNE_INTERPOLATOR_NOISEFLOOR").toDouble();
  knock_count_divisor = conf->value("TUNE_KNOCK_COUNT_DIVISOR").toDouble();
  min_knock_circle_radius = conf->value("TUNE_MIN_KNOCK_CIRCLE_RADIUS").toDouble();
  max_knock_circle_radius = conf->value("TUNE_MAX_KNOCK_CIRCLE_RADIUS").toDouble();
  cell_hystersis_rpm = conf->value("TUNE_CELL_HYSTERSIS_RPM").toDouble();
  cell_hystersis_map = conf->value("TUNE_CELL_HYSTERSIS_MAP").toDouble();
  cell_hystersis_maf = conf->value("TUNE_CELL_HYSTERSIS_MAF").toDouble();
  dark_ui = conf->value("DARKNESS",false).toBool();
}

void analyzer::on_return_to_start_btn_clicked() {
  emit return_to_main_page();
}

void analyzer::on_display_selector_currentIndexChanged(int) {
  conf_output_format = ui->display_selector->currentData().toInt();
  refresh_table_from_existing_data();
}

inline double analyzer::skew_trim(double value) {
  if(conf_arbitrary_input == true) return value; // arbitrary input non-trimmable
  double trim = clamp(value - conf_trim_skew,
                      100 - conf_trim_max,
                      100 + conf_trim_max);
  if(trim > 100 - conf_trim_min &&
     trim < 100 + conf_trim_min) {
    return 100;
  } else {
    return trim;
  }
}

void analyzer::on_trim_skew_box_valueChanged(double arg1) {
  conf_trim_skew = arg1;
  refresh_table_from_existing_data();
  ui->target_raw->setText(QString::number((arg1 + 100) * 1.28,'f',1));
}

void analyzer::on_trim_max_box_valueChanged(double arg1) {
  conf_trim_max = arg1;
  refresh_table_from_existing_data();
}

void analyzer::on_trim_min_box_valueChanged(double arg1) {
  conf_trim_min = arg1;
  refresh_table_from_existing_data();
}
