#include "definition.h"
#include "datalog.h"
#include "general.h"
#include "config.h"
#include <QFile>
#include <QTextStream>

definition::definition() {
  conf = new QSettings(CONFIG_SETTINGS_DOMAIN,CONFIG_SETTINGS_GROUP);
  QFile f("dictionary.txt"); // check for external dictionary first
  if(f.exists() == false) f.setFileName(":/dictionary.txt"); // fallback internal resource
  if(f.open(QIODevice::ReadOnly) == false) fatal_error("Could not open dictionary.");
  QTextStream in(&f);
  while(in.atEnd() == false) { // iterate
    QStringList line = split_csv_line(in.readLine(),true,false);
    header_lookup_table.insert(line.at(0),line);
  }
  f.close();
  verify_dictionary();
  QStringList empty;
  apply_new_header(empty); // init with nothing
  // configure boolean hash table
  get_boolean_data("TRUE",true);
  get_boolean_data("FALSE",false);
  // configure misc
  trim_type = TRIM_FORMAT_RAW;
}

void definition::verify_dictionary() {
  QString missing_values;
  bool all_ok = true;
  for(int x=1;x<STDMAP_N_VALUES;x++) {
    QString name = name_of_parameter(x);
    if(header_lookup_table.contains(name) == false) {
      all_ok = false;
      missing_values.append(name + "\n");
    }
  }
  if(all_ok == false) warning("Dictionary is missing the following values:\n" + missing_values);
}

void definition::get_boolean_data(QString lookup, bool is_true) {
  if(header_lookup_table.contains(lookup) == false) {
    fatal_error("Boolean interpreter for " + lookup + " missing from dictionary.");
  }
  QStringList def_keys = header_lookup_table.value(lookup);
  for(int x=0;x<def_keys.size();x++) {
    QString u = def_keys.at(x);
    boolean_lookup_table.insert(u.toUpper(),is_true);
  }
}

void definition::apply_new_header(QStringList header_in) {
  header.clear();
  header.append(header_in);
  field_mapping_list.clear();
  for(int x=1;x<STDMAP_N_VALUES;x++) {
    field_mapping_list.insert(x,guess_best_field(x));
  }
}

void definition::reset() {
  QStringList empty;
  apply_new_header(empty);
}

bool definition::exists(int value) {
  if(field_mapping_list.contains(value) == false || field_mapping_list.value(value) == -1) return false;
  return true;
}

bool definition::string_to_bool(QString str) {
  return boolean_lookup_table.value(str.toUpper());
}

int definition::index_of(int value) {
  return field_mapping_list.value(value);
}

int definition::index_of_item(QString value) {
  for(int x=0;x<header.size();x++) {
    if(QString::compare(header.at(x),value) == 0) return x;
  }
  return -1;
}

QString definition::name_of(int value) {
  if(value < 0 || value > header.size()) {
    return QString("ERR");
  } else {
    return header.at(value);
  }
}

int definition::guess_best_field(int value) {
  return guess_best_field(name_of_parameter(value));
}

int definition::guess_best_field(QString value) {
  if(header_lookup_table.contains(value) == false) return -1;
  if(header.isEmpty() == true) return -1;
  QStringList dictionary_line = header_lookup_table.value(value);
  // item 0 used to be the stock value which was unused, now it's the 'user' value.
  dictionary_line.replace(0,conf->value("USER_DICT_" + value,"UNDEFINED000").toString());
  if(dictionary_line.at(0) == "UNDEFINED000") dictionary_line.removeFirst(); // if doesn't exist
  for(int x=0;x<dictionary_line.size();x++) {
    for(int y=0;y<header.size();y++) {
      if(QString::compare(dictionary_line.at(x),
                          header.at(y),
                          Qt::CaseInsensitive) == 0) return y;
    }
  }
  return -1; // not found
}

QString definition::name_of_parameter(int parameter) {
  switch(parameter) {
  case STDMAP_LBLM:
    return QString("Trim Bank A");
    break;
  case STDMAP_RBLM:
    return QString("Trim Bank B");
    break;
  case STDMAP_LINT:
    return QString("INT Bank A");
    break;
  case STDMAP_RINT:
    return QString("INT Bank B");
    break;
  case STDMAP_BLMCELL:
    return QString("BLM Cell");
    break;
  case STDMAP_RPM:
    return QString("RPM");
    break;
  case STDMAP_MAP:
    return QString("MAP");
    break;
  case STDMAP_MAF:
    return QString("MAF");
    break;
  case STDMAP_COOLANT:
    return QString("Coolant Temp");
    break;
  case STDMAP_TIMESTAMP:
    return QString("Timestamp");
    break;
  case STDMAP_CLOSEDLOOP:
    return QString("Closed Loop");
    break;
  case STDMAP_WOT:
    return QString("WOT");
    break;
  case STDMAP_KNOCKCOUNT:
    return QString("Knock Counter");
    break;
  case STDMAP_TPS:
    return QString("TPS%");
    break;
  case -1:
    return QString("None");
    break;
  default:
    fatal_error("Tried to get definition string for parameter out of range");
    return QString("UNKNOWN"); // never reached
  }
}
