#ifndef ANALYZER_H
#define ANALYZER_H

#include <QWidget>
#include <QAbstractItemModel>
#include <QSettings>
#include <QGraphicsScene>

#include "data_filter.h"
#include "field_mapping.h"
#include "datalog.h"
#include "definition.h"
#include "table_geometry.h"

namespace Ui {
class analyzer;
}

// local cache of configuration parameters to avoid extra lookup time during
// analysis algorithms ...
class analyzer_parameters {
public:
  analyzer_parameters();
  QSettings *conf;
  double distance_rpm;
  double distance_map;
  double interpolator_slope;
  double interpolator_gain;
  double interpolator_noisefloor;
  double minimum_records;
  double ideal_low_trim;
  double ideal_high_trim;
  double integrator_reduction;
  double integrator_max;
  double integrator_min;
  double knock_count_divisor;
  double min_knock_circle_radius;
  double max_knock_circle_radius;
  double cell_hystersis_rpm;
  double cell_hystersis_map;
  double cell_hystersis_maf;
  bool dark_ui;
  void get();
};

class analyzer : public QWidget {
  Q_OBJECT

public:
  explicit analyzer(QWidget *parent = 0);
  ~analyzer();

  struct trim_point {
    float rpm;
    float map;
    float afgs;
    float trim;
  };

  struct knock_point {
    float rpm;
    float map;
    quint32 counts;
  };

  enum {
    FUEL_MODE_VE,
    FUEL_MODE_MAF,
    FUEL_MODE_UNKNOWN
  };

  enum {
    ENGINE_RADIAL,
    ENGINE_STRICT,
    ENGINE_LOOSE,
    ENGINE_UNDEFINED
  };

  // input fields from mainwindow ...
  QList<data_filter *> filter_list;
  QHash<int,field_mapping *> mapping_list;
  QHash<QString,datalog *> log_list;
  definition *def;
  bool leanest_bank;
  bool ignore_int;
  void load_new_analysis(); // called by mainwindow to rebuild vector tables etc

private slots:
  void on_knock_exec_btn_clicked();
  void on_table_to_clipboard_btn_clicked();
  void on_preset_selector_currentIndexChanged(int index);
  void on_engine_select_currentIndexChanged(int index);
  void on_table_edit_btn_clicked();
  void on_ve_table_new_btn_clicked();
  void on_table_delete_btn_clicked();
  void on_return_to_start_btn_clicked();
  void on_display_selector_currentIndexChanged(int index);
  void on_maf_table_new_btn_clicked();
  void on_trim_skew_box_valueChanged(double arg1);
  void on_trim_max_box_valueChanged(double arg1);
  void on_trim_min_box_valueChanged(double arg1);

signals:
  void return_to_main_page();

private:
  Ui::analyzer *ui;
  analyzer_parameters config;
  table_geometry *current_layout; // ptr to selected preset
  quint64 total_lines; // counter for total lines processed

  //-----------
  bool table_redraw_lock; // to avoid redundant recalculations of table data

  //-----------
  void set_busy_cursor(bool busy);
  void prepare_interface();
  void populate_engine_select_box();
  int selected_engine();

  //-----------
  QString get_filter_results();
  void restart_filter_counters();

  //-----------
  void process_file(datalog *log);
  void process_all_files();
  bool process_filter_set(QStringList *data);
  void process_row(QStringList *row);

  //-----------
  bool configure_analyzer();
  bool conf_blm_analysis;
  bool conf_dual_bank;
  bool conf_knock_analysis;
  bool conf_maf_analysis;
  bool conf_integrator;
  bool conf_arbitrary_input;
  int conf_output_format; // enum share with definition::TRIM_FORMAT_*
  double conf_trim_skew;
  double conf_trim_max;
  double conf_trim_min;

  //-----------
  void init_progress();
  void progress_line();
  void progress(QString str);
  void progress_error(QString str);
  void print_configuration(QString module, bool enabled);

  //-----------
  QVector<trim_point> trim_list;
  void restart_trim_list();
  void add_trim_vector(float trim, float rpm, float map, float afgs);
  void get_trim_cell(QStringList *row);

  //-----------
  double linear_distance(double xa, double xb, double ya, double yb); // 2d distance
  double linear_distance(trim_point a, trim_point b); // 2d distance between points

  //-----------
  QVector<knock_point> knock_list;
  void restart_knock_list();
  void add_knock_vector(quint32 count, float rpm, float map);
  void get_knock_cell(QStringList *row);
  quint32 last_knock_count;

  //-----------
  int get_mapping(int stdvalue);
  double get_value(QStringList *row, int stdvalue);
  bool parameter_is_defined(int stdvalue);
  void process_ve_radial(int row, int col);
  void process_ve_strict(int row, int col);
  void process_ve_loose(int row, int col);
  void process_maf_strict(int row);
  void process_maf_loose(int row);

  //-----------
  void reset_table_layout();
  void reset_maf_table_layout();
  void reset_ve_table_layout();
  void switch_maf_ui(bool maf);

  //-----------
  void analyze(); // re-run analyzer on existing vector tables
  void analyze_ve(); // branch for ve analysis
  void analyze_maf(); // branch for maf analysis
  void display_cell(int row, int col, QString str, QString tooltip); // draw text cell data
  void display_cell(int row, int col, double value, double n_recs); // draw numerical cell data
  void display_cell(int row, double value, double n_recs); // draw 2d numerical cell data
  double log_trim_to_multiplier(double trim_in); // convert input trim format to a multiplier
  QColor get_trim_shade(double trim_in); // get cell color variation
  QString get_trim_display_string(double value); // get numerical display string for trim value
  void refresh_table_from_existing_data(); // shortcut refresh of table from existing table data
  double skew_trim(double value); // get skewed trim value

  //--------------------
  QGraphicsScene knockmap;
  void add_knock_point(int rpm, int map, int count, int width, int height);
  void draw_knock_grid(int width, int height);
  int get_analyzer_fuel_mode_selection();

  //--------------------
  // table layout preset stuff
  QList <table_geometry *>preset_list;
  void reload_all_presets();
  void select_preset(table_geometry *layout);

  table_geometry *add_preset(table_geometry *layout);
  table_geometry *add_preset(QString name, QString afgs);
  table_geometry *add_preset(QString name, QString map, QString rpm, bool flipped);
  table_geometry *add_preset(QString mask, QString table, QString afgs);
  table_geometry *add_preset(QString mask, QString table, QString map, QString rpm, bool flipped);

  table_geometry *process_csv_preset(QString name, QString map, QString rpm, bool flipped);
  table_geometry *process_csv_preset(QString name, QString afgs);

  table_geometry *get_user_preset(int preset_number);
  void set_user_preset(int preset_number, table_geometry *layout);
  table_geometry *find_user_preset(int preset_number);
  table_geometry *add_user_preset(table_geometry *layout);
  void delete_user_preset(table_geometry *layout);
  void new_user_preset_ui(bool maf);
};

#endif // ANALYZER_H
