#include "config.h"
#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "about_ui.h"
#include "general.h"
#include <QStyleFactory>
#include <QMessageBox>
#include <QFileDialog>
#include <QDebug>
#include <QDragEnterEvent>
#include <QDropEvent>
#include <QMimeData>

MainWindow::MainWindow(QWidget *parent) :
  QMainWindow(parent),
  ui(new Ui::MainWindow) {

  // get config
  settings = new QSettings(CONFIG_SETTINGS_DOMAIN,CONFIG_SETTINGS_GROUP);

  // setup ui
  set_ui_style(settings->value("DARKNESS",false).toBool());
  ui->setupUi(this);

  master_log = NULL;

  settings_editor = new settings_ui();
  ui->settings_layout->addWidget(settings_editor);

  def = new definition();

  analysis = new analyzer();
  ui->analysis_layout->addWidget(analysis);
  analysis->def = def;

  connect(analysis,SIGNAL(return_to_main_page()),this,SLOT(return_to_main_page()));
  connect(settings_editor,SIGNAL(return_to_main()),this,SLOT(return_to_main_page()));

  populate_mappings();
  enable_controls(false);
  setAcceptDrops(true);

  ui->stackedWidget->setCurrentWidget(ui->loader_page);

  // configure trim type selector
  ui->trim_type_selector->addItem("0-255 (128=Good)",definition::TRIM_FORMAT_RAW);
  ui->trim_type_selector->addItem("Whole Percentage (100=Good)",definition::TRIM_FORMAT_PERCENT_WHOLE);
  ui->trim_type_selector->addItem("+/- Percentage (0=Good)",definition::TRIM_FORMAT_PERCENT_SIGNED);
  ui->trim_type_selector->addItem("Arbitrary Value (AFR, Etc)",definition::TRIM_FORMAT_ARBITRARY);
  ui->trim_type_selector->setCurrentIndex(ui->trim_type_selector->findData(definition::TRIM_FORMAT_RAW));

  setWindowTitle(VERSION_STRING);
}

void MainWindow::dragEnterEvent(QDragEnterEvent *e) {
  if(e->mimeData()->hasUrls()) {
    bool all_valid = true;
    foreach(const QUrl &url, e->mimeData()->urls()) {
      QString filename = url.toLocalFile();
      QFileInfo i(filename);
      if(i.completeSuffix() != "csv") all_valid = false;
    }
    if(all_valid == true) e->acceptProposedAction();
  }
}

void MainWindow::dropEvent(QDropEvent *e) {
  foreach(const QUrl &url, e->mimeData()->urls()) {
    QString filename = url.toLocalFile();
    QFileInfo i(filename);
    if(i.completeSuffix() == "csv") add_log(filename);
  }
}

MainWindow::~MainWindow() {
  delete ui;
}

void MainWindow::set_ui_style(bool dark) {
  qApp->setStyle(QStyleFactory::create("fusion"));
  static QPalette light_palette;
  QPalette dark_palette;
  // stolen from the internets somewhere ..?
  if(dark == true) {
    dark_palette.setColor(QPalette::Window, QColor(53,53,53));
    dark_palette.setColor(QPalette::WindowText, Qt::white);
    dark_palette.setColor(QPalette::Base, QColor(15,15,15));
    dark_palette.setColor(QPalette::AlternateBase, QColor(53,53,53));
    dark_palette.setColor(QPalette::ToolTipBase, Qt::white);
    dark_palette.setColor(QPalette::ToolTipText, Qt::white);
    dark_palette.setColor(QPalette::Text, Qt::white);
    dark_palette.setColor(QPalette::Button, QColor(53,53,53));
    dark_palette.setColor(QPalette::ButtonText, Qt::white);
    dark_palette.setColor(QPalette::BrightText, Qt::red);
    dark_palette.setColor(QPalette::Highlight, QColor(43,90,197).lighter());
    dark_palette.setColor(QPalette::HighlightedText, Qt::black);
    dark_palette.setColor(QPalette::Disabled, QPalette::Text, Qt::darkGray);
    dark_palette.setColor(QPalette::Disabled, QPalette::ButtonText, Qt::darkGray);
    qApp->setPalette(dark_palette);
  } else {
    qApp->setPalette(light_palette);
  }
}

void MainWindow::on_analyzer_btn_clicked() {
  ui->stackedWidget->setCurrentWidget(ui->analysis_page);
  analysis->load_new_analysis();
}

void MainWindow::return_to_main_page() {
  ui->stackedWidget->setCurrentWidget(ui->loader_page);
}

void MainWindow::add_mapping(int mapping) {
  field_mapping *f = new field_mapping(mapping,def);
  ui->mapping_container->insertWidget(0,f);
  analysis->mapping_list.insert(mapping,f);
}

void MainWindow::remove_mapping(int mapping) {
  if(analysis->mapping_list.contains(mapping) == false) {
    fatal_error("Mapping doesn't exist in remove_mapping()");
  }
  delete analysis->mapping_list[mapping];
  analysis->mapping_list.remove(mapping);
}

void MainWindow::mapping_set_visible(int mapping, bool visible) {
  if(analysis->mapping_list.contains(mapping) == false) {
    fatal_error("Mapping doesn't exist in mapping_set_visible()");
  }
  analysis->mapping_list.value(mapping)->setHidden(! visible);
}

void MainWindow::rename_mapping(int mapping, QString new_name) {
  if(analysis->mapping_list.contains(mapping) == false) {
    fatal_error("Mapping doesn't exist in rename_mapping()");
  }
  if(new_name.isEmpty() == true) {
    analysis->mapping_list.value(mapping)->reset_name();
  } else {
    analysis->mapping_list.value(mapping)->set_temporary_name(new_name);
  }
}

void MainWindow::add_filter(int field, int operation, float value) {
  if(def->exists(field) == false) return; // not found
  data_filter *f = new data_filter(def->index_of(field), operation, value, def);
  add_filter(f);
}

void MainWindow::add_filter() {
  data_filter *f = new data_filter(def);
  add_filter(f);
}

void MainWindow::add_filter(data_filter *f) {
  ui->filter_container->insertWidget(0,f);
  analysis->filter_list.append(f);
  connect(f,SIGNAL(delete_me(data_filter*)),this,SLOT(data_filter_delete(data_filter*)));
}

bool MainWindow::add_log(QString filename) {
  if(analysis->log_list.contains(filename) == true) {
    warning("The log at " + filename + " is already loaded.");
    return false;
  }
  datalog *x = new datalog(filename,master_log);
  if(x->is_valid() == false) { // did not load ok
    warning("Error loading log: " + x->last_error);
    delete x;
    return false;
  }
  if(master_log == NULL) {
    // this is now our master log...
    master_log = x;
    initialize_layout();
  }
  analysis->log_list[filename] = x; // add to list
  QListWidgetItem *list_item = new QListWidgetItem();
  list_item->setText(filename);
  ui->log_file_list->addItem(list_item);
  return true;
}

bool MainWindow::delete_log(QString filename) {
  if(analysis->log_list.contains(filename) == false) {
    fatal_error("The log at " + filename + " is not loaded in delete_log()");
  }
  datalog *x = analysis->log_list.take(filename); // get ptr and remove from list
  if(x == master_log) { // we are deleting master log
    if(analysis->log_list.size() == 0) { // this is the last entry, list is now empty
      master_log = NULL;
      def->reset();
      reset_layout();
    } else {
      master_log = analysis->log_list.values().at(0); // assign first as new master
      refresh_mappings();
    }
  }
  delete x;
  return true;
}

bool MainWindow::logs_loaded() {
  if(master_log == NULL) {
    return false;
  } else {
    return true;
  }
}

void MainWindow::initialize_layout() {
  if(logs_loaded() == false) return;
  def->apply_new_header(master_log->header);
  refresh_mappings();
  populate_filters();
  enable_controls(true);
}

void MainWindow::reset_layout() {
  refresh_mappings();
  remove_all_filters();
  enable_controls(false);
}

void MainWindow::enable_controls(bool value) {
  ui->grp_filters->setEnabled(value);
  ui->grp_mappings->setEnabled(value);
  ui->grp_config->setEnabled(value);
  ui->analyzer_btn->setEnabled(value);
}

void MainWindow::populate_mappings() {
  // add in reverse order to preserve spacer
  add_mapping(definition::STDMAP_KNOCKCOUNT);
  add_mapping(definition::STDMAP_RINT);
  add_mapping(definition::STDMAP_LINT);
  add_mapping(definition::STDMAP_RBLM);
  add_mapping(definition::STDMAP_LBLM);
  add_mapping(definition::STDMAP_MAF);
  add_mapping(definition::STDMAP_MAP);
  add_mapping(definition::STDMAP_RPM);
}

void MainWindow::refresh_mappings() {
  QList<int> list = analysis->mapping_list.keys();
  for(int x=0;x<list.size();x++) {
    field_mapping *p = analysis->mapping_list.value(list.at(x));
    p->guess();
  }
}

void MainWindow::remove_all_mappings() {
  QList<int> list = analysis->mapping_list.keys();
  for(int x=0;x<list.size();x++) {
    field_mapping *p = analysis->mapping_list.take(list.at(x));
    delete p;
  }
}

void MainWindow::remove_all_filters() {
  for(int x=0;x<analysis->filter_list.size();x++) {
    data_filter *p = analysis->filter_list.at(x);
    delete p;
  }
  analysis->filter_list.clear();
}

void MainWindow::enable_input(bool x) {
  if(x == false) {
    QApplication::setOverrideCursor(Qt::WaitCursor);
  } else {
    QApplication::restoreOverrideCursor();
  }
  ui->centralWidget->setEnabled(x);
  QApplication::processEvents();
}

void MainWindow::on_add_log_btn_clicked() {
  QStringList file_names = QFileDialog::getOpenFileNames(this,"Log CSV",get_default_path(),"CSV File (*.csv)");
  if(file_names.isEmpty() == true) return;
  set_default_path(file_names.at(0));
  enable_input(false);
  for(int x=0;x<file_names.size();x++) add_log(file_names.at(x));
  enable_input(true);
}

void MainWindow::on_remove_log_btn_clicked() {
  enable_input(false);
  QList<QListWidgetItem*> selection_list = ui->log_file_list->selectedItems();
  for(int x=0;x<selection_list.size();x++) {
    delete_log(selection_list.at(x)->text());
    delete selection_list.at(x);
  }
  enable_input(true);
}

void MainWindow::set_default_path(QString filename) {
  if(filename.isEmpty() == true) return;
  QFileInfo i(filename);
  QString path(i.absoluteFilePath());
  if(path.isEmpty() == true) return;
  settings->setValue("DEFAULT_LOG_DIRECTORY",path);
}

QString MainWindow::get_default_path() {
  return settings->value("DEFAULT_LOG_DIRECTORY","C://").toString();
}

void MainWindow::on_add_filter_btn_clicked() {
  if(logs_loaded() == false) return;
  add_filter();
}

void MainWindow::data_filter_delete(data_filter *f) {
  if(logs_loaded() == false) return;
  analysis->filter_list.removeOne(f);
  delete f;
}

void MainWindow::on_clear_all_filters_btn_clicked() {
  remove_all_filters();
}

void MainWindow::on_auto_filters_btn_clicked() {
  remove_all_filters();
  guess_filters();
}

void MainWindow::on_advanced_settings_btn_clicked() {
  ui->stackedWidget->setCurrentWidget(ui->settings_page);
}

void MainWindow::on_about_btn_clicked() {
  about_ui *x = new about_ui();
  x->show();
}

void MainWindow::on_filters_save_btn_clicked() {
  remember_filters();
}

void MainWindow::on_filters_reload_btn_clicked() {
  recall_filters();
}

void MainWindow::recall_filters() {
  if(logs_loaded() == false) return;
  remove_all_filters();
  int n_filters = n_custom_filters();
  for(int x=0;x<n_filters;x++) {
    QString filter_config = settings->value("CUSTOMFILTER_" + QString::number(x),"ERR,0,0").toString();
    data_filter *new_filter = new data_filter(filter_config,def);
    if(new_filter->is_valid() == true) {
      add_filter(new_filter);
    } else {
      delete new_filter;
    }
  }
}

void MainWindow::remember_filters() {
  if(logs_loaded() == false) return;
  int n_old_filters = n_custom_filters();
  for(int x=0;x<n_old_filters;x++) {
    settings->remove("CUSTOMFILTER_" + QString::number(x));
  }
  int n_filters = analysis->filter_list.size();
  settings->setValue("N_CUSTOMFILTERS",n_filters);
  for(int x=0;x<analysis->filter_list.size();x++) {
    data_filter *p = analysis->filter_list.at(x);
    settings->setValue("CUSTOMFILTER_" + QString::number(x),p->to_string());
  }
}

int MainWindow::n_custom_filters() {
  int n = settings->value("N_CUSTOMFILTERS",0).toInt();
  if(n < 0) fatal_error("Less than 0 custom filters in n_custom_filters()");
  return n;
}

void MainWindow::populate_filters() {
  guess_filters();
}

void MainWindow::guess_filters() {
  add_filter(definition::STDMAP_RPM,
             data_filter::GREATERTHAN,
             500);
  add_filter(definition::STDMAP_RPM,
             data_filter::LESSTHAN,
             6500);
  add_filter(definition::STDMAP_TPS,
             data_filter::GREATERTHAN,
             1.00);
  add_filter(definition::STDMAP_COOLANT,
             data_filter::GREATERTHAN_OR_EQ,
             60.00);
  add_filter(definition::STDMAP_CLOSEDLOOP,
             data_filter::IS_TRUE,
             0);
  add_filter(definition::STDMAP_WOT,
             data_filter::IS_FALSE,
             0);
}

void MainWindow::on_trim_type_selector_currentIndexChanged(int) {
  def->trim_type = ui->trim_type_selector->currentData().toInt();
  if(def->trim_type == definition::TRIM_FORMAT_ARBITRARY) {
    ui->ignore_int_sw->setEnabled(false);
    ui->lean_bank_sw->setEnabled(false);
    rename_mapping(definition::STDMAP_LBLM,"INPUT A");
    rename_mapping(definition::STDMAP_RBLM,"INPUT B");
    mapping_set_visible(definition::STDMAP_LINT, false);
    mapping_set_visible(definition::STDMAP_RINT, false);
  } else {
    ui->ignore_int_sw->setEnabled(true);
    ui->lean_bank_sw->setEnabled(true);
    rename_mapping(definition::STDMAP_LBLM);
    rename_mapping(definition::STDMAP_RBLM);
    mapping_set_visible(definition::STDMAP_LINT, true);
    mapping_set_visible(definition::STDMAP_RINT, true);
  }
}

void MainWindow::on_lean_bank_sw_toggled(bool checked) {
  analysis->leanest_bank = checked;
}

void MainWindow::on_ignore_int_sw_toggled(bool checked) {
  analysis->ignore_int = checked;
}

void MainWindow::on_stackedWidget_currentChanged(int arg1) {
  // prevent dropping on anything but main window
  if(arg1 == 0) {
    setAcceptDrops(true);
  } else {
    setAcceptDrops(false);
  }
}
